import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { EMPTY_DATA, MINIMUM_AGE, MINIMUM_EXPERIENCE, ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Collapse } from '../../../ui/Collapse';
import FormatDate, { getAge } from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import { StatusTypes } from '../../../ui/Status';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as styleTable from '../../../ui/Table/index.css';
import { Wallet } from '../../../ui/Wallet';
import { IBlockRules } from '../../../utils/IBlockRules';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import LS from '../../../utils/localStorage/localStorage';
import { Request2, specialQueryParamsKeys } from '../../../utils/request';
import { Translate } from '../../../utils/translate';
import { IStore } from '../../App/store';
import Img from '../../Img';
import RegistrationTimeline from '../../RegistrationTimeline';
import { RegistrationTimelineSize } from '../../RegistrationTimeline/types';
import { ITrustCard } from '../../Settings/Wallets/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../request';
import {
    APPROVING_PHOTO_REG_EXP,
    CGI_SEPARATOR,
    CURRENT_PHOTO_TYPE_SEPARATOR,
    DOC_STATUS_MAP,
    DOCUMENTS_KEYS,
    FRAUD_STATUSES,
    IS_FIRST_RIDING_TITLE,
    PHOTO_SHOW_MODE,
    VIDEO_SHOW_MODE,
} from './constants';
import { DeviceSearchModal } from "./DeviceSearchModal";
import * as style from './index.css';
import MediaGallery from './MediaGallery';
import ReissueDocChecks from './ReissueDocChecks/component';
import DocumentSelfies from './SelfiesBlock';
import { ISettingsItem } from './types';

const PHOTO_CGI = {
    DOC_TYPE: 'docType',
    PHOTO_TYPE: 'photoType',
    CHECK_TIMESTAMP: 'checkTimestamp',
    SHOW_MODE: 'showMode',
};

export interface IPhoto {
    id: string;
    background_video_uri: string;
    origin_chat: string;
    submitted_at: number;
    type: string;
    uri: string;
    verification_status: string;
    verified_at: number;
}

interface IDocumentsViewProps extends IBlockRules, IStore, RouteComponentProps<{ routeState: string }> {
    userId: string;
}

interface IDocumentsViewState {
    userInfo: Record<string, any> | null;
    settingData: Record<string, any> | null;
    isLoading: boolean;
    loadingError: Error | null;
    currentGalleryPhotos: string[] | null;
    currentPhotoType: string;
    currentPhotoIndex: number | null;
    showVideo: boolean;
    isRegisterModalOpen: boolean;
    isWorking: boolean;
    workingError: Error | null;
    autoCodeError: Error | null;
    dataAutoCodeIsLoading: boolean;
    modalAutoCodeIsOpen: boolean;
    ReRequestForChatIsOpen: boolean;
    autoCodeData: Record<string, any> | null;
    registrationFlow: Record<string, any>;
}

export default class DocumentsView extends React.Component<IDocumentsViewProps, IDocumentsViewState> {
    state: IDocumentsViewState = {
        isLoading: false,
        loadingError: null,
        userInfo: null,
        settingData: null,
        currentGalleryPhotos: null,
        currentPhotoType: '',
        currentPhotoIndex: null,
        showVideo: false,
        isRegisterModalOpen: false,
        isWorking: false,
        workingError: null,
        autoCodeError: null,
        dataAutoCodeIsLoading: false,
        modalAutoCodeIsOpen: false,
        ReRequestForChatIsOpen: false,
        autoCodeData: null,
        registrationFlow: {},
    };
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;
    ls = new LS();

    componentDidMount() {
        this.getData();
        this.formatRegistrationFlow();
    }

    componentDidUpdate(prevProps: Readonly<IDocumentsViewProps>): void {
        if (this.props.userId !== prevProps.userId) {
            this.getData();
        } else if (JSON.stringify(this.props.location) !== JSON.stringify(prevProps.location)) {
            this.getCgiData();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    formatRegistrationFlow() {
        const settings = this.props.AdminUser?.rawSettings;
        const settingValue = settings?.find(el => el.setting_key?.includes('registration_flow_new'))?.setting_value;
        let registrationFlow;
        try {
            registrationFlow = JSON.parse(settingValue)
                .reduce((prev, curr) => {
                    prev[curr.id] = {
                        order: curr.order,
                    };

                    return prev;
                }, {});
        } catch {
            registrationFlow = {};
        }

        this.setState({ registrationFlow });
    }

    getData() {
        const settings: ISettingsItem = this.props.AdminUser?.rawSettings
            ?.filter((_i: ISettingsItem) => _i.setting_key === 'interface_admin.data')[0];

        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_USER_INFO, { queryParams: { user_id: this.props.userId } })
                .then(response => {
                    const userInfo = response;

                    let settingData: Record<string, any> | null = null;

                    if (isValidJSONString(settings.setting_value)) {
                        settingData = settings && JSON.parse(settings.setting_value);
                        settingData = settingData?.admin_interface_settings?.user_data;
                    }

                    this.setState({
                        isLoading: false, userInfo, settingData,
                    }, () => {
                        this.getCgiData();
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError });
                });
        });
    }

    getCgiData() {
        const { userInfo } = this.state;
        const photos = userInfo?.photos ?? {};
        const cgi = this.getCgiObject();

        const docType = cgi?.get(PHOTO_CGI.DOC_TYPE) || '';
        const photoType = cgi?.get(PHOTO_CGI.PHOTO_TYPE) || '';
        const checkTimestamp = cgi?.get(PHOTO_CGI.CHECK_TIMESTAMP) || '';

        const currentPhotosArr = photos?.[docType]?.[photoType] || [];

        let currentPhotoIndex = 0;
        currentPhotosArr.some((currentPhoto: IPhoto, index) => {
            if ([currentPhoto?.verified_at?.toString(),
                currentPhoto?.submitted_at?.toString()].includes(checkTimestamp)) {
                currentPhotoIndex = index;

                return true;
            }

            return false;
        });

        const showMode = cgi?.get(PHOTO_CGI.SHOW_MODE) || '';
        const currentGalleryPhotos = docType && photoType ? [docType, photoType] : null;

        this.setState({
            currentPhotoIndex,
            currentGalleryPhotos,
            currentPhotoType: currentGalleryPhotos ? currentGalleryPhotos.join(CURRENT_PHOTO_TYPE_SEPARATOR) : '',
            showVideo: showMode === VIDEO_SHOW_MODE,
        });
    }

    setHashWithCgi(cgi: string) {
        location.hash = `${location.hash.split(CGI_SEPARATOR)[0]}?${cgi}`;
    }

    getCgiObject() {
        const params = location?.hash?.split(CGI_SEPARATOR)[1]
            && new URLSearchParams(location.hash.split(CGI_SEPARATOR)[1]);

        return params as URLSearchParams;
    }

    onCurrentPhotosChange(value: string) {
        const [docType, photoType] = value.split(CURRENT_PHOTO_TYPE_SEPARATOR);

        const cgi = this.getCgiObject();

        cgi?.set(PHOTO_CGI.DOC_TYPE, docType);
        cgi?.set(PHOTO_CGI.PHOTO_TYPE, photoType);

        this.setHashWithCgi(cgi.toString());
    }

    onPictureChange(currentPhotoIndex: number) {
        const { userInfo } = this.state;
        const photos = userInfo?.photos ?? {};
        const cgi = this.getCgiObject();

        const docType = cgi?.get(PHOTO_CGI.DOC_TYPE) || '';
        const photoType = cgi?.get(PHOTO_CGI.PHOTO_TYPE) || '';
        const currentPhotosArr = photos?.[docType]?.[photoType] ?? [];

        const checkTimestamp = currentPhotosArr?.[currentPhotoIndex]?.verified_at?.toString()
            ?? currentPhotosArr?.[currentPhotoIndex]?.submitted_at?.toString();

        cgi?.set(PHOTO_CGI.CHECK_TIMESTAMP, checkTimestamp);

        this.setHashWithCgi(cgi.toString());
    }

    onDocTypeChange(value: string) {
        const cgi = this.getCgiObject();

        cgi?.set(PHOTO_CGI.SHOW_MODE, value);

        this.setHashWithCgi(cgi.toString());
    }

    openRegisterModal() {
        this.setState({ isRegisterModalOpen: true });
    }

    closeRegisterModal() {
        this.setState({ isRegisterModalOpen: false });
    }

    registerUser() {
        this.setState({ isWorking: true, workingError: null }, () => {
            this.request.exec(REQUESTS.REGISTER_USER,
                { queryParams: { user_id: this.state.userInfo?.id } })
                .then(() => {
                    this.getData();
                    this.closeRegisterModal();
                    this.setState({ isWorking: false });

                })
                .catch((workingError) => {
                    this.setState({ workingError, isWorking: false });
                });
        });
    }

    closeModalAutoCode() {
        this.setState({ modalAutoCodeIsOpen: false });
    }

    checkAutoCode() {
        const DATE_LENGTH = 10;

        const drivingLicenseData: Dict<string> = UserInfoHandler.getDrivingLicenseData.call(this.state.userInfo);
        const drivingLicenseNumber = drivingLicenseData?.number_back || null;
        const drivingLicenseIssueDate = drivingLicenseData?.issue_date || null;

        const query = {
            license_number: drivingLicenseNumber,
            issue_date: drivingLicenseIssueDate?.substr(0, DATE_LENGTH),
        };

        drivingLicenseNumber && drivingLicenseIssueDate
        && this.setState({
            dataAutoCodeIsLoading: true,
            modalAutoCodeIsOpen: true,
            autoCodeError: null,
        }, () => {
            this.request.exec(REQUESTS.CHECK_AUTO_CODE, {
                queryParams: query,
            })
                .then(response => {
                    this.setState({
                        dataAutoCodeIsLoading: false,
                        autoCodeData: response ?? {},
                    });
                })
                .catch(autoCodeError => {
                    this.setState({
                        dataAutoCodeIsLoading: false,
                        autoCodeError,
                    });
                });
        });
    }

    openReRequestForChat() {
        this.setState({ ReRequestForChatIsOpen: true });
    }

    closeReRequestForChat() {
        this.setState({ ReRequestForChatIsOpen: false });
    }

    render() {
        const {
            loadingError, userInfo, currentGalleryPhotos, currentPhotoType, showVideo, isLoading, currentPhotoIndex,
            isRegisterModalOpen, workingError, isWorking, modalAutoCodeIsOpen, autoCodeData, dataAutoCodeIsLoading,
            autoCodeError, registrationFlow,
        } = this.state;

        const { BlockRules, userId } = this.props;

        if (loadingError) {
            return <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке документов' }}/>;
        }

        const photos = userInfo?.photos;

        let currentPhotos: IPhoto[] = [];
        if (currentGalleryPhotos) {
            const [docType, photoType] = currentGalleryPhotos;
            currentPhotos = photos?.[docType]?.[photoType];
            UserDocItem.sortPhotos(currentPhotos);
        }

        const settings = this.props.AdminUser?.rawSettings;
        const regFlowData = settings?.find(el => el.setting_key?.includes('registration_flow_new'));

        return <div className={style.user_documents}>
            <CommonUserInfo checkAutoCode={this.checkAutoCode.bind(this)}
                            isLoading={isLoading}
                            BlockRules={BlockRules}
                            userInfo={userInfo}/>
            <div className={style.additional_information}>
                <h4>{IS_FIRST_RIDING_TITLE}: </h4>
                <strong>{userInfo?.is_first_riding ? 'ДА' : 'НЕТ'}</strong>
            </div>

            <div className={style.additional_information}>
                <ReissueDocChecks userId={this.props.userId ?? ''}
                                  regFlowData={regFlowData}
                                  registration_flow={registrationFlow}/>
            </div>

            <div className={style.additional_information}>
                <RegistrationTimeline userId={userId} size={RegistrationTimelineSize.LARGE}/>
            </div>
            {
                BlockRules?.UserPhotoPassport &&
                <UserDocData docType={DOCUMENTS_KEYS.passport.key}
                             photoTypes={[DOCUMENTS_KEYS.passport.selfie.key,
                                 DOCUMENTS_KEYS.passport.biographical.key, DOCUMENTS_KEYS.passport.registration.key]}
                             userInfo={userInfo}
                             isLoading={isLoading}
                             BlockRules={BlockRules}
                             getData={this.getData.bind(this)}/>
            }
            {
                BlockRules?.UserPhotoLicense &&
                <UserDocData docType={DOCUMENTS_KEYS.driving_license.key}
                             photoTypes={[DOCUMENTS_KEYS.driving_license.front.key,
                                 DOCUMENTS_KEYS.driving_license.back.key, DOCUMENTS_KEYS.driving_license.selfie.key]}
                             userInfo={userInfo}
                             BlockRules={BlockRules}
                             isLoading={isLoading}
                             getData={this.getData.bind(this)}/>
            }
            {(BlockRules?.UserPhotoSelfie
                    && (BlockRules?.SecretX || BlockRules?.UserPhotoDocs)) &&
                <DocumentSelfies selfies={userInfo?.photos?.selfie}
                                 userId={userId}
                                 secretXRule={BlockRules?.SecretX}/>
            }

            {(currentGalleryPhotos && (BlockRules?.SecretX || BlockRules?.UserPhotoDocs))
                ? <MediaGallery BlockRules={BlockRules as Record<string, boolean>}
                                currentPhotos={currentPhotos}
                                currentPhotoType={currentPhotoType}
                                onCurrentPhotosChange={this.onCurrentPhotosChange.bind(this)}
                                onDocTypeChange={this.onDocTypeChange.bind(this)}
                                onPictureChange={this.onPictureChange.bind(this)}
                                openRegisterModal={this.openRegisterModal.bind(this)}
                                showVideo={showVideo}
                                userInfo={userInfo as Record<string, any>}
                                currentPhotoIndex={currentPhotoIndex}/>
                : null}
            {isRegisterModalOpen
                ? <Confirm error={workingError}
                           isWorking={isWorking}
                           onClose={this.closeRegisterModal.bind(this)}
                           accept={this.registerUser.bind(this)}
                           question={'Зарегистрировать пользователя?'}/>
                : null}
            {modalAutoCodeIsOpen
                ? <JsonModal obj={autoCodeData}
                             title={'Автокод ВУ'}
                             error={autoCodeError}
                             onClose={this.closeModalAutoCode.bind(this)}>
                    {dataAutoCodeIsLoading ? <Spin/> : null}
                </JsonModal>
                : null}
        </div>;
    }
}

interface ICommonUserInfoProps {
    isLoading: boolean;
    userInfo: Record<string, any> | null;
    checkAutoCode: () => {};
    BlockRules?: Record<string, any>;
}

interface ICommonUserInfoState {
    isDevicesModalOpen: boolean;
    isFavAddressOpen: boolean;
}

class CommonUserInfo extends React.Component<ICommonUserInfoProps, ICommonUserInfoState> {
    state: ICommonUserInfoState = {
        isDevicesModalOpen: false,
        isFavAddressOpen: false,
    };

    openDevicesModal() {
        this.setState({ isDevicesModalOpen: true });
    }

    closeDevicesModal() {
        this.setState({ isDevicesModalOpen: false });
    }

    openFavAddressModal(isFavAddressOpen) {
        this.setState({ isFavAddressOpen });
    }

    render() {
        const { isLoading, checkAutoCode, userInfo, BlockRules } = this.props;
        const { isDevicesModalOpen, isFavAddressOpen } = this.state;
        const favouritesAddress = userInfo?.favourites?.items || [];

        return <div className={style.common_user_info}>
            <CommonDocInfo isLoading={isLoading} docPhotos={userInfo?.photos}/>
            <Fraud isLoading={isLoading} yangData={userInfo?.yang ?? []}/>
            <div className={style.doc_controls}>
                {isLoading
                    ? <Spin/>
                    : <div>
                        <UserCards userId={userInfo?.id}/>
                        <div><Link onClick={this.openDevicesModal.bind(this)}>Устройства</Link></div>
                    </div>}
            </div>
            <div className={style.doc_controls}>
                <ul>
                    <li><Link onClick={checkAutoCode.bind(this)}>Автокод ВУ</Link></li>
                    {
                        (favouritesAddress.length && BlockRules?.UserAddresses)
                            ? <li><Link onClick={this.openFavAddressModal.bind(this, true)}>
                                Адреса ({favouritesAddress.length})
                            </Link></li>
                            : null
                    }

                </ul>
            </div>
            {isDevicesModalOpen
                ? <ManageDevices onClose={this.closeDevicesModal.bind(this)} user_id={userInfo?.id}/>
                : null}

            {isFavAddressOpen
                ? <FavAddressModal onClose={this.openFavAddressModal.bind(this, false)}
                                   favouritesAddress={favouritesAddress}/>
                : null}
        </div>;
    }
}

const FavAddressModal = (props) => {
    return <Window title={`Адреса`} onClose={props.onClose}>
        <table className={styleTable.table}>
            <thead>
                <tr>
                    <th>#</th>
                    <th>заголовок</th>
                    <th>дата создания</th>
                    <th>дата редактирования</th>
                    <th>адрес</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {
                    props?.favouritesAddress?.map((el, index) => {
                        return <tr key={index}>
                            <td>{index + 1}</td>
                            <td>
                                <div><strong>{el.title}</strong></div>
                                <div><i>{el.tags.join(', ')}</i></div>
                            </td>
                            <td><FormatDate value={el.created}/></td>
                            <td><FormatDate value={el.modified}/></td>
                            <td>{el.address_line}</td>
                            <td>
                                <Link target={'_blank'}
                                      href={`https://yandex.ru/maps?`
                                      + `ll=${el.longitude},${el.latitude}&z=18&pt=${el.longitude},${el.latitude}`}>
                                Я.Карта</Link>
                            </td>
                        </tr>;
                    })
                }
            </tbody>
        </table>
    </Window>;
};

interface IUserCardsProps {
    userId: string;
}

interface IUserCardsState {
    isCardsModalOpen: boolean;
    userTrustCards: ITrustCard[];
    error: Error | null;
    cardsLoading: boolean;
}

class UserCards extends React.Component<IUserCardsProps, IUserCardsState> {
    state: IUserCardsState = {
        isCardsModalOpen: false,
        userTrustCards: [],
        error: null,
        cardsLoading: true,
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getCardsData();
    }

    componentDidUpdate(prevProps: Readonly<IUserCardsProps>): void {
        if (this.props.userId !== prevProps.userId) {
            this.getCardsData();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getCardsData() {
        const user_id = this.props.userId;

        user_id && this.request.exec(REQUESTS.GET_CARDS_TRUST, {
            queryParams: {
                user_id,
            },
        })
            .then((response) => {
                const { cards } = response;

                this.setState({
                    userTrustCards: cards,
                    cardsLoading: false,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                    cardsLoading: false,
                });
            });
    }

    openCardsModal() {
        this.setState({ isCardsModalOpen: true });
    }

    closeCardsModal() {
        this.setState({ isCardsModalOpen: false });
    }

    render() {
        const { error, userTrustCards, isCardsModalOpen } = this.state;

        if (error) {
            return <SimpleError error={error} data={{ label: 'Ошибка при загрузке карт' }}/>;
        }

        const cards = userTrustCards;

        return <div className={style.user_cards}>
            <h4>Карты</h4>
            {
                cards && Array.isArray(cards)
                && cards.length === 1
                    ? <Wallet card={cards[0]}
                              active/>
                    : cards.length > 1
                        ? <Link onClick={this.openCardsModal.bind(this)}>Привязанные карты ({cards.length})</Link>
                        : <span>Нет карт</span>
            }
            {isCardsModalOpen
                ? <Window title={'Привязанные карты'}
                          onClose={this.closeCardsModal.bind(this)}>
                    <div className={style.userCardInfo}>
                        {cards.map((card, key) => {
                            return (
                                <Wallet key={key}
                                        card={card}
                                        active/>
                            );
                        })}
                    </div>
                </Window>
                : null}
        </div>;
    }
}

interface ICommonDocInfoProps {
    isLoading: boolean;
    docPhotos: Dict<{ passport: IPhoto[]; driving_license: IPhoto[] }>;
}

const CommonDocInfo = React.memo((props: ICommonDocInfoProps) => {
    const docPhotos = props.docPhotos;

    const getDocStatus = (documentType: string, documentName: string): React.ReactElement => {
        const lastDoc = docPhotos?.[documentType]?.[documentName]
            ?.[docPhotos?.[documentType]?.[documentName]?.length - 1];

        return lastDoc
            && <DocStatus type={lastDoc.verification_status}/>;
    };

    return <div className={style.common_doc_info}>
        <div className={style.doc_info_item}>
            {props.isLoading
                ? <Spin/>
                : <>
                    <h4>{DOCUMENTS_KEYS.passport.key}</h4>
                    <div>
                        {DOCUMENTS_KEYS.passport.selfie.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.passport.key,
                            DOCUMENTS_KEYS.passport.selfie.key,
                        )}
                    </div>
                    <div>
                        {DOCUMENTS_KEYS.passport.biographical.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.passport.key,
                            DOCUMENTS_KEYS.passport.biographical.key,
                        )}
                    </div>
                    <div>
                        {DOCUMENTS_KEYS.passport.registration.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.passport.key,
                            DOCUMENTS_KEYS.passport.registration.key,
                        )}
                    </div>
                </>}
        </div>
        <div className={style.doc_info_item}>
            {props.isLoading
                ? <Spin/>
                : <>
                    <h4>{DOCUMENTS_KEYS.driving_license.key}</h4>
                    <div>
                        {DOCUMENTS_KEYS.driving_license.front.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.driving_license.key,
                            DOCUMENTS_KEYS.driving_license.front.key,
                        )}
                    </div>
                    <div>
                        {DOCUMENTS_KEYS.driving_license.back.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.driving_license.key,
                            DOCUMENTS_KEYS.driving_license.back.key,
                        )}
                    </div>
                    <div>
                        {DOCUMENTS_KEYS.driving_license.selfie.display_text} {getDocStatus(
                            DOCUMENTS_KEYS.driving_license.key,
                            DOCUMENTS_KEYS.driving_license.selfie.key,
                        )}
                    </div>
                </>}
        </div>
    </div>;
});

export const DocStatus = React.memo((props: { type: string }) => {
    return <div className={`${style.status} ${style[DOC_STATUS_MAP?.[props?.type]?.status || StatusTypes.DEFAULT]}`}>
        {DOC_STATUS_MAP?.[props?.type]?.text || props.type}
    </div>;
});

interface IFraudProps {
    isLoading: boolean;
    yangData: IYang[];
}

interface IFraudState {
    data: IYang[];
    isAnyDefFraud: boolean;
    isAnyMaybeFraud: boolean;
    isFullInfoOpen: boolean;
    isMoreModalOpen: boolean;
}

interface IYang {
    assignment_ids: string[];
    comments: string[];
    created_at: string;
    fraud_reasons: string[];
    id: string;
    is_fraud: string;
    license_back_id: string;
    license_front_id: string;
    passport_biographical_id: string;
    passport_registration_id: string;
    passport_selfie_id: string;
    processed_at: string;
}

export class Fraud extends React.Component<IFraudProps, IFraudState> {
    state: IFraudState = {
        data: [],
        isAnyDefFraud: false,
        isAnyMaybeFraud: false,
        isFullInfoOpen: false,
        isMoreModalOpen: false,
    };

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IFraudProps>): void {
        if (this.props.yangData !== prevProps.yangData) {
            this.getData();
        }
    }

    getData() {
        const { yangData = [] } = this.props;
        const data = yangData.sort((a, b) => b.created_at.localeCompare(a.created_at));

        const isAnyDefFraud = data.some(item => item.is_fraud === FRAUD_STATUSES.DEFINITELY_FRAUD.key);
        const isAnyMaybeFraud = data.some(item => item.is_fraud === FRAUD_STATUSES.MAYBE_FRAUD.key);

        this.setState({ data, isAnyDefFraud, isAnyMaybeFraud });
    }

    openFullInfoModal() {
        this.setState({ isFullInfoOpen: true });
    }

    closeFullInfoModal() {
        this.setState({ isFullInfoOpen: false });
    }

    openMoreModal() {
        this.setState({ isMoreModalOpen: true });
    }

    closeMoreModal() {
        this.setState({ isMoreModalOpen: false });
    }

    render() {
        const { isLoading } = this.props;
        const { data, isAnyDefFraud, isAnyMaybeFraud, isFullInfoOpen, isMoreModalOpen } = this.state;
        const lastCheck: IYang = data && Array.isArray(data)
            ? data[0]
            : {} as IYang;

        const commentLength = 20;
        const comment = lastCheck?.comments?.length
            && `${lastCheck?.comments?.join(';')?.substr(0, commentLength)}...`
            || EMPTY_DATA;
        let isFraudClassname = '';
        if (lastCheck?.is_fraud === FRAUD_STATUSES.DEFINITELY_FRAUD.key) {
            isFraudClassname = style.fraud_def_warning;
        } else if (lastCheck?.is_fraud === FRAUD_STATUSES.MAYBE_FRAUD.key) {
            isFraudClassname = style.fraud_maybe_warning;
        }

        const reasons = lastCheck?.fraud_reasons
            && Array.isArray(lastCheck.fraud_reasons)
            && lastCheck.fraud_reasons?.length && `(${lastCheck.fraud_reasons.join(', ')})`
            || null;

        return <div className={style.fraud}>
            {isLoading
                ? <Spin/>
                : <>
                    <div className={style.fraud_title}>
                        <h4>Вердикт</h4>
                        {isAnyDefFraud
                            ? <>
                                <div data-tip data-for={'fraud_def'} className={`${style.image_fraud} ${style.def}`}/>
                                <ReactTooltip id={'fraud_def'}
                                              type="error"
                                              effect="solid">
                                    Пользователь ТОЧНО был уличён в мошенничестве
                                </ReactTooltip>
                            </>
                            : null}
                        {isAnyMaybeFraud
                            ? <>
                                <div data-tip
                                     data-for={'fraud_maybe'}
                                     className={`${style.image_fraud} ${style.maybe}`}/>
                                <ReactTooltip id={'fraud_maybe'}
                                              type="warning"
                                              effect="solid">
                                    Пользователь был уличён в ВОЗМОЖНОМ мошенничестве
                                </ReactTooltip>
                            </>
                            : null}
                    </div>
                    <FormatDate value={lastCheck && new Date(lastCheck.created_at)}/>
                    <div>
                        <strong>Мошенник</strong>: <span className={isFraudClassname}>
                            {FRAUD_STATUSES?.[lastCheck?.is_fraud?.toUpperCase()]?.display_text
                                || (lastCheck?.is_fraud)
                                || EMPTY_DATA}
                        </span>
                        <div>
                            {reasons}
                        </div>
                    </div>
                    <div className={style.fraud_comment} title={comment}>
                        {comment}
                    </div>
                    <Link className={style.more_link} onClick={this.openFullInfoModal.bind(this)}>Подробнее...</Link>
                    <Link onClick={this.openMoreModal.bind(this)}>JSON</Link>
                </>}
            {isFullInfoOpen
                ? <Window title={'История проверок'}
                          className={style.full_fraud_info}
                          onClose={this.closeFullInfoModal.bind(this)}>
                    <div>
                        {data && Array.isArray(data)
                            ? <table className={styleTable.table}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Дата</th>
                                        <th>is_fraud</th>
                                        <th>Причина</th>
                                        <th>Комментарий</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {data.map((item, index) => {
                                        let status = LabelStatus.DEFAULT;
                                        if (item.is_fraud === FRAUD_STATUSES.DEFINITELY_FRAUD.key) {
                                            status = LabelStatus.NEGATIVE;
                                        }

                                        if (item.is_fraud === FRAUD_STATUSES.MAYBE_FRAUD.key) {
                                            status = LabelStatus.WARNING;
                                        }

                                        return <tr key={index}>
                                            <td>{++index}</td>
                                            <td className={style.date_cell}>
                                                <FormatDate value={item && new Date(item.created_at)}/>
                                            </td>
                                            <td>
                                                <TLabel className={style.check_status}
                                                        status={status}
                                                        text={item.is_fraud || EMPTY_DATA}/>
                                            </td>
                                            <td>
                                                {item.fraud_reasons
                                                && Array.isArray(item.fraud_reasons)
                                                && item.fraud_reasons.length
                                                && `(${item.fraud_reasons.join(', ')})` || EMPTY_DATA}
                                            </td>
                                            <td>
                                                {item?.comments?.join(';') ?? EMPTY_DATA}
                                            </td>
                                        </tr>;
                                    })}
                                </tbody>
                            </table>
                            : null}
                    </div>
                </Window>
                : null}
            {isMoreModalOpen
                ? <JsonModal title={'Вердикт - JSON'} obj={data} onClose={this.closeMoreModal.bind(this)}/>
                : null}
        </div>;
    }
}

interface IUserDocDataProps {
    docType: string;
    photoTypes: string[];
    userInfo: Record<string, any> | null;
    isLoading: boolean;
    getData: () => void;
    BlockRules?: any;
}

interface IUserDocDataState {
    userFormattedInfo: Record<string, any> | null;
}

class UserDocData extends React.Component<IUserDocDataProps, IUserDocDataState> {
    state: IUserDocDataState = {
        userFormattedInfo: null,
    };

    componentDidMount(): void {
        this.formatData();
    }

    componentDidUpdate(prevProps: Readonly<IUserDocDataProps>): void {
        if (JSON.stringify(this.props.userInfo) !== JSON.stringify(prevProps.userInfo)) {
            this.formatData();
        }
    }

    formatData() {
        const { userInfo = {} } = this.props;

        if (userInfo) {
            let userFormattedInfo: { title: string; value: string | React.ReactElement }[][] = [];
            if (this.props.docType === DOCUMENTS_KEYS.passport.key) {

                const birthDate = UserInfoHandler.getBirthDay.call(userInfo);
                const age = getAge(birthDate);

                userFormattedInfo = [[
                    {
                        title: 'Имя',
                        value: UserInfoHandler.getPassportName.call(userInfo),
                    },
                    {
                        title: 'Фамилия',
                        value: UserInfoHandler.getPassportLastName.call(userInfo),
                    },
                    {
                        title: 'Отчество',
                        value: UserInfoHandler.getPassportMiddleName.call(userInfo),
                    },
                ],
                [
                    {
                        title: 'Биогр., страна',
                        value: UserInfoHandler.getPassportBio.call(userInfo),
                    },
                    {
                        title: 'Регистрация, страна',
                        value: UserInfoHandler.getPassportReg.call(userInfo),
                    },
                    {
                        title: 'Серия',
                        value: UserInfoHandler.getPassportSeries.call(userInfo),
                    },

                    {
                        title: 'Номер',
                        value: UserInfoHandler.getPassportNumber.call(userInfo),
                    },
                    {
                        title: 'Дата выдачи',
                        value: <FormatDate value={UserInfoHandler.getPassportIssueDate.call(userInfo)}
                                           onlyDate={true}/>,
                    },
                    {
                        title: 'Место выдачи',
                        value: UserInfoHandler.getPassportIssuePlace.call(userInfo),
                    },
                    {
                        title: 'Дата истечения',
                        value: <FormatDate value={UserInfoHandler.getExpirationDate.call(userInfo)}
                                           onlyDate={true}/>,
                    },
                ],
                [
                    {
                        title: 'Пол',
                        value: UserInfoHandler.getPassportGender.call(userInfo),
                    },
                    {
                        title: 'Дата рождения',
                        value: <FormatDate value={birthDate}
                                           onlyDate={true}/>,
                    },
                    {
                        title: 'Возраст',
                        value: age < MINIMUM_AGE ? <span className={style.too_young}>{age}</span> : age,
                    },
                    {
                        title: 'Место рождения',
                        value: UserInfoHandler.getBirthPlace.call(userInfo),
                    },
                ],
                [
                    {
                        title: 'Место жительства',
                        value: UserInfoHandler.getRegistrationPlace.call(userInfo),
                    },
                    {
                        title: 'Регистрация ДО',
                        value: <FormatDate value={UserInfoHandler.getRegistrationDateUntil.call(userInfo)}
                                           onlyDate={true}/>,
                    },
                ]];
            } else if (this.props.docType === DOCUMENTS_KEYS.driving_license.key) {
                const drivingLicenseData = UserInfoHandler.getDrivingLicenseData.call(userInfo);
                const experience = getAge(drivingLicenseData.experience_from != EMPTY_DATA
                    ? drivingLicenseData.experience_from
                    : drivingLicenseData.categories_b_valid_from_date);

                userFormattedInfo = [
                    [
                        {
                            title: 'Фамилия',
                            value: drivingLicenseData.last_name,
                        },
                        {
                            title: 'Имя',
                            value: drivingLicenseData.first_name,
                        },
                        {
                            title: 'Отчество',
                            value: drivingLicenseData.middle_name,
                        },
                        {
                            title: 'Дата рождения',
                            value: <FormatDate onlyDate value={drivingLicenseData.birth_date}/>,
                        },
                    ],
                    [
                        {
                            title: 'Страна перед',
                            value: drivingLicenseData.front_country,
                        },
                        {
                            title: 'Страна оборот',
                            value: drivingLicenseData.back_country,
                        },
                        {
                            title: 'Номер',
                            value: drivingLicenseData.number_front,
                        },
                        {
                            title: 'Категория',
                            value: drivingLicenseData.categories,
                        },
                        {
                            title: 'Механика',
                            value: UserInfoHandler.getMechanicTransmissionAllow.call(userInfo),
                        },
                        {
                            title: 'Cтаж с',
                            value: <FormatDate onlyDate
                                               value={
                                                   drivingLicenseData.experience_from != EMPTY_DATA
                                                       ? drivingLicenseData.experience_from
                                                       : drivingLicenseData.categories_b_valid_from_date
                                               }/>,
                        },
                    ],
                    [
                        {
                            title: 'Когда выданы',
                            value: <FormatDate value={drivingLicenseData.issue_date}
                                               onlyDate={true}/>,
                        },
                        {
                            title: 'Кем выданы',
                            value: drivingLicenseData.issued_by ?? EMPTY_DATA,
                        },
                        {
                            title: 'Опыт',
                            value: experience < MINIMUM_EXPERIENCE ?
                                <span className={style.too_young}>{experience}</span> : experience,
                        },
                        {
                            title: 'Действительны до',
                            value: <FormatDate value={drivingLicenseData.categories_b_valid_to_date}
                                               onlyDate={true}/>,
                        },
                    ],
                ];
            }

            this.setState({ userFormattedInfo });
        }
    }

    render() {
        const { userFormattedInfo } = this.state;
        const { userInfo = {}, docType, isLoading, photoTypes, getData, BlockRules } = this.props;
        const passportPhotos = userInfo?.photos?.[docType];
        const SecretX = BlockRules?.SecretX || BlockRules?.UserPhotoDocs;

        return <div className={`${style.user_doc_item_data} ${!SecretX ? style.secret_x : ''}`}>
            <h2>{docType === DOCUMENTS_KEYS.passport.key
                ? DOCUMENTS_KEYS.passport.display_name
                : DOCUMENTS_KEYS.driving_license.display_name}
            </h2>
            {isLoading
                ? <Spin/>
                : <div className={style.user_doc_data_container}>
                    <div className={style.user_doc_text}>
                        {userFormattedInfo
                            ? userFormattedInfo.map((passportDataBlock, index) => {
                                return <div key={index}>
                                    <table>
                                        <tbody>
                                            {passportDataBlock?.map((item, index) => {
                                                return <tr key={index}>
                                                    <td>
                                                        <div className={style.text_title}>{item.title}</div>
                                                    </td>
                                                    <td>
                                                        <div title={item.value} className={style.text_data}>
                                                            {item.value}
                                                        </div>
                                                    </td>
                                                </tr>;
                                            })}
                                        </tbody>
                                    </table>
                                </div>;
                            })
                            : null}
                    </div>
                    {
                        SecretX &&

                        <div className={style.user_doc_photo}>
                            {photoTypes?.map(photoType => {
                                return <UserDocItem docType={docType}
                                                    getData={getData.bind(this)}
                                                    key={photoType}
                                                    type={photoType}
                                                    user_id={userInfo?.id}
                                                    photos={passportPhotos && passportPhotos[photoType]}/>;
                            })}
                        </div>
                    }
                </div>}
        </div>;
    }
}

interface IUserDocItemProps {
    docType: string;
    getData: () => {};
    type: string;
    photos: IPhoto[];
    user_id: string;
}

interface IUserDocItemState {
    photos: IPhoto[] | null;
    confirmIsOpen: boolean;
    approvingPhotoId: string | null;
    isApproving: boolean;
    approveError: Error | null;
    photo: string | null;
    isPhotoLoading: boolean;
    photoError: Error | null;
}

export class UserDocItem extends React.Component<IUserDocItemProps, IUserDocItemState> {
    state: IUserDocItemState = {
        photos: null,
        confirmIsOpen: false,
        approvingPhotoId: null,
        isApproving: false,
        approveError: null,
        photo: null,
        isPhotoLoading: false,
        photoError: null,
    };
    request = new Request2({
        requestConfigs,
    });

    static getDocUrl(uri: string) {
        let host = location.host;
        if (location.host.includes('localhost')) {
            host = 'carsharing.yandex-team.ru';
        }

        return `//${host}${uri}`;
    }

    static sortPhotos(photos: IPhoto[]) {
        return photos?.sort((a, b) => b.verified_at - a.verified_at);
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getCgi(): string {
        const docType = this.props.docType;
        const photoType = this.props.type;
        const timestamp = this.state.photos?.[0]?.verified_at;

        return `?docType=${docType}&photoType=${photoType}&checkTimestamp=${timestamp}&showMode=${PHOTO_SHOW_MODE}`;
    }

    componentDidMount(): void {
        this.formatData();
    }

    componentDidUpdate(prevProps: Readonly<IUserDocItemProps>, prevState): void {
        if (JSON.stringify(this.props) !== JSON.stringify(prevProps)) {
            this.formatData();
        }

        if (this.state.photos && this.state.photos[0]?.id !== prevState.photos?.[0]?.id) {
            this.getPhoto();
        }
    }

    formatData() {
        let { photos = [] } = this.props;

        photos = UserDocItem.sortPhotos(photos);

        this.setState({ photos });
    }

    openApproveConfirm(uri: string) {
        const approvingPhotoRegExp = APPROVING_PHOTO_REG_EXP.exec(uri);
        const approvingPhotoId = approvingPhotoRegExp && approvingPhotoRegExp[1];

        this.setState({ confirmIsOpen: true, approvingPhotoId });
    }

    closeApproveConfirm() {
        this.setState({ isApproving: false, confirmIsOpen: false, approvingPhotoId: null });
    }

    approvePhoto() {
        const id = this.state.approvingPhotoId;

        id && this.setState({
            isApproving: true,
            approveError: null,
        }, () => {
            this.request.exec(REQUESTS.APPROVE_USERS_PHOTO, { queryParams: { photo_id: id } })
                .then(() => {
                    this.closeApproveConfirm();
                    this.props.getData();
                })
                .catch((approveError) => {
                    this.setState({
                        approveError,
                        isApproving: false,
                    });
                });
        });
    }

    getPhoto() {
        const { user_id } = this.props;
        const { photos } = this.state;

        this.setState({
            isPhotoLoading: true,
            photoError: null,
        }, () => {
            this.request.exec(REQUESTS.GET_USER_DOC_DATA, {
                queryParams: {
                    user_id,
                    photo_id: photos?.[0]?.id,
                },
                blob: true,
            })
                .then(photoBlob => {
                    const URL = window.URL || window.webkitURL;
                    const blobLink = URL.createObjectURL(photoBlob);

                    this.setState({
                        photo: blobLink,
                        isPhotoLoading: false,
                    });
                })
                .catch(photoError => {
                    this.setState({
                        photoError,
                        isPhotoLoading: false,
                    });
                });
        });
    }

    render() {
        const { docType, type } = this.props;
        const { photos, confirmIsOpen, isApproving, approveError } = this.state;

        const lastPhoto = photos && photos[0];
        const lastPhotoUri = lastPhoto && lastPhoto.uri || '';

        return <div className={style.user_doc_item}>
            <div>{DOCUMENTS_KEYS?.[docType]?.[type]?.display_text || type}</div>
            <div className={style.doc_info}>
                <div className={style.doc_date}>
                    {lastPhoto
                        ? lastPhoto.verified_at
                            ? <FormatDate value={lastPhoto.verified_at * ONE_SECOND}/>
                            : EMPTY_DATA
                        : 'НЕТ ФОТО'
                    }
                </div>
                <div>
                    {lastPhoto && <DocStatus type={lastPhoto.verification_status}/>}
                </div>
            </div>
            <div className={style.approve}>
                {lastPhoto && lastPhoto.verification_status !== DOC_STATUS_MAP.o.key
                    && <Link onClick={this.openApproveConfirm.bind(this, lastPhotoUri)}>Одобрить фото</Link>
                }
            </div>
            <Link href={`${location.hash.split(CGI_SEPARATOR)[0]}${this.getCgi()}`}>
                <Img className={style.photo_preview}
                     isLoading={this.state.isPhotoLoading}
                     error={this.state.photoError}
                     src={this.state.photo ?? ''}
                     thumbnail={true}>
                    <Spin size={'l'}/>
                </Img>
            </Link>
            {confirmIsOpen
                ? <Confirm accept={this.approvePhoto.bind(this)}
                           isWorking={isApproving}
                           onClose={this.closeApproveConfirm.bind(this)}
                           error={approveError}
                           question={'Одобрить фото?'}/>
                : null}
        </div>;
    }

}

interface IManageDevicesProps {
    user_id: string;
    currentDevice?: string;
    onClose: () => void;
}

interface IManageDevicesState {
    selected: boolean[];
    devices: IDevices[];
    devicesHistory: IDevicesHistory[];
    isLoading: boolean;
    error: Error | null;
    confirmError: Error | null;

    groupState: boolean;
    confirmIsOpen: boolean;
    accept: () => void;
    isWorking: boolean;
    question: string | React.ReactElement;
    selectedDevice: Record<string, any> | null;
    searchDevice: IDevices | null;
}

export interface IDevices {
    enabled: boolean;
    id: string;
    verified: boolean;
    device_id: string;
    user_id: string;
    description: { headers: { [key: string]: string } };
}

enum IDevicesHistoryActions {
    ADD = 'add',
    UPDATE_DATA = 'update_data',
    REMOVE = 'remove',
    PROPOSITION = 'proposition',
    CONFIRMATION = 'confirmation',
}

interface IDevicesHistory extends IDevices {
    action: IDevicesHistoryActions;
    event_id: number;
    timestamp: number;
}

const TRUE_KEY = 'true';
const FALSE_KEY = 'false';

export class ManageDevices extends React.Component<IManageDevicesProps, IManageDevicesState> {
    state: IManageDevicesState = {
        selected: [],
        devices: [],
        devicesHistory: [],
        isLoading: false,
        error: null,
        confirmError: null,

        groupState: false,
        confirmIsOpen: false,
        accept: () => {
        },
        isWorking: false,
        question: '',
        selectedDevice: null,

        searchDevice: null,
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getDevices();
    }

    componentDidUpdate(prevProps: Readonly<IManageDevicesProps>): void {
        if (prevProps.user_id !== this.props.user_id) {
            this.getDevices();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getDevices() {
        this.setState({ isLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_USER_DEVICES, { queryParams: { user_id: this.props.user_id } })
                .then(response => {
                    this.setState({
                        devices: response.devices || [],
                        devicesHistory: response.history || [],
                        selected: new Array(response.devices.length).fill(false),
                        isLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    checkboxHandler(index: number) {
        const selected = this.state.selected;
        selected[index] = !selected[index];
        this.setState({ selected });
    }

    changeGroupState() {
        this.setState({
            groupState: !this.state.groupState,
        });
    }

    isEveryDeselected() {
        return this.state.selected.every(_e => !_e);
    }

    setEnable() {
        this.setState({
            confirmError: null,
            confirmIsOpen: true,
            question: <span>Выполнить операцию <strong> ENABLE: {this.state.groupState + ''}</strong> ?</span>,
            accept: () => {
                this.setState({
                    isWorking: true,
                    confirmError: null,
                }, () => {
                    const devices: string[] = this.state.selected
                        .reduce((result: string[], current: boolean, index: number) => {
                            if (current) {
                                result.push(this.state.devices[index].id);
                            }

                            return result;
                        }, []);

                    this.request.exec(REQUESTS.ENABLE_DEVICE, {
                        queryParams: {
                            devices: devices.join(','),
                            user_id: this.props.user_id,
                        },
                        specialQueryParams: {
                            enabled: {
                                key: specialQueryParamsKeys.FORCE,
                                value: this.state.groupState,
                            },
                        },
                    })
                        .then(() => {
                            this.getDevices();
                            this.closeConfirm();
                        })
                        .catch(confirmError => {
                            this.setState({
                                confirmError,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    setVerify() {
        this.setState({
            confirmIsOpen: true,
            confirmError: null,
            question: <span>Выполнить операцию <strong> VERIFY: {this.state.groupState + ''}</strong> ?</span>,
            accept: () => {
                this.setState({
                    isWorking: true,
                    confirmError: null,
                }, () => {
                    const devices: string[] = this.state.selected
                        .reduce((result: string[], current: boolean, index: number) => {
                            if (current) {
                                result.push(this.state.devices[index].id);
                            }

                            return result;
                        }, []);

                    this.request.exec(REQUESTS.VERIFY_DEVICE, {
                        queryParams: {
                            devices: devices.join(','),
                            user_id: this.props.user_id,
                        },
                        specialQueryParams: {
                            verified: {
                                key: specialQueryParamsKeys.FORCE,
                                value: this.state.groupState,
                            },
                        },
                    })
                        .then(() => {
                            this.getDevices();
                            this.closeConfirm();
                        })
                        .catch((confirmError) => {
                            this.setState({
                                confirmError,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    closeConfirm() {
        this.setState({
            confirmIsOpen: false,
            isWorking: false,
        });
    }

    removeDevice(devices: string, e: KeyboardEvent) {
        this.setState({
            confirmIsOpen: true, question: `Удалить устройство ${devices}?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                    confirmError: null,
                }, () => {
                    this.request.exec(REQUESTS.REMOVE_DEVICE, { queryParams: { devices, user_id: this.props.user_id } })
                        .then(() => {
                            this.setState({
                                confirmIsOpen: false,
                                isWorking: false,
                            }, () => {
                                this.getDevices();
                            });
                        })
                        .catch((confirmError) => {
                            this.setState({
                                confirmError,
                                confirmIsOpen: false,
                            });
                        });
                });
            },
        });
    }

    openMoreInfoModal(selectedDevice: Record<string, any>) {
        this.setState({ selectedDevice });
    }

    closeMoreInfoModal() {
        this.setState({ selectedDevice: null });
    }

    openDeviceSearch(searchDevice: IDevices) {
        this.setState({ searchDevice });
    }

    render() {
        const { onClose } = this.props;
        const {
            error, devices, selected, groupState, selectedDevice, confirmIsOpen,
            accept, isWorking, question, confirmError, devicesHistory, searchDevice,
        } = this.state;

        if (error) {
            return <SimpleError error={error} data={{ label: 'Ошибка при загрузке устройств' }}/>;
        }

        const { currentDevice } = this.props;

        return <Window onClose={onClose} title={'Устройства пользователя'}>
            <div className={style.devices}>
                {
                    currentDevice
                        ? <h3>Рассматриваемое устройство: {currentDevice}</h3>
                        : null
                }
                <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th/>
                            <th/>
                            <th>id</th>
                            <th>Включено</th>
                            <th>Подтверждено</th>
                            <th/>
                            <th/>
                            <th/>
                        </tr>
                    </thead>
                    <tbody>
                        {devices
                            .sort((a: IDevices, b: IDevices) => a.id.localeCompare(b.id))
                            .map((item: IDevices, index: number) => {
                                return <tr key={index}
                                           className={item.id === this.props.currentDevice ? style.current : null}>
                                    <td>{index + 1}</td>
                                    <td>
                                        <Checkbox onChange={this.checkboxHandler.bind(this, index)}
                                                  checked={selected[index]}/>
                                    </td>
                                    <td className={style.device_id}>{item.id}</td>
                                    <td>
                                        <TLabel text={item.enabled ? 'Да' : 'Нет'}
                                                status={item.enabled ? LabelStatus.POSITIVE : LabelStatus.NEGATIVE}/>
                                    </td>
                                    <td>
                                        <TLabel text={item.verified ? 'Да' : 'Нет'}
                                                status={item.verified ? LabelStatus.POSITIVE : LabelStatus.NEGATIVE}/>
                                    </td>
                                    <td>
                                        <Link onClick={this.removeDevice.bind(this, item.id)}>
                                        Удалить
                                        </Link>
                                    </td>
                                    <td>
                                        <Link onClick={this.openMoreInfoModal.bind(this, item)}>
                                        Info
                                        </Link>
                                    </td>
                                    <td>
                                        <Link onClick={this.openDeviceSearch.bind(this, item)}>
                                        Поиск
                                        </Link>
                                    </td>
                                </tr>;
                            })}
                    </tbody>
                </table>
                <div className={style.group_state} onClick={this.changeGroupState.bind(this)}>
                    Значение Включено/Подтверждено: <strong>{groupState ? 'TRUE ' : 'FALSE'}</strong>
                    <Checkbox className={style.group_state_checkbox} checked={groupState}/>
                </div>
                {
                    selectedDevice
                        ? <JsonModal title={'Информация об устройстве'}
                                     obj={selectedDevice}
                                     onClose={this.closeMoreInfoModal.bind(this)}/>
                        : null
                }
                {
                    confirmIsOpen
                        ? <Confirm accept={accept}
                                   isWorking={isWorking}
                                   question={question}
                                   error={confirmError}
                                   onClose={this.closeConfirm.bind(this)}/>
                        : null
                }
                <Collapse title={`История: ${devicesHistory.length}`}
                          collapsed={true}>

                    <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Время</th>
                                <th>Действие</th>
                                <th>id устройства</th>
                                <th/>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                devicesHistory
                                    .sort((a, b) => {
                                        return b.event_id - a.event_id;
                                    })
                                    .map((el: IDevicesHistory, index: number) => {
                                        return <tr key={index}>
                                            <td>{++index}</td>
                                            <td><FormatDate value={el.timestamp * ONE_SECOND} withSecond={true}/></td>
                                            <td>{el.action}</td>
                                            <td>{el.device_id}</td>
                                            <td>
                                                <Link onClick={this.openMoreInfoModal.bind(this, el)}>
                                                Info
                                                </Link>
                                            </td>
                                        </tr>;
                                    })
                            }
                        </tbody>
                    </table>
                </Collapse>
                <div className={style.controls}>
                    <Button colorType={ButtonTypes.positive}
                            disabled={this.isEveryDeselected() || isWorking}
                            onClick={this.setEnable.bind(this)}>
                        Включить: {groupState ? TRUE_KEY : FALSE_KEY}
                    </Button>
                    <Button disabled={this.isEveryDeselected() || isWorking}
                            onClick={this.setVerify.bind(this)}>
                        Подтвердить: {groupState ? TRUE_KEY : FALSE_KEY}
                    </Button>
                </div>
            </div>

            {searchDevice
                ? <DeviceSearchModal onClose={this.openDeviceSearch.bind(this, null)}
                                     device={searchDevice}/>
                : null
            }
        </Window>;
    }
}
