import moment from 'moment';
import React, { useEffect, useState } from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes } from '../../../../ui/Button';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from "../../../../ui/FullModal";
import { Link } from '../../../../ui/Link';
import { Request2 } from '../../../../utils/request';
import { rubs } from '../../../../utils/rubs';
import CarNumber from '../../../CarNumber';
import DocForQueuePicker from '../../../Documents/DocForQueuePicker';
import { getFullKoap } from '../../../Fines/FineMessageType/ShortFineInfo/getFullKoap';
import { AVAILABLE_LESSORS, DISPUTE_DAYS_COUNT, FINES_TOOLTIP_ID, RESOLUTION_PREFIX } from '../constants';
import * as handleStyle from '../index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from '../request';

const START_2019_TIMESTAMP = 1546290000;
const FINISH_2019_TIMESTAMP = 1577826000;
const MAX_PHOTOS_COUNT = 3;

interface IFullFinesInfoItemProps {
    item: Dict<any>;
    index: number;
    showPhoto: (url: string) => void;
    showEmailButton?: boolean;
    onProtocolButtonClick: (item) => void;
    openResendConfirm: (id: string) => void;
    showModal: (type: string, id: string) => void;
    openResendConfirmWithDoc: (id: string) => void;
    style?: Dict<any>;
    onAttorneyButtonClick: (item?: any) => void;
    attorneyForQueueModalId: string | null;
}

export const FullFinesInfoItem = React.memo((props: IFullFinesInfoItemProps) => {
    const [isAttorney, setIsAttorney] = useState(false);
    const dispute_deadline = +moment(props.item.ruling_date).add(DISPUTE_DAYS_COUNT, 'days');
    const back = encodeURIComponent(location.hash);
    const request = new Request2({ requestConfigs });
    const [isPhotoModalOpened, openPhotoModal] = useState<boolean>(false);

    function setShowAttorney(car_id: string) {
        request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id } })
            .then((response) => {
                setIsAttorney(AVAILABLE_LESSORS.includes(response?.lessor));
            })
            .catch(() => {
                setIsAttorney(false);
            });
    }

    useEffect(() => {
        setShowAttorney(props.item?.car?.id);
    }, []);

    return <div key={props.item.id} className={handleStyle.fine_grid_wrapper} style={props.style}>
        {props.attorneyForQueueModalId === props.item?.id
            ? <DocForQueuePicker onClose={props.onAttorneyButtonClick}
                                 initValues={{
                                     user_id: props.item?.user?.id,
                                     fine_id: props.item?.id,
                                     document_name: 'fine_power_of_attorney',
                                 }}/>
            : null}
        <div className={handleStyle.dates}>
            <div className={handleStyle.index}>
                {props.index + 1}
            </div>
            <div className={handleStyle.dates_info}>
                <table>
                    <tbody>
                        <tr className={handleStyle.dates_info_row}>
                            <td>
                                <span><b>штраф: </b></span>
                            </td>
                            <td>
                                <FormatDate withSecond={true}
                                            value={props.item.violation_time * ONE_SECOND}/>
                            </td>
                        </tr>
                        <tr className={handleStyle.dates_info_row}>
                            <td>
                                <span><b>email: </b></span>
                            </td>
                            <td>
                                <FormatDate withSecond={true}
                                            value={props.item.charge_email_sent_at * ONE_SECOND}/>
                            </td>
                        </tr>
                        <tr className={handleStyle.dates_info_row}>
                            <td>
                                <span><b>push: </b></span>
                            </td>
                            <td>
                                <FormatDate withSecond={true}
                                            value={props.item.charge_push_sent_at * ONE_SECOND}/>
                            </td>
                        </tr>
                        <tr className={handleStyle.dates_info_row}>
                            <td>
                                <span><b>выставлен: </b></span>
                            </td>
                            <td>
                                <FormatDate withSecond={true}
                                            value={props.item.charged_at * ONE_SECOND}/>
                            </td>
                        </tr>
                        <tr className={handleStyle.dates_info_row}>
                            <td>
                                <span><b>платёж: </b></span>
                            </td>
                            <td>
                                <FormatDate withSecond={true}
                                            value={props.item.charge_passed_at * ONE_SECOND}/>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <div className={handleStyle.info}>
            <table className={handleStyle.info_table}>
                <thead>
                    <tr className={handleStyle.info_headers}>
                        <td className={handleStyle.info_td}>Сумма</td>
                        <td className={handleStyle.info_td}>Автомобиль</td>
                        <td className={handleStyle.info_td}>Место</td>
                        <td className={handleStyle.info_td}>Фото</td>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td className={`${handleStyle.info_td} ${handleStyle.info_rubs}`}>
                            <b>{props.item.sum_to_pay ? rubs(props.item.sum_to_pay) : EMPTY_DATA}</b>
                        </td>
                        <td className={`${handleStyle.info_td} ${handleStyle.info_car}`}>
                            <div className={handleStyle.info_carName}>
                                {props.item.car?.model?.name
                                    ? <Link href={'#/cars/' + props.item.car.id + `/info?back=${back}`}>
                                        {props.item.car.model.name}
                                    </Link>
                                    : EMPTY_DATA
                                }
                            </div>
                            <div className={handleStyle.car_number}>
                                {props.item.car?.number
                                    ? <CarNumber carInfo={props.item.car} externalTooltipId={FINES_TOOLTIP_ID}/>
                                    : EMPTY_DATA
                                }
                            </div>
                            <div>
                            СТС: {props.item.car?.registration_id || EMPTY_DATA}
                            </div>
                        </td>
                        <td className={handleStyle.info_td}>
                            <div>
                                {props.item.violation_place || EMPTY_DATA}
                            </div>
                            <div className={handleStyle.user}>
                                {props.item.user?.id
                                    ? <>
                                        <div><b>Пользователь</b></div>

                                        <Link href={'#/clients/' + props.item.user.id + '/info'}>
                                            {UserInfoHandler.getPrintName.call(props.item.user) || EMPTY_DATA}
                                        </Link></>
                                    : null}
                                {
                                    props.item?.order?.id
                                && <div><Link href={'#/session/' + props.item?.order?.id} target={'_blank'}>
                                    Поездка
                                </Link></div>
                                }
                            </div>
                        </td>
                        <td className={`${handleStyle.info_td} ${handleStyle.info_photos}`}>
                            {props.item.photos?.length
                                ? props.item.photos.slice(0, MAX_PHOTOS_COUNT).map((item: Dict<any>, index: number) => {
                                    return <div key={index} className={handleStyle.photo_div}>
                                        <Link onClick={props.showPhoto.bind(null, item?.url)}>
                                        фото {index + 1}
                                        </Link>
                                    </div>;
                                })
                                : EMPTY_DATA
                            }
                            {props.item.photos?.length > MAX_PHOTOS_COUNT
                                ? <Link onClick={() => openPhotoModal(true)}>Ещё...</Link>
                                : null
                            }
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>

        <div className={handleStyle.koap}>
            <div className={handleStyle.koap_wrapper}
                 data-tip={getFullKoap(props.item)}
                 data-for={FINES_TOOLTIP_ID}>
                <div className={handleStyle.koap_title}>
                    Статья КоАП {props.item.article_code ?? EMPTY_DATA}
                </div>

                <div className={handleStyle.koap_block}>
                    {getFullKoap(props.item)}
                </div>
            </div>
        </div>

        <div className={handleStyle.resolution}>
            <div className={`${handleStyle.resolution_items} ${handleStyle.resolution_number}`}>
                <div className={handleStyle.resolution_header}>Номер постановления</div>
                <div className={handleStyle.resolution_text}>
                    {props.item.ruling_number
                        ? <Link href={`${RESOLUTION_PREFIX}${props.item.ruling_number}`}
                                target={'_blank'}>
                            {props.item.ruling_number}
                        </Link>
                        : EMPTY_DATA
                    }
                </div>
            </div>
            <div className={`${handleStyle.resolution_items} ${handleStyle.resolution_dates}`}>
                <div className={handleStyle.resolution_header}>Постановление / Оспаривание до
                </div>
                <div className={handleStyle.resolution_text}>
                    <span> {<FormatDate value={props.item.ruling_date}
                                        onlyDate={true}/> ?? EMPTY_DATA} / </span>
                    <span> {<FormatDate value={dispute_deadline}
                                        onlyDate={true}/> ?? EMPTY_DATA} </span>
                </div>
            </div>
        </div>

        <div className={handleStyle.first_buttons_block}>
            {isAttorney
                && <div className={handleStyle.buttons_wrapper}>
                    <Button className={handleStyle.controlButton}
                            basic
                            onClick={props.onAttorneyButtonClick.bind(null, props.item)}>Доверенность</Button>
                </div>}
            <div className={handleStyle.buttons_wrapper}>
                {props.item.ruling_number
                    ? <Button className={handleStyle.controlButton}
                              basic
                              onClick={props.onProtocolButtonClick.bind(null, props.item)}>Протокол</Button>
                    : null
                }
            </div>
            {props.showEmailButton && <div className={handleStyle.buttons_wrapper}>
                <Button className={handleStyle.controlButton}
                        basic
                        onClick={props.openResendConfirm.bind(null, props.item.id)}>Письмо</Button>
            </div>}
        </div>
        <div className={handleStyle.second_buttons_block}>
            <div className={handleStyle.buttons_wrapper}>
                <Button className={handleStyle.controlButton}
                        basic
                        colorType={ButtonTypes.warning}
                        onClick={props.showModal.bind(null, 'reassignFineId', props.item.id)}>Перевыставить</Button>
            </div>
            <div className={handleStyle.buttons_wrapper}>
                {props.item.violation_time
                && props.item.violation_time > START_2019_TIMESTAMP
                && props.item.violation_time < FINISH_2019_TIMESTAMP
                    ? <Button className={handleStyle.controlButton}
                              basic
                              colorType={ButtonTypes.warning}
                              onClick={props.openResendConfirmWithDoc.bind(null, props.item.id)}>Письмо+док</Button>
                    : null
                }
            </div>
        </div>

        {isPhotoModalOpened
            ? <Window onClose={() => openPhotoModal(false)} title="Просмотр фото штрафа">
                {props.item.photos.map((item, index: number) => {
                    return <div key={index} className={handleStyle.photo_div}>
                        <Link onClick={props.showPhoto.bind(null, item?.url)}>
                            фото {index + 1}
                        </Link>
                    </div>;
                })
                }
            </Window>
            : null
        }
    </div>;
});
