import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Confirm, Window } from '../../../ui/FullModal';
import { NoInformation } from '../../../ui/NoInformation';
import { FinesRoutes } from '../../../utils/navigation';
import { Request2, specialQueryParamsKeys } from '../../../utils/request';
import { Translate } from '../../../utils/translate';
import { IStore } from '../../App/store';
import FinesDownPaymentsDialog from '../../FinesDownPaymentsDialog';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { FINES_TOOLTIP_ID } from './constants';
import { FullFinesInfo } from './FullFinesInfo';
import { FullFinesInfoItem } from './FullFinesInfo/FullFinesInfoItem';
import * as style from './index.css';
import { ReassignFineModal } from './ReassignFineModal/component';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from './request';
import { ShortFinesInfo } from './ShortFinesInfo';

interface FinesViewProps extends IStore {
    userId?: string | null;
    carId?: string | null;
    route: FinesRoutes;
}

const MAX_FULL_DATA_LENGTH = 10;

interface FinesViewState {
    [x: number]: any;

    data: Dict<any> [];
    notChargedData: Dict<any> [];
    charged_only: boolean;
    not_charged_only: boolean;
    showImgUrl: string;
    resendFineId: string;
    isLoading: boolean;
    isResendLoading: boolean;
    error: Error | null;
    resendError: Error | null;
    showFinesModal: boolean;
    question: string;
    accept: () => void;
    reassignFineId: string;
    isReducedMode: boolean;
    protocolError: Error | null;

    isWindowOpen: boolean;
    itemForWindow: Dict<any> | null;
    attorneyForQueueModalId: string | null;
}

export default class FinesView extends React.Component<FinesViewProps, FinesViewState> {
    state: FinesViewState = {
        data: [],
        notChargedData: [],
        charged_only: true,
        not_charged_only: false,
        showImgUrl: '',
        resendFineId: '',
        isLoading: false,
        isResendLoading: false,
        error: null,
        resendError: null,
        showFinesModal: false,
        question: '',
        accept: () => {},
        reassignFineId: '',
        isReducedMode: false,
        protocolError: null,

        isWindowOpen: false,
        itemForWindow: null,
        attorneyForQueueModalId: null,
    };
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    componentDidMount(): void {
        if (this.props.userId || this.props.carId) {
            this.getFines();
        }
    }

    componentDidUpdate(prevProps: Readonly<FinesViewProps>, prevState): void {
        if (prevProps.userId !== this.props.userId
            || prevProps.carId !== this.props.carId
            || prevState.charged_only !== this.state.charged_only) {
            this.getFines();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getFines() {
        this.setState({ isLoading: true }, () => {
            let queryParams: Dict<any> = {};
            if (this.props.userId) {
                queryParams = { user_id: this.props.userId };
            } else if (this.props.carId) {
                queryParams = { car_id: this.props.carId };
            }

            this.request.exec(REQUESTS.GET_DATA, {
                queryParams,
                specialQueryParams: {
                    charged_only: {
                        key: specialQueryParamsKeys.FORCE,
                        value: this.state.charged_only,
                    },
                },
            })
                .then((response) => {
                    let notChargedData = [];

                    if (!this.state.charged_only) {
                        notChargedData = response?.filter(el => {
                            return !el.charge_passed_at;
                        });
                    }

                    this.setState({
                        notChargedData,
                        data: response,
                        isReducedMode: Boolean(response?.length > MAX_FULL_DATA_LENGTH),
                        isLoading: false,
                        error: null,
                    }, () => {
                        ReactTooltip.rebuild();
                    });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    showPhoto(url: string, e: KeyboardEvent) {
        e?.preventDefault();
        this.setState({
            showImgUrl: url,
        });
    }

    onClosePhoto() {
        this.setState({
            showImgUrl: '',
        });
    }

    openResendConfirm(resendFineId: string) {
        this.setState({
            resendFineId,
            question: 'Переслать письмо?',
            accept: this.resendFineLetter.bind(this),
        });
    }

    openResendConfirmWithDoc(resendFineId: string) {
        this.setState({
            resendFineId,
            question: 'Переслать письмо с документами?',
            accept: this.resendFineLetterWithDocs.bind(this),
        });
    }

    closeResendConfirm() {
        this.setState({ resendFineId: '' });
    }

    resendFineLetter() {
        this.setState({ isResendLoading: true }, () => {
            this.request.exec(REQUESTS.RESEND_FINE_EMAIL,
                { queryParams: { fine_id: this.state.resendFineId }, body: {} })
                .then(() => {
                    this.setState({ isResendLoading: false, resendError: null });
                    this.closeResendConfirm();
                })
                .catch((resendError) => {
                    this.setState({ isResendLoading: false, resendError });
                });
        });
    }

    resendFineLetterWithDocs() {
        const body = {
            'tag_config': {
                'tag_name_template': 'email_fine_pdd_3m',
                'tag_data_template': '{ "template_args": { "sum": "<fine.payment_total_without_discount>", "autocode_url": "https://www.mos.ru/shtrafy/result?dap=<fine.ruling_number>", "final_sum": "<fine.payment_total>", "paragraph": "<fine.article>", "violation_place": "<fine.violation_place>", "car_model": "<fine.car.model>", "ruling_number": "<fine.ruling_number>", "car_plate": "<fine.car.number>", "strike_date": "<fine.violation_time>", "photo_code_block": "<fine.violation_photo_block>" }, "comment": "#<fine.ruling_number>", "attachments": [<fine.violation_detailed_document>] }',
            },
        };
        this.setState({ isResendLoading: true }, () => {
            this.request.exec(REQUESTS.RESEND_FINE_EMAIL,
                { queryParams: { fine_id: this.state.resendFineId }, body })
                .then(() => {
                    this.setState({ isResendLoading: false, resendError: null });
                    this.closeResendConfirm();
                })
                .catch((resendError) => {
                    this.setState({ isResendLoading: false, resendError });
                });
        });
    }

    showModal(key, value) {
        this.setState({
            [key]: value,
        });
    }

    showChargedOnly(charged_only) {
        if (this.state.not_charged_only && charged_only) {
            this.setState({
                charged_only,
                not_charged_only: false,
            });
        } else {
            this.setState({
                charged_only,
            });
        }
    }

    showNotChargedOnly(not_charged_only) {
        if (this.state.charged_only && not_charged_only) {
            this.setState({
                not_charged_only,
                charged_only: false,
            });
        } else {
            this.setState({
                not_charged_only,
            });
        }
    }

    openDecree(res) {
        window.open(res.url, 'Download');
    }

    changeReducedMode(isReducedMode) {
        this.setState({ isReducedMode });
    }

    onProtocolButtonClick(item) {
        this.request.exec(REQUESTS.DECREE_INFO, { queryParams: { fine_id: item.id } })
            .then((response) => {
                this.openDecree(response);
            })
            .catch(protocolError => {
                this.setState({ protocolError });
            });
    }

    onShortItemClick(itemForWindow) {
        this.setState({ itemForWindow, isWindowOpen: true });
    }

    openWindow(isWindowOpen) {
        this.setState({ isWindowOpen });
    }

    onAttorneyButtonClick(item) {
        this.setState({ attorneyForQueueModalId: !this.state.attorneyForQueueModalId ? item?.id : null });
    }

    render() {
        const {
            data, isLoading, error, showFinesModal, showImgUrl, reassignFineId, resendFineId, isWindowOpen,
            question, resendError, accept, isResendLoading, charged_only, not_charged_only, isReducedMode,
            protocolError, itemForWindow, notChargedData, attorneyForQueueModalId,
        } = this.state;

        const currentData = not_charged_only ? notChargedData : data;
        const finalData = currentData?.filter((item) => item.needs_charge) ?? [];

        return (
            isLoading
                ? <Spin/>
                : error
                    ? <SimpleError error={error}/>
                    : <>
                        {showFinesModal && this.props.userId
                            ? <FinesDownPaymentsDialog onClose={this.showModal.bind(this, 'showFinesModal', false)}
                                                       userId={this.props.userId}/>
                            : null
                        }

                        <div className={style.fines_controls}>
                            <Button className={style.fines_modal_button}
                                    onClick={this.showModal.bind(this, 'showFinesModal', true)}>
                                Рассрочка штрафов
                            </Button>

                            <div>Списанные:
                                <Checkbox checked={charged_only}
                                          onChange={this.showChargedOnly.bind(this)}
                                          className={style.checkbox}/>
                            </div>

                            <div>Неоплаченные:
                                <Checkbox checked={not_charged_only}
                                          onChange={this.showNotChargedOnly.bind(this)}
                                          className={style.checkbox}/>
                            </div>

                            <div>Сокращенный вид:
                                <Checkbox checked={isReducedMode}
                                          onChange={this.changeReducedMode.bind(this)}
                                          className={style.checkbox}/>
                            </div>
                        </div>

                        <div className={style.fines}>
                            <ReactTooltip id={FINES_TOOLTIP_ID} className={style.tooltip} html={true}/>

                            {finalData?.length
                                ? isReducedMode
                                    ? <>
                                        <ShortFinesInfo data={finalData}
                                                        route={this.props.route}
                                                        onShortItemClick={this.onShortItemClick.bind(this)}/>

                                        {isWindowOpen
                                            ? <Window title={'Просмотр штрафа'}
                                                      onClose={this.openWindow.bind(this, false)}>
                                                <div className={style.fines}>
                                                    <FullFinesInfoItem index={0}
                                                                       item={itemForWindow ?? {}}
                                                                       onProtocolButtonClick={this.onProtocolButtonClick
                                                                           .bind(this)}
                                                                       openResendConfirm={this.openResendConfirm
                                                                           .bind(this)}
                                                                       openResendConfirmWithDoc={this.openResendConfirm
                                                                           .bind(this)}
                                                                       onAttorneyButtonClick={this.onAttorneyButtonClick
                                                                           .bind(this)}
                                                                       attorneyForQueueModalId={attorneyForQueueModalId}
                                                                       showModal={this.showModal.bind(this)}
                                                                       showPhoto={this.showPhoto.bind(this)}/>
                                                </div>
                                            </Window>
                                            : null
                                        }
                                    </>
                                    : <>
                                        <FullFinesInfo data={finalData}
                                                       showEmailButton={this.props.AdminUser
                                                           ?.blockRules?.email_fine_gibdd_button}
                                                       openDecree={this.openDecree.bind(this)}
                                                       route={this.props.route}
                                                       openResendConfirm={this.openResendConfirm.bind(this)}
                                                       showModal={this.showModal.bind(this)}
                                                       showPhoto={this.showPhoto.bind(this)}
                                                       onProtocolButtonClick={this.onProtocolButtonClick.bind(this)}
                                                       onAttorneyButtonClick={this.onAttorneyButtonClick.bind(this)}
                                                       attorneyForQueueModalId={attorneyForQueueModalId}
                                                       openResendConfirmWithDoc={this.openResendConfirmWithDoc
                                                           .bind(this)}/>
                                    </>

                                : <NoInformation/>}
                        </div>

                        {protocolError
                            ? <Window onClose={() => this.setState({ protocolError: null })}>
                                <SimpleError error={protocolError}/>
                            </Window>
                            : null
                        }

                        {showImgUrl
                            ? <Window onClose={this.onClosePhoto.bind(this)} className={style.fine_photos}>
                                <img src={showImgUrl}
                                     style={{ height: 'auto', maxHeight: 'calc(100vh - 150px)' }}/>
                            </Window>
                            : null
                        }

                        {reassignFineId.length
                            ? <ReassignFineModal fineId={reassignFineId}
                                                 onClose={this.showModal.bind(this, 'reassignFineId', '')}/>
                            : null
                        }

                        {resendFineId
                            ? <Confirm question={question}
                                       onClose={this.closeResendConfirm.bind(this)}
                                       isWorking={isResendLoading}
                                       error={resendError}
                                       accept={accept}/>
                            : null
                        }
                    </>
        );
    }
}
