import { isString } from 'jssip/lib/Utils';
import React, { useEffect, useState } from 'react';

import { EMPTY_DATA, ONE_DAY, ONE_SECOND, ONE_WEEK_IN_DAYS } from '../../../constants';
import { useRequestHandler } from '../../../hooks/useRequestHandler';
import { Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { Status, StatusTypes } from '../../../ui/Status';
import { Request2 } from '../../../utils/request';
import { Copy } from '../../Copy';
import Spin from '../../Spin';
import { BillingCurrentTable } from '../BillingView/BillingViewCurrent/BillingCurrentTable';
import { IBillingCurrentTask } from '../BillingView/types';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';

interface IDebtInfoProps {
    debt: string;
    userId: string;
}

export const DebtInfo = React.memo((props: IDebtInfoProps) => {
    const { debt, userId } = props;
    const isDebtActive = debt !== EMPTY_DATA;

    const debtWithoutRubs: string = isString(debt) && debt.length > 1
        ? parseFloat(debt) + ''
        : EMPTY_DATA;

    const [isDebtModalOpen, openDebtModal] = useState<boolean>(false);

    return <div>
        <Copy text={debtWithoutRubs}>
            <strong>задолженность: </strong>
        </Copy>

        {isDebtActive
            ? <>
                <Status status={StatusTypes.NEGATIVE} text={''}/>
                <Link onClick={openDebtModal.bind(null, true)}>
                    {debt}
                </Link>
            </>
            : debt
        }

        {isDebtModalOpen
            ? <DebtInfoModal userId={userId} closeDebtModal={openDebtModal.bind(null)}/>
            : null
        }
    </div>;
});

interface IDebtInfoModalProps {
    closeDebtModal: () => void;
    userId: string;
}

const DebtInfoModal = React.memo((props: IDebtInfoModalProps) => {
    const { closeDebtModal, userId } = props;
    const request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    const [data, setData] = useState<IBillingCurrentTask[]>();
    const [needToReload, setNeedToReload] = useState<boolean>(false);

    const until = +new Date();
    const week = ONE_WEEK_IN_DAYS * ONE_DAY;
    const since = until - week;

    const options = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_BILLING_INFO,
            requestOptions: {
                queryParams: {
                    user_id: userId,
                    since: Math.round(since / ONE_SECOND),
                    until: Math.round(until / ONE_SECOND),
                },
            },
        };
    }, [userId]);

    const responseHandler = (data) => {
        setData(data?.current ?? []);
        setNeedToReload(false);

        return data;
    };

    const [isLoading, , error, getBillingInfo] = useRequestHandler<any[]>(request, options, responseHandler);

    useEffect(() => {
        getBillingInfo();

        return () => request.abort();
    }, [needToReload]);

    return <Window onClose={closeDebtModal.bind(null)}
                   title={'Задолженность'}
                   error={error}>
        {isLoading
            ? <Spin/>
            : <BillingCurrentTable userId={userId}
                                   filteredData={data ?? []}
                                   areAllExpanded={false}
                                   reloadData={setNeedToReload.bind(null, true)}/>
        }
    </Window>;
});
