import * as React from 'react';
import { useSelector } from 'react-redux';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import UserInfo, { UserInfoHandler } from '../../../models/user';
import { hasWebphoneActiveCall, isWebphoneReady } from '../../../reducers/webphoneReducer';
import FormatDate from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import { Status, StatusTypes, UserStatus, UserStatuses } from '../../../ui/Status';
import * as tblStyle from '../../../ui/Table/index.css';
import { chunksErrorHandling } from '../../../utils/chunksErrorHandling';
import { IBlockRules } from '../../../utils/IBlockRules';
import LS from '../../../utils/localStorage/localStorage';
import { NAVIGATION } from '../../../utils/navigation';
import { Request2 } from '../../../utils/request';
import { IStore } from '../../App/store';
import { BonusControlDialog } from '../../BonusControlDialog';
import { Copy } from '../../Copy';
import Spin, { CenteredSpin } from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import UserDrivingStyle from '../UserCardSidebar/UserDrivingStyle';
import { DebtInfo } from './DebtInfo';
import * as style from './index.css';

const Call2 = React.lazy(() => chunksErrorHandling(() => import('../../Webphone/ui/buttons/Call2')));
const LastCalls = React.lazy(() => chunksErrorHandling(() => import('../../Webphone/ui/buttons/LastCalls')));

interface IShortUserInfoBlockProps extends IStore, IBlockRules {
    userInfo: UserInfo;
    chatTagId?: string;
    tags?: any[];
}

const experimentsFormatting = (_settings: any) => {
    const settings: any = _settings && _settings
        .filter(_i => _i.setting_key === 'interface_admin.experiments')[0];
    let settingData: any;
    try {
        settingData = settings && JSON.parse(settings.setting_value);
    } catch (err) {
        settingData = {};
    }

    return settingData;
};

export const ShortUserInfoBlock = React.memo((props: IShortUserInfoBlockProps) => {
    const ls = new LS();
    const isScooterEnv = ls.envIsScooter();
    const samsaraUrl = 'https://samsara.yandex-team.ru/search?rawQuery=';
    const params = '&queues=4212&withSubQueues=4212';
    const uh = UserInfoHandler;
    const email = uh.getEmail.call(props.userInfo);
    const experiments = uh.getExperiments.call(props.userInfo);
    const isPlusEnabled = uh.isPlus.call(props.userInfo);

    const setting: any = props.AdminUser
        && props.AdminUser.rawSettings
        && experimentsFormatting(props.AdminUser.rawSettings) || {};

    const debt = uh.getDebt.call(props.userInfo);
    const userId = uh.getId.call(props.userInfo);

    const [historyIsShown, showHistory] = React.useState(false);
    const [verifyIsShown, showVerify] = React.useState(false);
    const [isWorking, setWorking] = React.useState(false);
    const [error, setError] = React.useState(null);
    const [bonusIsShown, showBonus] = React.useState(false);
    const isWebphoneAvailable = useSelector(isWebphoneReady);
    const hasActiveCall = useSelector(hasWebphoneActiveCall);

    const request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });
    const accept = () => {
        setWorking(true);
        setError(null);
        request.exec(REQUESTS.EDIT_USER, {
            body: {
                id: userId,
                is_phone_verified: false,
            },
        })
            .then(() => {
                showVerify(false);
                setWorking(false);
            })
            .catch((err) => {
                setWorking(false);
                setError(err);
            });
    };

    const bonuses = uh.getBonuses.call(props.userInfo);
    const status = uh.getStatus.call(props.userInfo);
    const ya_account = uh.getYaAccount.call(props.userInfo);
    const genderType = uh.getPassportGender.call(props.userInfo);

    const notChats = !location.hash.includes(NAVIGATION.ROUTES?.CHATS?.uri);

    return <div className={style.container}>
        <div><strong>пол:</strong> {genderType}</div>
        {
            historyIsShown
            && <PhoneHistory user_id={userId}
                             onClose={showHistory.bind(null, false)}/>
        }
        {
            verifyIsShown
            && <Confirm question={'Верифицировать номер?'}
                        error={error}
                        isWorking={isWorking}
                        accept={accept}
                        title={uh.getPhone.call(props.userInfo)}
                        onClose={showVerify.bind(null, false)}/>
        }
        {
            bonusIsShown
            && <BonusControlDialog onClose={showBonus.bind(null, false)}
                                   userId={userId}/>
        }
        <div>
            <strong>тел: </strong>
            <Status status={uh.isPhoneVerified.call(props.userInfo)
                ? StatusTypes.POSITIVE
                : StatusTypes.NEGATIVE}
                    text={''}/>
            <Copy text={uh.getPhone.call(props.userInfo)}
                  internalTooltipId={'user_phone'}>
                {uh.getPhone.call(props.userInfo)}
            </Copy>
            {isWebphoneAvailable
                ? <React.Suspense fallback={<CenteredSpin size={'s'}/>}>
                    <Call2 buttonClassName={style.callButton}
                           containerClassName={style.callButtonContainer}
                           disabled={hasActiveCall}
                           userInfo={props.userInfo}
                           chatTagId={props.chatTagId}
                           client={uh.getPhone.call(props.userInfo)}/>
                    <LastCalls userId={userId}/>
                </React.Suspense>
                : null
            }
        </div>
        <div><strong>статус:</strong> <UserStatus text={status}/></div>
        {
            props?.BlockRules?.BlockReason && status !== UserStatuses.ACTIVE
                ? <div className={style.blocking_reason}>
                    <strong>stage</strong>: {uh.getRegistrationChatStage.call(props.userInfo)}
                </div>
                : null
        }
        {notChats
            ? <div className={style.phone_controls}>
                (<Link onClick={showHistory.bind(null, true)}>
                    история
                </Link> / <Link onClick={showVerify.bind(null, true)}>
                    проверить
                </Link>)
            </div>
            : null
        }
        {email && <div className={style.user_info_block_item}>
            <Copy text={email}><strong>email: </strong></Copy>
            <span className={style.user_info_block_item_email}>
                <Link href={`${samsaraUrl}${email}${params}`} target={'_blank'}> {email}</Link></span>
        </div>}
        <div><strong>бонусы: </strong>
            {
                props.BlockRules
                && props.BlockRules['bonus_button']
                && !location.href.includes(NAVIGATION.ROUTES.CHATS.uri)
                    ? <Link onClick={showBonus.bind(null, true)}>
                        {bonuses}
                    </Link>
                    : bonuses
            }
        </div>
        <div>
            <strong>Я.Плюс: </strong>
            <span className={`${isPlusEnabled ? style.is_plus_enabled : style.is_plus_disabled}`}>{ya_account}
                <i> ({isPlusEnabled ? 'вкл' : 'выкл'})</i></span>
        </div>

        <DebtInfo userId={userId} debt={debt}/>

        {notChats && experiments.length && <>
            <div><strong>эксперименты: </strong>
                <u><span data-tip={true} data-for={'experiments'} className={'problem-item'}>
                    {experiments.length}
                </span></u>
            </div>

            <ReactTooltip id={'experiments'}
                          aria-haspopup="true"
                          type="info"
                          effect="solid">
                {experiments
                    .map((item: any, index: number) => {
                        return <div key={index}>
                            {item} : <strong>{setting && setting[item] || EMPTY_DATA}</strong>
                        </div>;
                    })}
            </ReactTooltip>
        </> || null}
        {!isScooterEnv
            ? <UserDrivingStyle className={style.aggression} userId={userId}/>
            : null
        }
    </div>;
});

class PhoneHistory extends React.Component<any, any> {
    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });
    state: any = {
        isLoading: true,
        data: [],
        err: null,
    };

    componentDidMount(): void {
        this.setState({
            isLoading: true,
        }, () => {
            this.props.user_id
            && this.request.exec(REQUESTS.PHONE_HISTORY, {
                queryParams: { user_id: this.props.user_id },
            })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        data: response,
                    });
                })
                .catch((err) => {
                    this.setState({
                        isLoading: false,
                        err,
                    });
                });
        });
    }

    render() {
        const count = this.state.data && this.state.data.history
            && Array.isArray(this.state.data.history)
            && this.state.data.history.length || EMPTY_DATA;

        return <Window onClose={this.props.onClose}
                       error={this.state.err}
                       title={`История работы с телефоном: ${count}`}>
            {this.state.isLoading
                ? <Spin/>
                : <div>
                    {count && <table className={tblStyle.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>телефон</th>
                                <th>действие</th>
                                <th>дата действия</th>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                this.state.data.history
                                && this.state.data.history.map((_el: any, index: number) => {
                                    return <tr key={index}>
                                        <td>{++index}</td>
                                        <td>{_el.phone}</td>
                                        <td>{_el.event}</td>
                                        <td><FormatDate value={_el.timestamp * ONE_SECOND} withSecond={true}/></td>
                                    </tr>;
                                })
                            }
                        </tbody>
                    </table>
                        || <NoInformation/>
                    }
                </div>
            }
        </Window>;
    }
}
