import moment from 'moment';
import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import DatePicker from '../../../ui/DatePicker';
import { Input } from '../../../ui/Input';
import { Request2 } from '../../../utils/request';
import { RESPONSE_STATUSES } from '../../SimpleError';
import * as styles from './index.css';
import { REQUESTS, USER_ACTIONS_VIEW_REQUESTS } from './request';
import { UserActionMap } from './UserActionMap';
import { friendlyData } from './UserActionsFriendlyData';
import { UserActionTable } from './UserActionTable';

interface IUserActionsViewProps {
    userId: string;
}

export interface IUserRequest {
    app_version?: string;
    code?: number;
    device_id?: string;
    experiments?: [];
    host?: string;
    ip?: string;
    latitude?: string;
    longitude?: number;
    path?: string;
    post?: string;
    query?: string;
    reqid?: string;
    response?: string;
    timestamp?: number;
    user_agent?: string;
    name?: string;
    car_number?: string;
    ui_message?: string;
    landing_titles?: string[];
    isFriendly?: boolean;
}

interface IUserActionsViewState {
    [x: number]: number;

    error: Error | null;
    isLoading: boolean;
    requests: IUserRequest[];
    since: number;
    until: number;
    timeError: boolean;
    last_item_ts: number;
    isMap: boolean;
    isBase: boolean;
    isOnlyErrors: boolean;
    filter: string;
}

const DEFAULT_SEARCH_SUBSTR = 2;

export default class UserActionsView extends React.Component<IUserActionsViewProps, IUserActionsViewState> {
    state: IUserActionsViewState = {
        error: null,
        isLoading: false,
        requests: [],
        since: +moment().subtract(DEFAULT_SEARCH_SUBSTR, 'd'),
        until: new Date().getTime(),
        timeError: false,
        last_item_ts: 0,
        isMap: false,
        isBase: false,
        isOnlyErrors: false,
        filter: '',
    };

    request = new Request2({ requestConfigs: USER_ACTIONS_VIEW_REQUESTS });

    getRequests(last_item_ts: number) {
        const { userId } = this.props;
        const { since, until } = this.state;

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_REQUESTS, {
                queryParams: {
                    user_id: userId,
                    until: Math.floor(+last_item_ts || until / ONE_SECOND),
                    since: Math.floor(since / ONE_SECOND),
                },
            })
                .then((response) => {
                    let { requests = [], complete } = response;

                    requests = friendlyData(requests);

                    if (+last_item_ts) {
                        requests = [
                            ...this.state.requests,
                            ...requests,
                        ];
                    }

                    this.setState({
                        last_item_ts: !complete ? requests[requests.length - 1]
                            && requests[requests.length - 1].timestamp : 0,
                        requests,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                    });
                });
        });
    }

    modifyDate(key, value) {
        this.setState({
            timeError: false,
        });

        this.setState({
            [key]: +value,
        });
    }

    onChange(key, value) {
        this.setState({
            [key]: value,
        });
    }

    getMore() {
        const { last_item_ts } = this.state;
        this.getRequests(last_item_ts);
    }

    getDataForTimelapse() {
        this.setState({
            last_item_ts: 0,
        }, () => {
            this.getMore();
        });
    }

    componentDidMount() {
        this.getRequests(0);
    }

    componentDidUpdate(prevProps: Readonly<IUserActionsViewProps>, prevState: Readonly<IUserActionsViewState>) {
        const { since, until } = this.state;

        if (since !== prevState.since
            || until !== prevState.until) {
            if (since && until && !(since < until)) {
                this.setState({
                    timeError: true,
                });
            }
        }
    }

    onChangeFilter(filter) {
        this.setState({
            filter,
        });
    }

    render() {
        let { error, isLoading, since, until, requests, last_item_ts, isMap, isBase, isOnlyErrors } = this.state;

        if (isBase) {
            requests = requests.filter((request) => {
                return request.name;
            });
        }

        if (isOnlyErrors) {
            requests = requests.filter((request) => {
                return request.code !== RESPONSE_STATUSES.CODE_200;
            });
        }

        return (
            <>
                <div className={styles.header}>
                    <DatePicker className={styles.since}
                                value={since}
                                placeholder={'C'}
                                onChange={this.modifyDate.bind(this, 'since')}/>
                    <DatePicker className={styles.until}
                                value={until}
                                placeholder={'По'}
                                onChange={this.modifyDate.bind(this, 'until')}/>
                    <Button className={styles.button}
                            onClick={this.getDataForTimelapse.bind(this)}>Запросить</Button>
                    <Input className={styles.filter}
                           value={this.state.filter}
                           placeholder={'Локальный фильтр'}
                           onChange={this.onChangeFilter.bind(this)}/>
                    <div className={styles.is_base}>
                        <span>
                            {isBase ? 'Выкл базовые события' : 'Вкл базовые события'}
                        </span>
                        <Checkbox checked={isBase}
                                  onChange={this.onChange.bind(this, 'isBase', !isBase)}/>
                    </div>
                    <div className={styles.is_map}>
                        <span>
                            {isMap ? 'Выкл карту' : 'Вкл карту'}
                        </span>
                        <Checkbox checked={isMap}
                                  onChange={this.onChange.bind(this, 'isMap', !isMap)}/>
                    </div>
                    <div className={styles.is_error}>
                        <span>
                            {isOnlyErrors ? 'Только ошибки' : 'Все события'}
                        </span>
                        <Checkbox checked={isOnlyErrors}
                                  onChange={this.onChange.bind(this, 'isOnlyErrors', !isOnlyErrors)}/>
                    </div>
                </div>
                <div className={isMap ? styles.container : undefined}>
                    <UserActionTable withMap={isMap}
                                     error={error}
                                     isLoading={isLoading}
                                     requests={requests?.filter(el => {
                                         return !this.state.filter || el.path?.includes(this.state.filter);
                                     })}
                                     lastItemTs={last_item_ts}
                                     getMore={this.getMore.bind(this)}/>
                    {isMap &&
                    <UserActionMap requests={requests}/>
                    }
                </div>
            </>
        );
    }
}
