import * as React from 'react';

import { TAG_ACTION } from '../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import * as tblStyles from '../../../../ui/Table/index.css';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../../request';
import * as styles from './index.css';

interface IUserBanHistoryProps {
    onClose: () => void;
    userInfo: any;
}

interface IUserBanHistoryState {
    error: Error | null;
    isLoading: boolean;
    currentTagsBans: any[];
    historyTagsBans: any[];
}

export class UserBanHistory extends React.Component<IUserBanHistoryProps, IUserBanHistoryState> {
    state: IUserBanHistoryState = {
        error: null,
        isLoading: false,
        currentTagsBans: [],
        historyTagsBans: [],
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            Promise.all([
                this.props.userInfo?.id && this.request.exec(REQUESTS.GET_USER_TAGS, {
                    queryParams: {
                        object_id: this.props.userInfo.id,
                    },
                }),
                this.props.userInfo?.id && this.request.exec(REQUESTS.GET_TAGS_HISTORY, {
                    queryParams: {
                        object_id: this.props.userInfo.id,
                        duration: '30d',
                    },
                }),
            ])
                .then(response => {
                    const currentTagResponse = response[0];
                    const historyTagResponse = response[1];

                    const currentTags = currentTagResponse?.records || [];
                    const currentTagsBans = currentTags
                        .filter((tag: any) => tag.tag && tag.tag.indexOf('blocked_') === 0);

                    const historyTags = historyTagResponse?.records || [];
                    const historyTagsBans = historyTags
                        .filter((tag: any) => tag.tag_name && tag.tag_name.indexOf('blocked_') === 0);

                    currentTagsBans.forEach((currentTagBan: any) => {
                        const { object_id } = currentTagBan;

                        const currentTagHistory = historyTagsBans.find((historyTagBan: any) => {
                            return historyTagBan.action === TAG_ACTION.ADD && historyTagBan.object_id === object_id;
                        });

                        if (currentTagHistory) {
                            currentTagBan.user_data = Object.assign({}, currentTagHistory.user_data_full);
                            currentTagBan.timestamp = currentTagHistory.timestamp;
                        }
                    });

                    this.setState({
                        currentTagsBans,
                        historyTagsBans,
                        isLoading: false,
                    });
                }).catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IUserBanHistoryProps>) {
        if (this.props.userInfo?.id !== prevProps.userInfo?.id) {
            this.getData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const { error, isLoading } = this.state;
        const { onClose } = this.props;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Блокировки'}
                    error={error}>
                {isLoading ? <Spin /> : <div>
                    <div className={styles.ban_window}>
                        <h3>Активные блокировки:</h3>
                        {this.state.currentTagsBans && Array.isArray(this.state.currentTagsBans)
                        && this.state.currentTagsBans.length ?
                            <table className={`${tblStyles.table} ${tblStyles.clearFirstOpacity}`}>
                                <tbody>
                                    <tr>
                                        <th>Название тега</th>
                                        <th>Время навешивания</th>
                                        <th>Пользователь</th>
                                        <th>Комментарий</th>
                                    </tr>
                                    {this.state.currentTagsBans.map((currentTag: any, index) => {

                                        let userDisplayName = 'Пользователь';
                                        if (currentTag.user_data) {
                                            userDisplayName = UserInfoHandler
                                                .getPrintNameWithoutPn
                                                .call(currentTag.user_data);
                                        }

                                        return <tr key={index}>
                                            <td>{currentTag.display_name
                                                || currentTag.tag
                                                || currentTag.tag_id
                                                || EMPTY_DATA}
                                            </td>
                                            <td>
                                                {currentTag.timestamp
                                                    ? <FormatDate value={currentTag.timestamp * ONE_SECOND}
                                                                  withSecond={true}/>
                                                    : EMPTY_DATA}

                                            </td>
                                            <td>
                                                {currentTag.user_data
                                                    ? <Link href={`#/clients/${currentTag.user_data.id}/info`}>
                                                        {userDisplayName}
                                                    </Link>
                                                    : EMPTY_DATA}
                                            </td>
                                            <td>
                                                {currentTag.comment ? currentTag.comment : ''}
                                            </td>
                                        </tr>;
                                    })}
                                </tbody>
                            </table>
                            : <h4>Нет активных блокировок</h4>}
                        <h3>История блокировок:</h3>
                        {this.state.historyTagsBans && Array.isArray(this.state.historyTagsBans)
                        && this.state.historyTagsBans.length
                            ? <table className={`${tblStyles.table} ${tblStyles.clearFirstOpacity}`}>
                                <tbody>
                                    <tr>
                                        <th>Название тега</th>
                                        <th>Действие</th>
                                        <th>Время</th>
                                        <th>Пользователь</th>
                                        <th>Комментарий</th>
                                    </tr>
                                    {this.state.historyTagsBans.map((historyTag: any, index) => {

                                        let userDisplayName = 'Пользователь';
                                        if (historyTag.user_data_full) {
                                            userDisplayName = UserInfoHandler
                                                .getPrintNameWithoutPn
                                                .call(historyTag.user_data);
                                        }

                                        return <tr key={index}>
                                            <td>{historyTag.tag_display_name
                                                || historyTag.tag_name
                                                || historyTag.tag_id
                                                || EMPTY_DATA}
                                            </td>
                                            <td className={`${styles[historyTag.action]}`}>{historyTag.action}</td>
                                            <td>
                                                {historyTag.timestamp
                                                    ? <FormatDate value={historyTag.timestamp * ONE_SECOND}
                                                                  withSecond={true}/>
                                                    : EMPTY_DATA}

                                            </td>
                                            <td>
                                                {historyTag.user_data_full
                                                    ? <Link href={`#/clients/${historyTag.user_data_full.id}/info`}>
                                                        {userDisplayName}
                                                    </Link>
                                                    : EMPTY_DATA}
                                            </td>
                                            <td>
                                                {historyTag.tag_details && historyTag.tag_details.comment
                                                    ? historyTag.tag_details.comment
                                                    : ''}
                                            </td>
                                        </tr>;
                                    })}</tbody>
                            </table>
                            : <h4>Нет истории блокировок</h4>}
                    </div>
                </div>}
            </Window>
        );
    }
}
