import * as React from 'react';

import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { ErrorSign, ErrorsModal } from '../../../../ui/ErrorBar';
import { Window } from '../../../../ui/FullModal';
import ProfileProgressBar from '../../../../ui/ProfileProgressBar/index';
import { Request2 } from '../../../../utils/request';
import { CenteredSpin } from '../../../Spin';
import { CC_REQUESTS, REQUESTS } from '../../CCView/request';
import { DrivingAchievements } from './DrivingAchievements/component';
import styles from './index.css';
import { IDrivingAchievements, IDrivingProfile, IDrivingStyle, IDrivingStyleItem } from './types';

interface IUserDrivingStyleProps {
    className?: string;
    userId: string;
}

export const UserDrivingStyle = React.memo((props: IUserDrivingStyleProps) => {
    const { className, userId } = props;
    const [isModalOpen, openModal] = React.useState<boolean>(false);
    const [isErrorModalOpen, openErrorModal] = React.useState<boolean>(false);

    const request = new Request2({ requestConfigs: CC_REQUESTS });

    const requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_USER_PROFILE,
            requestOptions: {
                queryParams: {
                    user_id: userId,
                },
            },
        };
    }, [userId]);

    const [
        isLoading,
        data,
        error,
        getUserDrivingStyle,
    ] = useRequestHandler<IDrivingProfile | null>(request, requestOptions);

    const userDrivingStyle: IDrivingStyle = data?.driving_style ?? {} as IDrivingStyle;
    const achievements: IDrivingAchievements[] = data?.achievements ?? [] as IDrivingAchievements[];
    const currentDrivingStyleItem: IDrivingStyleItem | undefined = userDrivingStyle.items
        ?.find((el) => el.focused);

    const title = userDrivingStyle?.last_change?.title || 'Профиль вождения';
    const descriptionsTitle = currentDrivingStyleItem?.descriptions?.title;

    React.useEffect(() => {
        getUserDrivingStyle();
    }, [requestOptions]);

    return <>
        {error && isErrorModalOpen
            ? <ErrorsModal onClose={() => {
                openErrorModal(false);
            }}
                           errors={[error]}/>
            : null
        }
        {isModalOpen && currentDrivingStyleItem
            ? <Window className={styles.modal}
                      title={title}
                      onClose={() => openModal(false)}>
                <ProfileProgressBar size={'l'} currentDrivingStyleItem={currentDrivingStyleItem}/>
                <h2>{currentDrivingStyleItem.caption}</h2>
                {descriptionsTitle && <h3>{descriptionsTitle}</h3>}
                {currentDrivingStyleItem.descriptions?.items?.map(({ text, icon }, index) => {
                    return <div key={`${descriptionsTitle}-${index}`} className={styles.description}>
                        <img src={icon} height="50" width="50"/>
                        <span>{text}</span>
                    </div>;
                })}

                <DrivingAchievements achievements={achievements}/>
            </Window>
            : null
        }
        <div className={`${styles.aggressionRing} ${className ?? ''}`} title={title}>
            {error
                ? <ErrorSign onClick={openErrorModal.bind(null, true)}/>
                : isLoading
                    ? <CenteredSpin size={'s'}/>
                    : currentDrivingStyleItem
                        ? <ProfileProgressBar onClick={() => openModal(true)}
                                              currentDrivingStyleItem={currentDrivingStyleItem}
                                              size={'s'}/>
                        : null
            }
        </div>
    </>;
});
