import moment from 'moment';
import * as React from 'react';

import { WithSearchControls } from '../../../../../decorators/SearchControls';
import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import Checkbox from '../../../../ui/Checkbox';
import FormatDate from '../../../../ui/FormatDate';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import { TLabel } from '../../../../ui/Table';
import * as styleTable from '../../../../ui/Table/index.css';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { getStatusType } from '../../../TagsDeepHistory2/HistoryVirtualList';
import { CommentBlock } from '../CommentBlock';
import * as style from '../index.css';
import { shouldShowTagComment } from '../shouldShowTagComment';
import { ChatLink } from '../UserCurrentTagList/ChatLink';
import { requestConfigs, REQUESTS } from './request';

const SINCE_SUBTRACT = 30;
export const HISTORY_TYPES = {
    USER: 'USER',
    ACCOUNT: 'ACCOUNT',
};

interface ITagsHistoryState {
    duration: number;
    tagsHistoryIsLoading: boolean;
    tagHistory: any[];
    tagsHistoryError: null | Error;
    filterByTags: any[];
    isContractFines: boolean;
    contractFines: Record<string, any>[];
    fineTagsList: string[];
}

@WithSearchControls({ until: moment(), since: moment().subtract(SINCE_SUBTRACT, 'd') })
export class TagsHistory extends React.Component<any, ITagsHistoryState> {
    state: ITagsHistoryState = {
        duration: 30,
        tagsHistoryIsLoading: false,
        tagHistory: [],
        tagsHistoryError: null,
        filterByTags: [],
        isContractFines: false,
        contractFines: [],
        fineTagsList: [],
    };
    request = new Request2({
        requestConfigs,
    });
    prefix = 'fines.history_fines_contract';

    componentDidMount(): void {
        this.getFinesSettings();
        this.getTagsHistory();
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if ((this.props.object_id !== prevProps.object_id
            || this.props.since !== prevProps.since
            || this.props.until !== prevProps.until) ||
            (prevProps.isUpdating !== this.props.isUpdating && this.props.isUpdating)) {
            this.getTagsHistory();
        }
    }

    getTagsHistory() {
        const object_id = this.props.object_id;

        this.setState({
            tagsHistoryIsLoading: true, tagsHistoryError: null,
        }, () => {
            const type = this.props.historyType || HISTORY_TYPES.USER;
            const [config, params] = [
                REQUESTS[`GET_${type}_HISTORY_TAGS`],
                {
                    queryParams: {
                        object_id,
                        since: this.props.since,
                        until: this.props.until,
                    },
                },
            ];

            this.request.exec(config, params)
                .then(response => {
                    const tagHistory = response.records || [];
                    this.setState({
                        tagsHistoryIsLoading: false,
                        tagHistory,
                    }, () => {
                        if (this.state.isContractFines) {
                            this.getContractFines();
                        }
                    });
                })
                .catch((tagsHistoryError) => {
                    this.setState({ tagsHistoryIsLoading: false, tagsHistoryError });
                });
        });
    }

    onSelectFilter(filterByTags) {
        this.setState({ filterByTags });
    }

    getFinesSettings() {
        this.setState({
            tagsHistoryIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_FINES_SETTINGS, { queryParams: { prefix: this.prefix } })
                .then(response => {
                    const settingsString = response?.settings?.[0]?.setting_value;
                    let finesList = [];
                    if (isValidJSONString(settingsString)) {
                        finesList = JSON.parse(settingsString);
                    }

                    const fineTagsList: string[] = finesList.map((el: { tag: string; tag_description: string }) => {
                        return el.tag;
                    });

                    this.setState({
                        tagsHistoryIsLoading: false,
                        fineTagsList,
                    });
                })
                .catch(error => {
                    this.setState({
                        tagsHistoryError: error,
                        tagsHistoryIsLoading: false,
                    });
                });
        });
    }

    onCheckboxChange() {
        this.setState({ isContractFines: !this.state.isContractFines }, () => {
            this.getContractFines();
        });
    }

    getContractFines() {
        const { isContractFines, contractFines, tagHistory, fineTagsList } = this.state;

        if (isContractFines && !contractFines.length) {
            const filteredData = tagHistory?.filter((el) => {
                return fineTagsList.includes(el.tag_name);
            });

            this.setState({ contractFines: filteredData ?? [] });
        }
    }

    render() {
        const {
            tagsHistoryError, tagsHistoryIsLoading, tagHistory, filterByTags,
            isContractFines, contractFines,
        } = this.state;

        if (tagsHistoryError) {
            return <SimpleError error={tagsHistoryError}
                                data={{ label: 'Ошибка при загрузке истории тегов' }}/>;
        }

        if (tagsHistoryIsLoading) {
            return <Spin/>;
        }

        const tags = !isContractFines ? tagHistory : contractFines;
        const optionsObj: any = {};
        tags?.forEach((el: any) => {
            optionsObj[el.tag_name] = el.tag_display_name;
        });
        const options: any[] = Object.entries(optionsObj)?.map(el => {
            return { text: el[1], value: el[0] };
        });

        return <div>
            <div className={style.filter_controls}>
                <Select options={options}
                        initialValues={filterByTags}
                        disabled={!options?.length}
                        placeholder={'фильтр по тегам'}
                        multiSelect
                        onSelect={this.onSelectFilter.bind(this)}/>
                <div className={style.checkbox}>
                    Штрафы по контракту
                    <Checkbox checked={isContractFines} onChange={this.onCheckboxChange.bind(this)}/>
                </div>
            </div>
            <table className={styleTable.table}>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Действие</th>
                        <th/>
                        <th>Тег/Комментарий</th>
                        <th/>
                        <th>Автор</th>
                        <th/>
                    </tr>
                </thead>
                <tbody>
                    {tags.filter((item) => {
                        const match: any = filterByTags?.indexOf?.(item?.tag_name) > -1;
                        if (!filterByTags?.length) {
                            return true;
                        }

                        return match;
                    }).map((item: any, index: number) => {
                        const showTagComment = shouldShowTagComment(item.tag_name, this.props.BlockRules);

                        return <tr key={index}>
                            <td>{index + 1}</td>
                            <td>
                                <TLabel className={style.history_tag_label}
                                        status={getStatusType(item.action)}
                                        text={item.action || EMPTY_DATA}/>
                            </td>
                            <td><FormatDate value={item.timestamp * ONE_SECOND} withSecond={true}/></td>
                            <td>
                                <div>
                                    <strong>{item.tag_display_name || item.tag_name || EMPTY_DATA}</strong>
                                </div>
                                {showTagComment
                                    && <CommentBlock comment={item.tag_details
                                        && item.tag_details.comment || EMPTY_DATA}/>}
                            </td>
                            <td><ChatLink item={item}/></td>
                            <td>
                                {
                                    item.user_data_full && item.user_data_full.id
                                        ? <Link target={'_blank'} href={`#/clients/${item.user_data_full.id}/info`}>
                                            {item.user_data_full && item.user_data_full.username || EMPTY_DATA}
                                        </Link>
                                        : <span>{item.user_data_full
                                            && item.user_data_full.username
                                            || EMPTY_DATA}
                                        </span>
                                }
                            </td>
                            <td>
                                <div>
                                    <Link onClick={this.props.showTagDetails.bind(null, item)}>
                                    info
                                    </Link>
                                </div>
                                {
                                    item.event_id && <div>
                                        <Link onClick={this.props.showTagEventDetails.bind(null, item)}>
                                        details
                                        </Link>
                                    </div>
                                }
                            </td>
                        </tr>;
                    })}
                </tbody>
            </table>
        </div>;
    }
}
