import React from 'react';

import { Dict } from '../../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import { useRequestQueueHandler } from '../../../../../hooks/useRequestQueueHandler';
import FormatDate from '../../../../../ui/FormatDate';
import { Window } from '../../../../../ui/FullModal';
import { JsonModal } from '../../../../../ui/FullModal/JsonModal';
import { Link } from '../../../../../ui/Link';
import * as styleTable from '../../../../../ui/Table/index.css';
import { isValidJSONString } from '../../../../../utils/isValidJSONString';
import { Request2 } from '../../../../../utils/request';
import CarNumber from '../../../../CarNumber';
import Spin from '../../../../Spin';
import styles from './index.css';

interface UserTagRoutModal {
    onClose: () => void;
    tag: Dict<any>;
}

interface Item {
    car_id: string;
    car_number: string;
    end_time: number;
    start_time: number;
    tag_id: string;
}

const REQUESTS = {
    GET_DATA: 'GET_DATA',
};

const requestConfigs = {
    [REQUESTS.GET_DATA]: {
        api: 'car/tag/details',
        cgiObj: {
            tag_id: null,
        },
    },
};

const UserTagRoutModal = (props: UserTagRoutModal) => {
    const { onClose, tag } = props;

    let tasks: Item[] = [];
    if (isValidJSONString(tag?.route?.tasks)) {
        tasks = JSON.parse(tag?.route?.tasks);
    }

    const tags = tasks?.filter(task => task.tag_id).map(task => task.tag_id);
    const request = new Request2({ requestConfigs });

    const getQueue = () => {
        return tags.map((tag_id) => {
            return {
                requestName: REQUESTS.GET_DATA,
                requestOptions: {
                    queryParams: {
                        tag_id,
                    },
                },
            };
        });
    };

    const responseHandler = (response => {
        return response.reduce((_p, _c) => {
            const tag_id = _c?.history?.[0]?.tag_id;
            if (!_p.hasOwnProperty(tag_id)) {
                _p[tag_id] = _c;
            }

            return _p;
        }, {});
    });
    const [currentQueue] = React.useState(getQueue());
    const [
        isLoading,
        response, ,
        getCarTags,
    ] = useRequestQueueHandler<Dict<any>>(request, currentQueue, responseHandler);
    const [detailID, showDetails] = React.useState(null);

    React.useEffect(() => {
        getCarTags();
    }, []);

    return <Window onClose={onClose} title={'Маршрут'}>
        Ссылка на <Link target={'_blank'}
                        href={`https://courier.yandex.ru/mvrp-map/#${tag?.route?.routing_task_id}`}>маршрут</Link>
        <div>Теги: {isLoading ? `Загрузка информации...` : `Загружено`}</div>
        <table className={`${styleTable.table}`}>
            <thead>
                <tr>
                    <th>#</th>
                    <th>старт / финиш</th>
                    <th/>
                    <th/>
                    <th className={styles.comment}>комментарий</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {
                    tasks.map((el, index) => {
                        const detail = response?.[el.tag_id] || {};

                        return <tr key={index}>
                            <td>{index + 1}</td>
                            <td>
                                <div>
                                    <strong> <FormatDate value={el.start_time * ONE_SECOND} onlyDate/>
                                    </strong>
                                </div>
                                <div>{
                                    el.start_time !== el.end_time
                                        ? <>
                                            <FormatDate value={el.start_time * ONE_SECOND}
                                                        onlyTime
                                                        withSecond/> - <FormatDate value={el.end_time * ONE_SECOND}
                                                                                   onlyTime
                                                                                   withSecond/>
                                        </>
                                        : <FormatDate value={el.start_time * ONE_SECOND} onlyTime withSecond/>
                                }

                                </div>
                            </td>
                            <td>{el.car_number && <CarNumber value={el.car_number}/> || EMPTY_DATA}</td>
                            <td>{
                                el.car_number && <Link target={'_blank'} href={`#/cars/${el.car_id}/info`}>
                                авто
                                </Link> || EMPTY_DATA}
                            </td>

                            <td>

                                {
                                    isLoading ? <Spin size={'s'}/> : <>
                                        <div><strong>{detail?.tag?.tag}</strong></div>
                                        <div>{detail?.tag?.comment || EMPTY_DATA}</div>
                                    </>
                                }
                            </td>
                            <td><Link onClick={showDetails.bind(null, el.tag_id)}
                                      disable={isLoading}>детали</Link></td>
                        </tr>;
                    })
                }
            </tbody>
        </table>
        {
            detailID !== null
            && <JsonModal onClose={showDetails.bind(null, null)}
                          title={`Детали по тегу ${response?.[detailID || '']?.tag?.tag}`}
                          obj={response?.[detailID || '']}/>
        }
    </Window>;
};

export default UserTagRoutModal;
