import * as React from 'react';

import { Dict } from '../../../../../../types';
import { ONE_SECOND } from '../../../../../constants';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { IHistoryItem } from '../types';
import { USER_MODEL_DESC } from '../userModel';
import * as style from './index.css';

interface IHistoryViewProps {
    history: IHistoryItem[] | null;
}

export const HistoryView = React.memo((props: IHistoryViewProps) => {
    const { history } = props;

    return <div>
        <h2>История изменений</h2>
        <div className={style.history_container}>
            {history?.length
                ? <table>
                    <tbody>
                        {history?.map(historyItem => {
                            return <HistoryRow key={historyItem.event_id} historyItem={historyItem}/>;
                        })}
                    </tbody>
                </table>
                : <span className={style.empty_history}>История изменений пустая</span>
            }
        </div>
    </div>;
});

interface IHistoryRowProps {
    historyItem: IHistoryItem;
}

interface IHistoryRowState {
    newDataDiffElements: JSX.Element[];
    newDataLength: number;
    historyItem: IHistoryItem | null;
    ifFullShow: boolean;
}

interface IDiffData {
    old: string | boolean | number | null;
    new: string | boolean | number | null;
}

const SHORT_SHOW_LENGTH = 3;

class HistoryRow extends React.Component<IHistoryRowProps, IHistoryRowState> {
    state = {
        newDataDiffElements: [],
        newDataLength: 0,
        historyItem: null,
        ifFullShow: false,
    };

    componentDidMount() {
        const { historyItem } = this.props;
        const newData = historyItem.newData && Object.entries(historyItem.newData);
        let newDataLength = 0;

        const newDataDiffElements = newData
            && newData.reduce((res: JSX.Element[], curr) => {
                const [objKey, objValue] = curr;
                Object.entries(objValue).forEach((diff: [string, IDiffData]) => {
                    newDataLength++;
                    res.push(<DiffElement key={`${objKey}.${diff[0]}`} objKey={objKey} diff={diff}/>);
                });

                return res;
            }, []);
        this.setState({ newDataDiffElements, newDataLength, historyItem });
    }

    changeShowFull() {
        this.setState({ ifFullShow: !this.state.ifFullShow });
    }

    render() {
        const { historyItem = {} as Dict<any> , newDataLength, ifFullShow, newDataDiffElements } = this.state;

        return historyItem && <tr className={style.diff_row}>
            <td>
                <FormatDate value={historyItem.timestamp * ONE_SECOND}/>
                <div>
                    <Link href={`#/clients/${historyItem.user_id}/info`}>Пользователь</Link>
                </div>
                {newDataLength > SHORT_SHOW_LENGTH
                    ? <div>
                        <Link onClick={this.changeShowFull.bind(this)}>
                            {`${ifFullShow ? `Скрыть` : `Развернуть`}(${newDataLength})`}
                        </Link>
                    </div>
                    : null}
            </td>
            <td>
                {newDataDiffElements
                    ? ifFullShow
                        ? newDataDiffElements
                        : newDataDiffElements.slice(0, SHORT_SHOW_LENGTH)
                    : <div className={style.diff}>
                        <span className={style.no_diff}>Изменений не было</span>
                    </div>}
            </td>
        </tr>;
    }
}

const DOC_TYPE_KEY = 'doc_type';

interface IDiffElementProps {
    objKey: string;
    diff: [string, IDiffData];
}

const DiffElement = React.memo((props: IDiffElementProps) => {
    const { diff, objKey } = props;
    const [key, diffValue] = diff;
    const { old: oldValue, new: newValue } = diffValue;

    if (key === DOC_TYPE_KEY) {
        return null;
    }

    return <div className={style.diff}>
        <span className={style.key}>
            {
                `${USER_MODEL_DESC?.[objKey]?.title}`
                + `-`
                + `${USER_MODEL_DESC?.[objKey]?.fields?.[key]?.display_name ?? key}`
            }
        </span>:
        <span>
            <span className={`${style.value} ${style.old}`}>{oldValue?.toString()}</span>
            →
        </span>
        <span className={`${style.value} ${style.new}`}>{newValue?.toString()}</span>
    </div>;
});
