import * as React from 'react';

import { Dict } from '../../../../../types';
import { Button, CancelButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { CORE_USER_INFO_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { HistoryView } from './HistoryView/component';
import * as style from './index.css';
import { IHistoryItem } from './types';
import { USER_MODEL, USER_MODEL_DESC } from './userModel';

const PASSPORT_KEY = 'passport';
const DRIVING_LICENSE_KEY = 'driving_license';

interface IUserEditorProps {
    onClose: () => void;
    userId: string;
}

interface IUserInfo {
    commonUserInfo: Dict<string | boolean | number | Date | null>;
    userPassportInfo: Dict<string | boolean | number | Date | null>;
    userDrivingLicenseInfo: Dict<string | boolean | number | Date | null>;
}

interface IUserEditorState {
    userInfo: IUserInfo | null;
    userInfoInit: IUserInfo | null;
    isOpenEditConfirm: boolean;
    isWorking: boolean;
    history: IHistoryItem[] | null;
    emptyFields: string[];
    dataLoadingError: Error | null;
    historyLoadingError: Error | null;
    error: Error | null;
}

export default class UserEditor extends React.Component<IUserEditorProps, IUserEditorState> {
    state: IUserEditorState = {
        userInfo: null,
        userInfoInit: null,
        isOpenEditConfirm: false,
        isWorking: false,
        history: null,
        emptyFields: [],
        dataLoadingError: null,
        historyLoadingError: null,
        error: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData({ getUser: true, getUserEditHistory: true });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData(props: { getUser: boolean; getUserEditHistory: boolean }) {
        const { userId = null } = this.props;

        if (props.getUser && userId) {
            this.request.exec(REQUESTS.GET_FULL_USER_INFO, { queryParams: { user_id: userId } })
                .then(response => {
                    const userInfo = this.formatData(response);

                    this.setState({
                        userInfo,
                        userInfoInit: deepCopy(userInfo),
                    });
                })
                .catch(dataLoadingError => {
                    this.setState({ dataLoadingError });
                });
        }

        if (props.getUserEditHistory) {
            this.request.exec(REQUESTS.GET_USER_HISTORY_INFO, { queryParams: { user_id: userId } })
                .then(response => {

                    const passportHistory = response?.documents?.passport;
                    const drivingLicenseHistory = response?.documents?.driving_license;

                    let history: IHistoryItem[] = response.history;
                    history = history?.map(historyItem => {
                        historyItem.object = this.formatHistoryData(
                            historyItem.object || {},
                            passportHistory,
                            drivingLicenseHistory,
                        );

                        return historyItem;
                    });

                    history.reduce((prev: { object: Dict<any> }, curr) => {
                        const prevObject = prev.object;
                        const currObject = curr.object;

                        Object.keys(currObject).forEach(objKey => {

                            const prevKeys = Object.keys(prevObject[objKey]);
                            const currKeys = Object.keys(currObject[objKey]);

                            const allKeys = Array.from(new Set([...prevKeys, ...currKeys]));

                            allKeys.forEach(key => {
                                let currValue = currObject[objKey][key];
                                let prevValue = prevObject[objKey][key];

                                !currValue && (currValue = null);
                                !prevValue && (prevValue = null);

                                if (currValue !== prevValue) {
                                    const diff = { old: prevValue, new: currValue };
                                    if (curr.newData) {
                                        if (curr.newData[objKey]) {
                                            curr.newData[objKey][key] = diff;
                                        } else {
                                            curr.newData[objKey] = { [key]: diff };
                                        }
                                    } else {
                                        curr.newData = { [objKey]: { [key]: diff } };
                                    }
                                }
                            });
                        });

                        return curr;
                    }, {
                        object:
                            {
                                commonUserInfo: {},
                                userPassportInfo: {},
                                userDrivingLicenseInfo: {},
                            },
                    });
                    history.reverse();

                    this.setState({ history });

                })
                .catch(historyLoadingError => {
                    this.setState({ historyLoadingError });
                });
        }
    }

    formatData(userObjData: Dict<any>): IUserInfo {
        const user: Dict<any> = Object.assign({}, USER_MODEL, userObjData);

        const commonUserInfo: Dict<string | boolean | number> = {
            id: user?.id,
            first_name: user?.first_name,
            last_name: user?.last_name,
            pn: user?.pn,
            status: user?.status,
            username: user?.username,
            uid: user?.uid,
            phone: user?.setup?.phone?.number,
            phoneVerified: user?.setup?.phone?.verified,
            email: user?.setup?.email?.address,
            is_mechanic_transmission_allowed: user?.is_mechanic_transmission_allowed,
            is_first_riding: user?.is_first_riding,
        };

        let userPassport: Dict<any> = {};
        if (Array.isArray(user?.documents?.passport)) {
            userPassport = user?.documents?.passport
                ?.filter(passportItem => passportItem.revision === user?.passport_revision)?.[0];
        } else {
            userPassport = user?.documents?.passport?.[user?.passport_revision || ''];
        }

        const userPassportInfo = userPassport
            ? {
                doc_value: userPassport.doc_value,
                first_name: userPassport.first_name,
                last_name: userPassport.last_name,
                middle_name: userPassport.middle_name,
                birth_place: userPassport.birth_place,
                citizenship: userPassport.citizenship,
                gender: userPassport.gender,
                subdivision_code: userPassport.subdivision_code,
                birth_date: userPassport.birth_date || null,
                issue_date: userPassport.issue_date || null,
                issued_by: userPassport.issued_by || null,
                registration_apartment: userPassport.registration_apartment,
                registration_housing: userPassport.registration_housing,
                registration_house: userPassport.registration_house,
                registration_letter: userPassport.registration_letter,
                registration_street: userPassport.registration_street,
                registration_locality: userPassport.registration_locality,
                registration_area: userPassport.registration_area,
                registration_region: userPassport.registration_region,
                expiration_date: userPassport.expiration_date || null,
                biographical_country: userPassport.biographical_country ?? null,
                registration_country: userPassport.registration_country ?? null,
                registration_expiration_date: userPassport.registration_expiration_date || null,
            }
            : USER_MODEL.documents.passport[''];

        let userDrivingLicense: Dict<any> = {};
        if (Array.isArray(user?.documents?.driving_license)) {
            userDrivingLicense = user?.documents?.driving_license
                ?.filter(drivingLicenseItem => drivingLicenseItem.revision === user?.driving_license_revision)?.[0];
        } else {
            userDrivingLicense = user?.documents?.driving_license?.[user?.driving_license_revision || ''];
        }

        const userDrivingLicenseInfo = userDrivingLicense
            ? {
                first_name: userDrivingLicense.first_name,
                last_name: userDrivingLicense.last_name,
                middle_name: userDrivingLicense.middle_name,
                number_front: userDrivingLicense.number_front,
                number_back: userDrivingLicense.number_back,
                prev_licence_number: userDrivingLicense.prev_licence_number,
                categories: userDrivingLicense.categories,
                birth_date: userDrivingLicense.birth_date || null,
                issue_date: userDrivingLicense.issue_date || null,
                issued_by: userDrivingLicense.issued_by || null,
                experience_from: userDrivingLicense.experience_from || null,
                prev_licence_issue_date: userDrivingLicense.prev_licence_issue_date || null,
                categories_b_valid_from_date: userDrivingLicense.categories_b_valid_from_date || null,
                categories_b_valid_to_date: userDrivingLicense.categories_b_valid_to_date || null,
                front_country: userDrivingLicense.front_country,
                back_country: userDrivingLicense.back_country,

            }
            : USER_MODEL.documents.driving_license[''];

        return {
            commonUserInfo, userPassportInfo, userDrivingLicenseInfo,
        };
    }

    formatHistoryData(userObjData: Dict<any>, passportHistory: Dict<any>, drivingLicenseHistory: Dict<any>) {
        const user = Object.assign({}, USER_MODEL, userObjData);
        const commonUserInfo: Dict<string | boolean | number | null> = {
            id: user?.id,
            first_name: user?.first_name,
            last_name: user?.last_name,
            pn: user?.pn,
            status: user?.status,
            username: user?.username,
            uid: user?.uid,
            phone: user?.setup?.phone?.number,
            phoneVerified: user?.setup?.phone?.verified,
            email: user?.setup?.email?.address,
            is_mechanic_transmission_allowed: user?.is_mechanic_transmission_allowed,
            is_first_riding: user?.is_first_riding,
        };

        const userPassportInfo: Dict<string | boolean | number> =
            passportHistory
                ?.filter(passportHistoryItem => passportHistoryItem.revision === user?.passport_revision)?.[0]
            ?? USER_MODEL.documents.passport[''];

        const userDrivingLicenseInfo: Dict<string | boolean | number> =
            drivingLicenseHistory
                ?.filter(drivingLicenseHistoryItem => {
                    return drivingLicenseHistoryItem.revision === user?.driving_license_revision;
                })?.[0]
            ?? USER_MODEL.documents.driving_license[''];

        return {
            commonUserInfo, userPassportInfo, userDrivingLicenseInfo,
        };
    }

    buildInputs(obj: Dict<any>, objKey: string) {
        const schema = USER_MODEL_DESC[objKey].fields ?? {};
        const currentObject = obj[objKey] ?? {};

        return <div className={style.column} key={objKey}>
            <h2>{USER_MODEL_DESC[objKey].title}</h2>
            <FormConstructor onChange={this.onFormChange.bind(this, objKey)}
                             initialData={currentObject}
                             schema={schema}/>
            {!Object.keys(schema).length
                ? <span className={style.no_data}>Нет данных</span>
                : null}
        </div>;
    }

    onFormChange(objKey: string, data: Dict<any>) {
        const userInfo = this.state.userInfo
            || { commonUserInfo: {}, userPassportInfo: {}, userDrivingLicenseInfo: {} };
        userInfo[objKey] = deepCopy(data);
        this.setState({ userInfo });
    }

    openEditConfirm() {
        this.setState({ isOpenEditConfirm: true });
    }

    closeEditConfirm() {
        this.setState({ error: null, isOpenEditConfirm: false });
    }

    buildInfoForSend() {
        const userInfo: Dict<any> = deepCopy(this.state.userInfo || {});

        userInfo.commonUserInfo.is_phone_verified = userInfo.commonUserInfo.phoneVerified;

        const userDataForSend = userInfo.commonUserInfo;
        if (Object.keys(userInfo.userPassportInfo).length) {
            userDataForSend.passport = userInfo.userPassportInfo;
        }

        if (Object.keys(userInfo.userDrivingLicenseInfo).length) {
            userDataForSend.driving_license = userInfo.userDrivingLicenseInfo;
        }

        return userDataForSend;
    }

    updateAllUserInfo() {
        const userDataForSend = this.buildInfoForSend();

        const emptyFields = Object.keys(userDataForSend)
            ?.filter(key => key !== PASSPORT_KEY && key !== DRIVING_LICENSE_KEY && userDataForSend[key] === '');

        this.setState({ emptyFields }, () => {
            !this.state.emptyFields.length && this.updateUserInfo(userDataForSend);
        });
    }

    updateWithEmptyFields() {
        const userDataForSend = this.buildInfoForSend();
        this.updateUserInfo(userDataForSend);
    }

    closeEmptyFieldsConfirm() {
        this.closeEditConfirm();
        this.setState({ emptyFields: [], error: null });
    }

    updateUserInfo(userDataForSend: Dict<any>) {
        this.setState({ isWorking: true, error: null }, () => {
            this.request.exec(REQUESTS.EDIT_USER_INFO, { body: userDataForSend })
                .then(() => {
                    this.setState({ isWorking: false });
                    this.getData({ getUser: false, getUserEditHistory: true });
                    this.closeEditConfirm();
                    this.closeEmptyFieldsConfirm();
                })
                .catch(error => {
                    this.setState({ error, isWorking: false });
                });
        });
    }

    render() {
        const {
            dataLoadingError, historyLoadingError, userInfoInit,
            history, isOpenEditConfirm, error, isWorking, emptyFields,
        } = this.state;

        return <Window onClose={this.props.onClose.bind(this)} title={'Редактирование пользователя'}>

            <div className={style.user_editor}>
                <div className={style.editor}>
                    {dataLoadingError
                        ? <SimpleError error={dataLoadingError}
                                       data={{ label: 'Ошибка при загрузке данных пользователя' }}/>
                        : userInfoInit
                            ? <>
                                <div className={style.column_container}>
                                    {userInfoInit && Object.keys(userInfoInit).map(key => {
                                        return this.buildInputs(userInfoInit || {}, key);
                                    })}
                                </div>
                                <div className={coreStyle.button_container}>
                                    <CancelButton onClick={this.props.onClose}/>
                                    <Button onClick={this.openEditConfirm.bind(this)}>Изменить</Button>
                                </div>
                            </>
                            : <Spin/>}
                </div>
                {historyLoadingError
                    ? <SimpleError error={historyLoadingError}
                                   data={{ label: 'Ошибка при загрузке истории изменения пользователя' }}/>
                    : history
                        ? <HistoryView history={history}/>
                        : <Spin/>}
            </div>

            {isOpenEditConfirm
                ? <Confirm question={'Изменить данные о пользователе?'}
                           error={error}
                           isWorking={isWorking}
                           accept={this.updateAllUserInfo.bind(this)}
                           onClose={this.closeEditConfirm.bind(this)}/>
                : null}
            {emptyFields?.length
                ? <Confirm question={`Остались пустыми поля: ${emptyFields
                    .map(key => USER_MODEL_DESC?.commonUserInfo?.fields?.[key]?.display_name ?? key)
                    .join(', ')}. Всё равно изменить?`}
                           error={error}
                           isWorking={isWorking}
                           accept={this.updateWithEmptyFields.bind(this)}
                           onClose={this.closeEmptyFieldsConfirm.bind(this)}/>
                : null}
        </Window>;
    }
}
