import * as React from 'react';

import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import Select from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../request';
import * as style from './index.css';

interface IPushProps {
    user_id: string;
}

interface IPushState {
    tags: any[];
    push: string[];
    isLoading: boolean;
    error: Error | null;
    confirmIsOpen: boolean;
}

const USER_PUSH_TAG_TYPE = 'user_push';

export default class AddPush extends React.PureComponent<IPushProps, IPushState> {
    state = {
        tags: [],
        push: [],
        isLoading: false,
        error: null,
        confirmIsOpen: false,
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getTags();
    }

    componentDidUpdate(prevProps: Readonly<IPushProps>): void {
        if (this.props.user_id !== prevProps.user_id) {
            this.getTags();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getTags() {
        this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS)
            .then(response => {
                const tags = response?.records?.filter?.((_el: any) => _el.type === USER_PUSH_TAG_TYPE);
                this.setState({
                    tags: tags.map(tag => {
                        return { text: tag.display_name || tag.name, value: tag.name };
                    }),
                });
            });
    }

    onSelect(value: any) {
        this.setState({ push: [value] });
    }

    openConfirm() {
        this.setState({ confirmIsOpen: true });
    }

    closeConfirm() {
        this.setState({ confirmIsOpen: false });
    }

    pushHandler() {
        const data = {
            'priority': 0,
            'tag': this.state.push && this.state.push.toString(),
            'comment': 'by admin interface',
            'object_id': this.props.user_id,
        };

        this.setState({ isLoading: true }, () => {
            this.request.exec(REQUESTS.ATTACH_USER_TAG, { body: data })
                .then(() => {
                    this.setState({ isLoading: false, confirmIsOpen: false });
                })
                .catch(() => {
                    this.setState({ isLoading: false });
                });
        });
    }

    render() {
        const selectedPush: any = this.state.tags.filter((_el: any) => _el.value == this.state.push)[0];
        const confirmQuestion = <div>Отправить пуш:{selectedPush && selectedPush.text || EMPTY_DATA}?</div>;

        return <div className={style.push_control}>
            {this.state.error
                ? <SimpleError data={{ label: 'Пуши' }} error={this.state.error}/>
                : <>
                    <Select onSelect={this.onSelect.bind(this)}
                            containerClassName={style.push}
                            options={this.state.tags}
                            placeholder={'Пуш'}
                            initialValues={this.state.push}/>

                    <Button disabled={!this.state.push.length}
                            onClick={this.openConfirm.bind(this)}>Отправить пуш</Button>

                    {this.state.confirmIsOpen
                        ? <Confirm isWorking={this.state.isLoading}
                                   accept={this.pushHandler.bind(this)}
                                   question={confirmQuestion}
                                   error={null}
                                   onClose={this.closeConfirm.bind(this)}/>
                        : null
                    }
                </>
            }
        </div>;
    }
}
