import React from 'react';

import { ONE_SECOND } from '../../../../../constants';
import { UserInfoHandler } from '../../../../../models/user';
import DatePicker from '../../../../../ui/DatePicker';
import { isObjectEqual } from '../../../../../utils/isObjectEqual';
import { Request2 } from '../../../../../utils/request';
import { deepCopy } from '../../../../../utils/utils';
import { UserInfoContext } from '../../../context';
import {
    DEFAULT_TAG,
    DISPLAY_TYPES,
    DisplayTypesTranslate,
    generateQuery,
    REQUEST_TAGS,
    REQUEST_TYPES,
} from '../../constants';
import { requestConfigs, REQUESTS } from '../../requests';
import { SuccessCallCheckbox } from '../Checkbox/component';
import { RequestTypePanel } from '../RequestTypesPanel/RequestTypePanel';
import { IMapToState } from './index';
import style from './index.css';
import { samsaraAdapter } from './samsaraAdapter';

interface IHeaderToolbarProps extends IMapToState {
    userId: string;

    setData: (data: any) => void;
    setLoading: (isLoading: boolean) => void;
    setSamsaraLoading: (setSamsaraLoading: boolean) => void;
    setError: (error: Error | null) => void;
}

interface IHeaderToolbarState {
    activeDisplayType: DISPLAY_TYPES;
    activeRequestTypes: { [type in REQUEST_TYPES]: boolean };
    dateFrom: number | null;
    dateTo: number | null;
    isActiveCheckbox: boolean;
    data: any;
    samsaraData: any;
}

export class HeaderToolbar extends React.Component<IHeaderToolbarProps, IHeaderToolbarState> {
    state: IHeaderToolbarState = {
        activeDisplayType: DISPLAY_TYPES.events,
        activeRequestTypes: {
            [REQUEST_TYPES.incoming]: true,
            [REQUEST_TYPES.outgoing]: true,
            [REQUEST_TYPES.chat]: false,
            [REQUEST_TYPES.email]: false,
            [REQUEST_TYPES.sms]: false,
            [REQUEST_TYPES.dispatcher]: false,
            [REQUEST_TYPES.push]: false,
            [REQUEST_TYPES.introScreen]: false,
            [REQUEST_TYPES.smm]: false,
        },
        dateFrom: null,
        dateTo: null,
        isActiveCheckbox: false,

        data: {},
        samsaraData: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IHeaderToolbarProps>, prevState: Readonly<IHeaderToolbarState>) {
        const wereReqTypesChanged = !isObjectEqual(this.state.activeRequestTypes, prevState.activeRequestTypes);
        const wasDateToChanged = this.state.dateTo !== prevState.dateTo;
        const wasDateFromChanged = this.state.dateFrom !== prevState.dateFrom;
        const wasCheckboxChanged = this.state.isActiveCheckbox !== prevState.isActiveCheckbox;
        const wasUserChanged = this.props.userId !== prevProps.userId;
        const wasDisplayTypeChanged = this.state.activeDisplayType !== prevState.activeDisplayType;
        const wereFiltersChanged = wereReqTypesChanged || wasDateToChanged || wasDateFromChanged;

        if (wereFiltersChanged || wasUserChanged) {
            this.getData();
        }

        if (wasCheckboxChanged || wasDisplayTypeChanged) {
            this.prepareData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getData() {
        this.props.setLoading(true);

        const userQueryParams = this.getQueryParams('user_id');
        const performerQueryParams = this.getQueryParams('performer_id');

        Promise.all([
            this.request.exec(REQUESTS.GET_REQUESTS, { queryParams: userQueryParams }),
            this.request.exec(REQUESTS.GET_REQUESTS, { queryParams: performerQueryParams }),
        ])
            .then(response => {
                if (response?.length) {
                    const data: any = { tags: [], users: [] };

                    response.forEach(resp => {
                        resp.tags ? data.tags = [...data.tags, ...resp.tags] : null;
                        resp.users ? data.users = [...data.users, ...resp.users] : null;
                    });

                    this.setState({ data }, () => this.checkSamsara());
                    this.prepareData(data);
                } else {
                    this.setState({ data: response });
                    this.prepareData(response);
                }

                this.props.setLoading(false);
                this.props.setError(null);
            })
            .catch(error => {
                this.props.setLoading(false);
                this.props.setError(error);
            });

    }

    getQueryParams(userType: 'user_id' | 'performer_id') {
        let tagsString = '';
        const tagsEntries = Object.entries(this.state.activeRequestTypes);
        tagsEntries.forEach(el => el[1] ? tagsString += `${REQUEST_TAGS[el[0]]},` : null);

        tagsString += DEFAULT_TAG;

        const queryParams: any = {
            [userType]: this.props.userId,
            tags: tagsString,
            rev: true,
        };

        this.state.dateFrom ? queryParams.since = this.state.dateFrom / ONE_SECOND : null;
        this.state.dateTo ? queryParams.until = this.state.dateTo / ONE_SECOND : null;

        return queryParams;
    }

    checkSamsara() {
        const { activeRequestTypes, samsaraData } = this.state;

        const email = UserInfoHandler.getEmail.call(this.context);
        const samsaraRule = this.props.BlockRules.GetSamsaraByDefault;
        const needSamsara = email && samsaraRule && activeRequestTypes[REQUEST_TYPES.email] && !samsaraData;

        if (needSamsara) {
            this.props.setSamsaraLoading(true);

            this.request.exec(REQUESTS.GET_EMAILS, {
                body: {
                    ...generateQuery(email),
                },
            })
                .then(response => {
                    const samsaraData = samsaraAdapter(response.tickets) ?? [];

                    const newData = {
                        ...this.state.data,
                        tags: [...this.state.data.tags, ...samsaraData],
                    };

                    this.setState({ data: newData, samsaraData }, () => {
                        this.props.setSamsaraLoading(false);
                        this.prepareData(newData);
                    });
                })
                .catch(error => {
                    this.props.setSamsaraLoading(false);
                    this.props.setError(error);
                });
        }
    }

    prepareData(newData?: any) {
        const data = newData ? deepCopy(newData) : deepCopy(this.state.data);
        const allTags = data.tags;
        let finalData: any;

        if (this.state.isActiveCheckbox) {
            const filteredTags = allTags?.filter(tag => {
                return tag.call?.enter || !tag.call;
            }) ?? [];

            if (this.state.activeDisplayType === DISPLAY_TYPES.timeline) {
                finalData = { tags: filteredTags, users: data.users };
            } else {
                finalData = { tags: this.buildTree(filteredTags), users: data.users };
            }
        } else {
            if (this.state.activeDisplayType === DISPLAY_TYPES.timeline) {
                finalData = { tags: allTags, users: data.users };
            } else {
                finalData = { tags: this.buildTree(allTags), users: data.users };
            }
        }

        this.props.setData(finalData);
    }

    buildTree(tags) {
        const parents: any = {};
        const children = new Set();

        tags?.slice()?.forEach(el => {
            if (el.task_tag) {
                children.add(el);
            } else {
                parents[el.tag_id] = el;
            }
        });

        Array.from(children)?.forEach((el: any) => {
            const parentId = el.task_tag.tag_id;
            const parent = parents[parentId];
            parent && !parent.children ? parent.children = [] : null;
            parent?.children?.push?.(el);

        });

        return Object.values(parents);
    }

    changeDisplayType(activeDisplayType: DISPLAY_TYPES) {
        this.setState({ activeDisplayType });
    }

    changeRequestType(newRequestType: REQUEST_TYPES) {
        const currentValue = this.state.activeRequestTypes[newRequestType];
        const activeRequestTypes = {
            ...this.state.activeRequestTypes,
            [newRequestType]: !currentValue,
        } as { [type in REQUEST_TYPES]: boolean };

        this.setState({ activeRequestTypes });
    }

    changeDateFrom(dateFrom) {
        this.setState({ dateFrom });
    }

    changeDateTo(dateTo) {
        this.setState({ dateTo });
    }

    changeActiveCheckbox() {
        const isActiveCheckbox = !this.state.isActiveCheckbox;
        this.setState({ isActiveCheckbox });
    }

    render() {
        let {} = this.props;
        const { activeDisplayType, dateFrom, dateTo, isActiveCheckbox } = this.state;

        const displayTypeEntries = Object.entries(DisplayTypesTranslate) ?? [];

        return <div>
            <div className={style.display_type_wrapper}>
                {displayTypeEntries.map(item => {
                    const type = item[0];
                    const label = item[1];
                    const isActive = type === activeDisplayType;

                    return <div onClick={this.changeDisplayType.bind(this, type)}
                                key={type}
                                className={`${style.item} ${isActive ? style.active : ''}`}>
                        {label}
                    </div>;
                })}
            </div>

            <div className={style.separator_hor}/>

            <div className={style.controls_line}>
                <div className={style.left_side}>
                    <RequestTypePanel requestTypes={this.state.activeRequestTypes}
                                      onClick={this.changeRequestType.bind(this)}/>

                    <div className={style.separator_vert}/>
                </div>

                <div className={style.right_side}>
                    <DatePicker value={dateFrom}
                                onChange={this.changeDateFrom.bind(this)}
                                placeholder={'От'}
                                className={style.date}/>
                    <DatePicker value={dateTo}
                                onChange={this.changeDateTo.bind(this)}
                                placeholder={'До'}
                                className={style.date}/>

                    <SuccessCallCheckbox isActive={isActiveCheckbox}
                                         changeActive={this.changeActiveCheckbox.bind(this)}/>
                </div>
            </div>
        </div>;
    }
}

HeaderToolbar.contextType = UserInfoContext;
