import React, { ReactElement } from 'react';

import PlayIcon from '../../../../../../svg-components/play.component.svg';
import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import { UserInfoHandler } from '../../../../../models/user';
import { FormatDateInString } from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { CallDuration } from '../../../UserRequestHelpers/CallDuration';
import { CallLink } from '../../../UserRequestHelpers/CallLink';
import { REQUEST_TYPES, REQUEST_TYPES_TAGS } from '../../constants';
import style from './index.css';

export function getRequestType(tagsDescription, tag): REQUEST_TYPES {
    const tagType = tagsDescription[tag.tag_name]?.type;

    return REQUEST_TYPES_TAGS[tagType] as REQUEST_TYPES;
}

export function getReqDay(tag): string {
    return tag.timestamp
        ? FormatDateInString({ value: tag.timestamp * ONE_SECOND, onlyDate: true })
        : '';
}

export function getReqTime(tag): string {
    if (tag.call) {
        const callConnect = (tag.call.enter || tag.call.connect) * ONE_SECOND;
        const callConnectStr = FormatDateInString({ value: callConnect, onlyTime: true });

        const callExit = tag.call.exit * ONE_SECOND ?? '';
        const callExitStr = FormatDateInString({ value: callExit, onlyTime: true });

        return `${callConnectStr} - ${callExitStr}`;
    }

    return tag.timestamp
        ? FormatDateInString({ value: tag.timestamp * ONE_SECOND, onlyTime: true })
        : '';

}

export const ReqLink = React.memo((props: { tag: any; type: REQUEST_TYPES }) => {
    const { tag = {}, type } = props;

    switch (type) {
    case REQUEST_TYPES.chat:
    case REQUEST_TYPES.dispatcher:
        return getChatLink(type, tag);

    case REQUEST_TYPES.outgoing:
    case REQUEST_TYPES.incoming:
        const label = <div className={style.request_link}>
            <PlayIcon className={style.play}/>
            <CallDuration tag={tag}/>
        </div>;

        return tag.call?.exit
            ? <CallLink tag={tag} linkLabel={label}/>
            : null;

    case REQUEST_TYPES.email:
        return tag.sams_link ?? null;

    }

    return EMPTY_DATA;
});

export const getChatLink = (type, tag) => {
    const { tag_id, object_id, tag_details } = tag;
    const topic_link = tag_details?.topic_link;
    const isChat = type === REQUEST_TYPES.chat;
    const href = `#/${isChat ? 'chats' : 'outgoing'}?user_id=${object_id}&chat_id=${topic_link}&tag_id=${tag_id}`;
    const title = isChat ? 'Чат' : 'Исх. задача';

    return <Link href={href} target={'_blank'} className={style.request_link}>
        {title}
    </Link>;
};

export const ReqOperatorLink = React.memo((props: { users: any[]; tag: any; userId?: string }): ReactElement | null => {
    const performer = props.tag.tag_details?.performer;
    const operatorId = performer === props.userId
        ? props.tag.object_id ?? performer
        : performer ?? props.tag.user_id;
    const operatorInfo = props.users.find(el => el.id === operatorId) ?? {};
    const operatorName = UserInfoHandler.getPrintNameWithId.call(operatorInfo);
    const operatorLink = `#/clients/${operatorId}/info`;

    return operatorId
        ? <Link href={operatorLink} target={'_blank'} className={style.format_link}>
            {operatorName || operatorId}
        </Link>
        : null;
});

export const ReqEntity = React.memo((props: { tagDescription: any; tag: any }) => {
    const name = props.tagDescription[props.tag.tag_name]?.display_name ?? props.tag.tag_name;

    return <div>
        {name ? <div className={style.simple_text}>{name}</div> : null}
    </div>;
});

export function sortTags(isDecreaseOrder: boolean, tags: any[]) {
    if (isDecreaseOrder) {
        return tags.sort((el1, el2) => el2.timestamp - el1.timestamp);
    }

    return tags.sort((el1, el2) => el1.timestamp - el2.timestamp);

}
