import React from 'react';

import { Dict } from '../../../../../../types';
import { isObjectEqual } from '../../../../../utils/isObjectEqual';
import { Request2 } from '../../../../../utils/request';
import { requestConfigs, REQUESTS } from '../../requests';
import { OrderControl } from '../OrderControl/component';
import style from './index.css';
import { sortTags } from './RequestsTableHelpers';
import { RequestTableTR } from './RequestTableTr';

interface IRequestsTableProps {
    data: any;
    userId: string;
}

interface IRequestsTableState {
    tagsDescription: Dict<any>;
    isDecreaseOrder: boolean;
    openedTags: string[];
    tagsForRender: any[];
}

export class RequestsTable extends React.Component<IRequestsTableProps, IRequestsTableState> {
    state: IRequestsTableState = {
        tagsDescription: {},
        openedTags: [],
        isDecreaseOrder: true,
        tagsForRender: [],
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();

        if (this.props.data.tags) {
            this.changeTagsOrder();
        }
    }

    componentDidUpdate(prevProps: Readonly<IRequestsTableProps>, prevState: Readonly<IRequestsTableState>) {
        const wasOrderChanged = this.state.isDecreaseOrder !== prevState.isDecreaseOrder;
        const wasDataChanged = this.props.data.tags && !isObjectEqual(prevProps.data.tags, this.props.data.tags);

        if (wasDataChanged || wasOrderChanged) {
            this.changeTagsOrder();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getData() {
        this.request.exec(REQUESTS.GET_TAGS)
            .then(response => {
                const tagsDescription = {};
                response?.records?.forEach(el => tagsDescription[el.name] = el);
                this.setState({ tagsDescription });
            });
    }

    showChildren(tag: any) {
        const index = this.state.openedTags.indexOf(tag.tag_id);

        if (index === -1) {
            this.setState({ openedTags: [...this.state.openedTags, tag.tag_id] });
        } else {
            const openedTags = this.state.openedTags.slice();
            openedTags.splice(index, 1);

            this.setState({ openedTags });
        }
    }

    changeOrder() {
        this.setState({ isDecreaseOrder: !this.state.isDecreaseOrder });
    }

    changeTagsOrder() {
        const tagsForRender = sortTags(this.state.isDecreaseOrder, this.props.data.tags);
        this.setState({ tagsForRender });
    }

    checkIfTimeline(tags) {
        return tags.every(el => !el.children);
    }

    render() {
        const { data, userId } = this.props;
        const { tagsDescription, isDecreaseOrder, tagsForRender } = this.state;
        const isTimeline = this.checkIfTimeline(tagsForRender);

        return <div>
            <table className={style.table}>
                <thead className={style.table_header}>
                    <tr>
                        <td>#</td>
                        <td className={style.icons_td}>Тип</td>
                        <td className={style.date_td} onClick={this.changeOrder.bind(this)}>
                            <div className={style.date}>
                                <div>Дата и время</div>
                                <OrderControl isDecreaseOrder={isDecreaseOrder}/>
                            </div>
                        </td>
                        <td className={style.request_td}>Обращение</td>
                        <td className={style.entity_td}>Очередь</td>
                        <td>Классификация</td>
                        <td className={style.operator_td}>Оператор</td>
                        <td/>
                    </tr>
                </thead>

                <tbody>
                    {tagsForRender.map((tag, index) => {
                        const isOpened = this.state.openedTags.includes(tag.tag_id);

                        return <RequestTableTR tag={tag}
                                               userId={userId}
                                               index={index + 1}
                                               tagsDescription={tagsDescription}
                                               key={tag.tag_id}
                                               isOpened={isOpened}
                                               users={data.users}
                                               isTimeline={isTimeline}
                                               showChildren={this.showChildren.bind(this)}/>;
                    })
                    }
                </tbody>
            </table>
        </div>;
    }
}
