import React, { useState } from 'react';

import { ClassificationType, Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { CategorizationItem, TagRecordHandler } from '../../../models/tag';
import { UserInfoHandler } from '../../../models/user';
import { Link } from '../../../ui/Link';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as tblStyle from '../../../ui/Table/index.css';
import { isValidGUIDString } from '../../../utils/utils';
import { AttachUserRequestModal, USER_REQUEST_TAG } from '../../Header/CallsNotify/AttachUserRequestModal';
import { CallDuration } from '../UserRequestHelpers/CallDuration';
import { CallLink } from '../UserRequestHelpers/CallLink';
import { Dates } from '../UserRequestHelpers/Dates';
import { IUserRequestsData } from './component';
import { FILTER_TYPES } from './FilterRequests/component';
import * as style from './index.css';

interface ITableProps {
    data: IUserRequestsData | null;
    id: string;
    showClassification: () => void;
    selectedFilters: FILTER_TYPES[] | null;
    tags: Dict<string>;
    showChatHistory: () => void;
    samsaraTickets?: any[];
}

const match_sams = 'sams';
const SAMS_LABEL = 'SAMS';

export const UserRequestsTable = React.memo((props: ITableProps) => {
    const match_call = 'cc_';
    const PHONE_LABEL = 'PHONE';

    const match_push = 'push';
    const PUSH_LABEL = 'PUSH';

    const match_chat = 'chat';
    const CHAT_LABEL = 'CHAT';

    const match_email = 'mail';
    const EMAIL_LABEL = 'EMAIL';

    const match_introscreen = 'landing';
    const INTROSCREEN_LABEL = 'SCREEN';

    const match_sms = 'sms_';
    const SMS_LABEL = 'SMS';

    const match_smm = 'user_support_smm';
    const SMM_LABEL = 'SMM';

    const [tagData, setTagData]: any = useState(null);

    return <>
        <table className={tblStyle.table}>
            <thead>
                <tr>
                    <th>#</th>
                    <th>дата</th>
                    <th/>
                    <th>канал/тема</th>
                    <th/>
                    <th>tag_name</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {
                    mixWithSamsara(props.data?.tags, props.samsaraTickets, props.selectedFilters)
                        .sort((b, a) => a.timestamp - b.timestamp)
                        .map((tag, index) => {
                            const topic_link = tag?.tag_details?.topic_link;
                            const { object_id, tag_id, tag_name = '' } = tag;
                            const categorization: CategorizationItem[] = TagRecordHandler.getCategorization.call(tag);
                            const isAttachedRequest = tag?.tag_name === USER_REQUEST_TAG;
                            const attachInfo = tag?.tag_details?.fields || [];

                            const attachData: any = {};
                            attachInfo?.forEach(item => attachData[item.key] = item.value);
                            const { call_id, user_id } = attachData;

                            const outgoingHref = `#/outgoing?user_id=${object_id}&chat_id=${topic_link}&tag_id=${tag_id}`;
                            const chatHref = `#/chats?user_id=${object_id}&chat_id=${topic_link}&tag_id=${tag_id}`;

                            return <tr key={`${tag.tag_id}_${index}`}>
                                <td>{index + 1}</td>
                                <td>
                                    <Dates tag={tag}/>
                                </td>
                                <td>
                                    {
                                        (tag.call?.enter || tag.call?.connect) &&
                                    <>
                                        <CallDuration tag={tag}/>
                                        <div>
                                            <CallLink tag={tag} />
                                        </div>
                                    </>
                                    }
                                    {isAttachedRequest && call_id
                                        ? <Link href={getCallHref(call_id)} target={'_blank'}>
                                        Прослушать
                                        </Link>
                                        : null
                                    }
                                    {topic_link
                                        ? <div>
                                            <Link onClick={props.showChatHistory.bind(null, true, tag)}>
                                            история чата
                                            </Link>
                                        </div>
                                        : null
                                    }
                                    {tag_id && topic_link && object_id
                                        ? <div>
                                            {topic_link.includes('outgoing_')
                                                ? <Link href={outgoingHref} target={'_blank'}>
                                                    Исходящая коммуникация
                                                </Link>
                                                : <Link href={chatHref} target={'_blank'}>Чат</Link>
                                            }
                                        </div>
                                        : null
                                    }
                                    <div className={style.sams_tags}>
                                        {
                                            tag?.sams_tags?.map((el, index) => {
                                                return <div key={index}
                                                            style={{ backgroundColor: el?.value?.color }}
                                                            className={style.sams_tag_item}>{el?.value?.name}</div>;
                                            })
                                        }
                                    </div>
                                </td>
                                <td className={style.cat_labels}>
                                    <div>
                                        {tag_name.includes(match_call)
                                            ? <TLabel status={LabelStatus.POSITIVE} text={PHONE_LABEL}/>
                                            : tag_name.includes(match_push)
                                                ? <TLabel status={LabelStatus.NEGATIVE_DARK} text={PUSH_LABEL}/>
                                                : tag_name.includes(match_introscreen)
                                                    ? <TLabel status={LabelStatus.WARNING_DARK}
                                                              text={INTROSCREEN_LABEL}/>
                                                    : tag_name.includes(match_sms)
                                                        ? <TLabel status={LabelStatus.WARNING_DARK} text={SMS_LABEL}/>
                                                        : tag_name.includes(match_chat)
                                                            ? <TLabel status={LabelStatus.INFO} text={CHAT_LABEL}/>
                                                            : tag_name.includes(match_email)
                                                                ? <TLabel status={LabelStatus.NEGATIVE_DARK}
                                                                          text={EMAIL_LABEL}/>
                                                                : tag_name.includes(match_smm)
                                                                    ? <TLabel status={LabelStatus.NEGATIVE_DARK}
                                                                              text={SMM_LABEL}/>
                                                                    : tag_name === match_sams
                                                                        ? <TLabel status={LabelStatus.POSITIVE}
                                                                                  text={SAMS_LABEL}/>
                                                                        : isAttachedRequest
                                                                            ? <TLabel status={LabelStatus.POSITIVE}
                                                                                      text={'ATTACH'}/>
                                                                            : null
                                        }
                                    </div>
                                    <div onClick={props.showClassification.bind(null, tag.tag_id,
                                        tag?.tag_name?.includes(match_chat)
                                    && ClassificationType.chat || ClassificationType.call_incoming, tag)}>
                                        {
                                            (!tag?.tag_name?.includes(match_push)
                                            && !tag?.tag_name?.includes(match_sams)
                                            && !tag?.tag_name?.includes(match_introscreen)
                                            && !tag?.tag_name?.includes(match_sms)) &&
                                        (!categorization.length
                                            ? <TLabel status={LabelStatus.DEFAULT} text={`разметить`}/>
                                            : null
                                        )
                                        }
                                    </div>
                                    {
                                        tag?.tag_name === match_sams && <div>артикл ({tag?.articles?.length})</div>
                                    }
                                    {tag?.tag_name?.includes(match_call)
                                        ? <div onClick={setTagData.bind(null, {
                                            tag_id: tag?.tag_id || tag?.tag_details?.tag_id,
                                            call_id: tag?.tag_details?.call_id ?? '',
                                            user_id: tag?.object_id ?? '',
                                        })}>
                                            <TLabel status={LabelStatus.WARNING_DARK} text={'привязать пользователя'}/>
                                        </div>
                                        : null
                                    }
                                </td>

                                <td className={style.cat_labels}>
                                    <div onClick={props.showClassification.bind(null, tag.tag_id,
                                        tag?.tag_name?.includes(match_chat)
                                    && ClassificationType.chat || ClassificationType.call_incoming, tag)}>
                                        {categorizationsLabels(categorization)}
                                    </div>
                                </td>

                                <td>
                                    <div className={style.comment}>
                                        {tag?.tag_name?.includes(match_introscreen)
                                            ? tag?.tag_name
                                            : tag.sams_status || tag?.tag_details?.comment}
                                    </div>
                                    <div className={style.tag_name}>
                                        {tag.sams_subj || props.tags?.[tag.tag_name] || tag.tag_name}
                                    </div>
                                    {tag.call_url &&
                                <div><Link href={tag.call_url} target={'_blank'}>прослушать звонок</Link></div>}
                                    {isAttachedRequest && user_id
                                        ? <Link href={getOriginalReqHref(user_id)} target={'_blank'}>
                                        Перейти к обращениям
                                        </Link>
                                        : null
                                    }
                                </td>
                                <td>
                                    <div>
                                        {(props.id !== tag.object_id)
                                            && tag.object_id
                                            && isValidGUIDString(tag.object_id)
                                            && <div>
                                                    tag object_id : <Link href={`#/clients/${tag.object_id}/info`}>
                                                    {UserInfoHandler
                                                        .getPrintNameWithId
                                                        .call(props.data?._users[tag.object_id])|| tag.object_id}
                                                </Link>
                                            </div>}

                                        {props.id !== tag?.tag_details?.performer && tag
                                    && isValidGUIDString(tag?.tag_details?.performer || '') && <div>
                                            <div>performer:</div>
                                            <div>
                                                <Link href={`#/clients/${tag?.tag_details?.performer}/info`}>
                                                    {props.data?._users && UserInfoHandler.getPrintNameWithId
                                                        .call(props.data?._users[tag?.tag_details?.performer || ''])
                                                || tag?.tag_details?.performer}
                                                </Link>
                                            </div>
                                        </div>}
                                    </div>
                                </td>
                            </tr>;
                        })
                }
            </tbody>
        </table>

        {tagData
            ? <AttachUserRequestModal openBindClientModal={setTagData.bind(null)}
                                      values={tagData}/>
            : null
        }
    </>;
});

const mixWithSamsara = (tags, samsaraTickets, filters) => {
    if (!filters.includes(FILTER_TYPES.MAIL)) {
        return tags || [];
    }

    return (tags || []).concat(
        ...(samsaraTickets?.map(el => {
            return {
                timestamp: Math.floor(el?.params?.dateCreate?.value / ONE_SECOND),
                dateOpen: Math.floor(el?.params?.dateOpen?.value / ONE_SECOND),
                articles: el.articles,
                tag_id: null,
                sams_tags: el?.params?.tags?.value,
                tag_name: match_sams,
                event_id: el.ticketId,
                sams_subj: <span>
                    <span className={style.sams_queue}>
                        {el?.params?.queue?.value?.queueName}
                    </span>: {el?.params?.subject?.value}</span>,
                sams_status: <span>{el?.params?.status?.value}
                    <Link href={`https://samsara.yandex-team.ru/ticket/${el.ticketId}/articles`}
                          target={'_blank'}> Тикет #{el?.params?.ticketNumber?.value}</Link></span>,
            };
        }) || []));
};

const getCallHref = (callId) => {
    return `https://tel.yandex-team.ru/queuemetricscallrec?callid=${callId}`;
};

const getOriginalReqHref = (userId) => {
    return `#/clients/${userId}/request_aggregator`;
};

const categorizationsLabels = (data): any => {
    return data.map(item => {
        const status = item.comment ? LabelStatus.POSITIVE : LabelStatus.WARNING;

        return <TLabel status={status}
                       className={style.label}
                       key={item?.category?.id}
                       text={item?.category?.meta?.label ?? EMPTY_DATA}/>;
    });
};
