import React from 'react';

import { ClassificationType, Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND, TIMERS } from '../../../constants';
import { TagRecord } from '../../../models/tag';
import { BasicUserInfo, UserInfoHandler } from '../../../models/user';
import { Collapse2 } from '../../../ui/Collapse2';
import { IBlockRules } from '../../../utils/IBlockRules';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { ChatHistoryView } from '../../Chats/ChatHistoryView/component';
import ClassificationDialog from '../../Chats/ClassificationDialog';
import { USER_REQUEST_TAG } from '../../Header/CallsNotify/AttachUserRequestModal';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { UserInfoContext } from '../context';
import { FILTER_TYPES, FilterRequests } from './FilterRequests/component';
import * as style from './index.css';
import { generateQuery, requestConfigs, REQUESTS } from './request';
import { UserRequestsTable } from './UserRequestsTable';

interface IUserRequestsProps extends IBlockRules {
    id: string;
    user_type: RequestUserType;
    tags: Dict<string>;
    samsara?: boolean;
    selectedFilters?: FILTER_TYPES[];
    since?: number;
    until?: number;
}

export interface IUserRequestsData {
    tags: typeof TagRecord[];
    users: typeof BasicUserInfo[];
    _users: Dict<typeof BasicUserInfo>;
}

interface IUserRequestsState {
    isLoading: boolean;
    error: Error | null;
    response: IUserRequestsData | null;
    showClassification: boolean;
    tag_id: string;
    request_type: ClassificationType;
    selectedFilters: FILTER_TYPES[];
    since: "" | number;
    until: "" | number;

    [x: number]: any;

    tag: any | null;
    showChatHistory: boolean;
    samsaraIsLoading: boolean;
    samsara: any;
    samsaraError: Error | null;
    getSamsara: boolean;
}

export enum RequestUserType {
    user = 'user_id',
    performer = 'performer_id'
}

export default class UserRequests extends React.Component<IUserRequestsProps, IUserRequestsState> {
    state: IUserRequestsState = {
        isLoading: false,
        error: null,
        response: null,
        showClassification: false,
        request_type: ClassificationType.chat,
        tag_id: '',
        selectedFilters: [],
        since: this.props.since ?? '',
        until: this.props.until ?? '',
        tag: null,
        showChatHistory: false,
        samsaraIsLoading: false,
        samsara: null,
        samsaraError: null,
        getSamsara: false,
    };

    request = new Request2({ requestConfigs });
    private previousContext;

    constructor(props) {
        super(props);

        const defaultSelectedFilters: FILTER_TYPES[] = [
            FILTER_TYPES.PHONE,
            FILTER_TYPES.OUTGOING,
            FILTER_TYPES.WEBPHONE,
            FILTER_TYPES.MESSENGER,
        ];

        this.state.selectedFilters = this.props.selectedFilters?.length
            ? this.props.selectedFilters
            : this.props.BlockRules?.GetSamsaraByDefault
                ? [...defaultSelectedFilters, FILTER_TYPES.MAIL]
                : defaultSelectedFilters;

    }

    getData() {
        this.props.id && this.setState({
            isLoading: true,
            response: null,
        }, () => {
            const tags = this.state.selectedFilters?.map(el => {
                return el === FILTER_TYPES.SMM ? el : `@${el}`;
            }).join(',') + `,${USER_REQUEST_TAG}`;
            this.request.exec(REQUESTS.ARCHIVE, {
                queryParams: {
                    [this.props.user_type]: this.props.id,
                    rev: true,
                    since: Math.floor(+this.state.since / ONE_SECOND),
                    until: Math.floor(+this.state.until / ONE_SECOND),
                    tags,
                },
            })
                .then(response => {
                    this.setState({
                        error: null,
                        isLoading: false,
                        response: {
                            ...response,
                            _users: response?.users?.reduce((_p, _c) => {
                                return { ..._p, [_c.id]: _c };
                            }, {}),
                        },
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    getEmails(email) {
        this.setState({
            samsaraIsLoading: true,
            getSamsara: true,
        }, () => {
            this.request.exec(REQUESTS.GET_EMAILS, {
                body: {
                    ...generateQuery(email),
                },
            })
                .then(samsara => {
                    this.setState({
                        samsaraIsLoading: false,
                        samsaraError: null,
                        samsara,
                    });
                })
                .catch((samsaraError) => {
                    this.setState({
                        samsaraError,
                        samsaraIsLoading: false,
                    });
                });
        });

    }

    componentDidUpdate(prevProps: Readonly<IUserRequestsProps>, prevState: Readonly<IUserRequestsState>) {
        const needSamsara = this.state.selectedFilters?.includes(FILTER_TYPES.MAIL) && !this.state.samsara;
        const prevNeedSamsara = prevState.selectedFilters?.includes(FILTER_TYPES.MAIL);

        if (!isObjectEqual(this.previousContext, this.context) || needSamsara !== prevNeedSamsara) {
            const email = UserInfoHandler.getEmail.call(this.context);

            if (needSamsara && this.props.samsara && email !== EMPTY_DATA) {
                this.getEmails(email);
            }
        }

        this.previousContext = this.context;

        if (this.props.id !== prevProps.id) {
            this.getData();
        }
    }

    componentDidMount(): void {
        this.getData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    showClassification(showClassification, tag_id, request_type, tag) {
        this.setState({
            showClassification,
            tag_id,
            request_type,
            tag,
        });
    }

    onClassified() {
        this.setState({
            showClassification: false,
        }, () => {
            setTimeout(this.getData.bind(this), TIMERS.UPDATE_SELECTED);
        });
    }

    selectItem(item: FILTER_TYPES) {
        this.setState((prev) => {
            const selectedFilters = [...prev.selectedFilters];
            const index = selectedFilters.indexOf(item);
            if (index > -1) {
                selectedFilters.splice(index, 1);
            } else {
                selectedFilters.push(item);
            }

            return {
                selectedFilters,
            };
        }, () => {
            this.getData();
        });
    }

    onChangeDate(type, value) {
        this.setState({
            [type]: value,
        }, () => {
            this.getData();
        });
    }

    showChatHistory(showChatHistory, tag) {
        this.setState({
            tag,
            showChatHistory,
        });
    }

    render() {
        const error = <SimpleError error={this.state.error}/>;
        const samsaraError = <SimpleError error={this.state.samsaraError}/>;

        return <div className={style.component}>
            <div>
                <strong>{this.props?.user_type?.toLocaleUpperCase()}</strong>
                {this.state.isLoading ? <Spin size={'s'}/> : null}
                {this.state.samsaraIsLoading ? '  Загрузка из samsara...' : null}
            </div>

            {this.state.error
                ? <Collapse2 initialExpanded={false}
                             title={'Ошибка загрузки данных'}
                             headerClassname={style.error_collapse}
                             expandText={'Развернуть'}
                             children={error}/>
                : null
            }

            {this.state.samsaraError
                ? <Collapse2 initialExpanded={false}
                             title={'Ошибка Сансары'}
                             headerClassname={style.error_collapse}
                             expandText={'Развернуть'}
                             children={samsaraError}/>
                : null
            }

            <FilterRequests selectedFilters={this.state.selectedFilters}
                            selectItem={this.selectItem.bind(this)}
                            since={this.state.since}
                            until={this.state.until}
                            onChangeDate={this.onChangeDate.bind(this)}/>

            <UserRequestsTable data={this.state.response}
                               id={this.props.id}
                               selectedFilters={this.state.selectedFilters}
                               tags={this.props.tags}
                               samsaraTickets={this.state.samsara?.tickets}
                               showChatHistory={this.showChatHistory.bind(this)}
                               showClassification={this.showClassification.bind(this, true)}/>

            {
                this.state.showClassification
                && <ClassificationDialog type={this.state.request_type}
                                         onClassified={this.onClassified.bind(this)}
                                         tag_id={this.state.tag_id}
                                         user_id={this.props.id}
                                         chat_id={this.state?.tag?.tag_details?.topic_link}
                                         onClose={this.showClassification.bind(this, false)}/>
            }
            {
                this.state.showChatHistory
                && <ChatHistoryView user_id={this.props.id}
                                    chat_id={this.state?.tag?.tag_details?.topic_link}
                                    tag_id={this.state?.tag?.tag_details?.tag_id || this.state?.tag?.tag_id}
                                    linked_tag_id={this.state.tag?.tag_details?.linked_tag_id}
                                    onClose={this.showChatHistory.bind(this, false)}/>
            }
        </div>;
    }
}

UserRequests.contextType = UserInfoContext;
