import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { Button, ButtonTypes } from '../../../../ui/Button';
import Checkbox from '../../../../ui/Checkbox';
import DatePicker from '../../../../ui/DatePicker';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import Select from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import style from '../index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { IAddManageRoleCommon, ICurrentRole } from '../types';

interface IRole {
    text: string;
    value: string;
    description: string;
}

interface IAddManageRoleProps extends IAddManageRoleCommon {
    currentRole: ICurrentRole;
}

enum BoolFields {
    isActive = 'isActive',
    isForced = 'isForced'
}

interface IAddManageRoleState extends Dict<any> {
    [key: string]: any;
    rolesIsLoading: boolean;
    roles: IRole[];
    role: string[];
    noise: number;
    deadline: moment.Moment | string;
}

const TEN_YEARS = 10;

export class AddManageRole extends React.Component<IAddManageRoleProps, IAddManageRoleState> {
    state: IAddManageRoleState = {
        rolesIsLoading: false,
        roles: [],
        role: [],
        noise: new Date().getTime(),
        deadline: moment().add(TEN_YEARS, 'y'),
        isActive: true,
        isForced: false,
    };
    request = new Request2({
        requestConfigs,
    });

    onChange(value: string) {
        this.setState({
            deadline: value,
        });
    }

    onSelect(value: string) {
        this.setState({
            role: [value],
        });
    }

    getRoles() {
        this.setState({ rolesIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ROLES, { queryParams: { report: 'compact' } })
                .then((response) => {

                    this.setState({
                        rolesIsLoading: false,
                        role: this.props?.currentRole?.role_id && [this.props?.currentRole?.role_id] || [],
                        isActive: this.props.currentRole && this.props.currentRole.active
                            ? this.props.currentRole.active === '1'
                            : true,
                        isForced: this.props?.currentRole?.forced || false,
                        deadline: this.props.currentRole && this.props.currentRole.deadline
                            && moment(new Date(+this.props.currentRole.deadline * ONE_SECOND))
                            || moment().add(TEN_YEARS, 'y'),
                        roles: response && response.report.map(item => {
                            let description = item.role_description;
                            if(item.role_is_idm === '1') {
                                description = `IDM: ${item.role_description}`;
                            }

                            return { text: item.role_id, value: item.role_id, description };
                        }) || [],
                    });
                });
        });
    }

    onChangeActive(type: string, value: string) {
        this.setState({
            [type]: value,
        });
    }

    componentDidMount() {
        this.getRoles();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    render() {
        return <Window {...this.props}
                       className={`${style.add_role_window}`}
                       title={'Добавление роли'}
                       error={this.props.error}
                       onClose={this.props.onClose}>
            {this.state.rolesIsLoading ? <Spin size={'l'}/>
                : <>
                    <Select options={this.state.roles}
                            placeholder={'Список ролей...'}
                            onSelect={this.onSelect.bind(this)}
                            initialValues={this.state.role}/>

                    <DatePicker value={this.state.deadline}
                                placeholder={'Дедлайн'}
                                onChange={this.onChange.bind(this)}/>

                    <div className={style.boolean_form_item}>
                        <span>Активная роль:</span>
                        <Checkbox checked={!!this.state.isActive}
                                  onChange={this.onChangeActive.bind(this, BoolFields.isActive)}/>
                    </div>
                    <div className={style.boolean_form_item}>
                        <span>Forced:</span>
                        <Checkbox checked={!!this.state.isForced}
                                  onChange={this.onChangeActive.bind(this, BoolFields.isForced)}/>
                    </div>

                    <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                        <Button basic
                                colorType={ButtonTypes.negative}
                                onClick={this.props.onClose.bind(this)}>Отмена</Button>
                        <Button isLoading={this.props.isWorking}
                                colorType={ButtonTypes.positive}
                                onClick={this.props.action.bind(this, this.state)}>Создать</Button>
                    </div>
                </>
            }
        </Window>;
    }
}
