import moment from 'moment';
import * as React from 'react';

import { ONE_SECOND } from '../../../../constants';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Collapse } from '../../../../ui/Collapse';
import { Window } from '../../../../ui/FullModal';
import coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { getEnvironment } from '../getEnvironment';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { IAddManageIDMRole, IAddManageRoleCommon } from '../types';
import style from './index.css';
import { schema } from './schema';

interface IAddManageRoleIDMState {
    rolesIsLoading: boolean;
    rolesIdm: IRoleIDM[];
    selectedRoles: string[];
    deadline: number | null;
    deprive_after_days: string;
    comment: string;
    showSimpleError: boolean;
    error: Error | null | string;
    approvers: {full_name: string; username: string}[];
}

interface IRoleIDM {
    text: string;
    value: string;
    description: string;
}

const DEBOUNCE_TIMER = 250;

export class AddManageRoleIDM extends React.Component<IAddManageIDMRole, IAddManageRoleIDMState> {
    state: IAddManageRoleIDMState = {
        rolesIsLoading: false,
        rolesIdm: [],
        selectedRoles: [],
        deadline: null,
        deprive_after_days: '',
        comment: '',
        error: null,
        showSimpleError: false,
        approvers: [],
    };
    request = new Request2({
        requestConfigs,
    });

    timer: any = null;

    componentDidUpdate(prevProps: Readonly<IAddManageRoleCommon>, prevState: Readonly<IAddManageRoleIDMState>): void {
        const selectedRolesLength = this.state.selectedRoles.length !== prevState.selectedRoles.length;
        const deadline = this.state.deadline !== prevState.deadline;
        const deprive_after = this.state.deprive_after_days !== prevState.deprive_after_days;

        if (selectedRolesLength || deadline || deprive_after) {
            if (this.state.selectedRoles.length === 1 && (this.state.deadline || this.state.deprive_after_days)) {
                clearTimeout(this.timer);
                this.timer = setTimeout(this.simulateIDMRole.bind(this), DEBOUNCE_TIMER);
            }

            if(this.state.selectedRoles.length !== 1) {
                this.setState({
                    showSimpleError: false,
                    error: null,
                    approvers: [],
                });
            }
        }
    }

    simulateIDMRole() {
        let deprive_at = '';

        if (this.state?.deadline) {
            const deadline = this.state.deadline ?? 0;
            deprive_at = moment(deadline * ONE_SECOND).format('YYYY-MM-DD');
        }

        this.request.exec(REQUESTS.SINGLE_REQUEST_IDM_ROLE, {
            body: {
                deprive_at,
                deprive_after_days: this.state.deprive_after_days,
                path: `/${this.state.selectedRoles[0]}/`,
                user: this.props.username,
                user_type: 'user',
                system: getEnvironment(),
                simulate: true,
            },
        },
        ).then((res) => {
            this.setState({
                showSimpleError: false,
                error: null,
                approvers: res?.approvers?.[0],
            });
        }).catch(error => {//от IDM приходят разные ошибки о сроке роли или наличии в уже одобренном статусе. Для наглядности ошибка показывается по-разному
            let errorData = error;
            let showSimpleError = true;

            if (error?.data?.message === 'Invalid data sent') {
                showSimpleError = false;
                errorData = error.data.errors?.deprive_at?.[0] ?? error.data.errors?.deprive_after_days?.[0];

            } else if (error?.data?.message) {
                showSimpleError = false;
                errorData = error.data.message;
            }

            this.setState({
                showSimpleError: showSimpleError,
                error: errorData,
            });
        },
        );
    }

    onSelect(value: string[]) {
        this.setState({
            selectedRoles: value,
        });
    }

    getRoles() {
        this.setState({ rolesIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ROLES, { queryParams: { report: 'compact' } })
                .then((response) => {
                    this.setState({
                        rolesIsLoading: false,
                        rolesIdm: response && response.report.filter(item => item.role_is_idm === '1').map(item => {
                            return { text: item.role_id, value: item.role_id, description: item.role_description };
                        }) || [],
                    });
                });
        });
    }

    componentDidMount() {
        this.getRoles();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onFormChange(formData) {
        this.setState({
            deadline: formData?.deadline || null,
            deprive_after_days: formData?.deprive_after_days?.toString() || '',
            comment: formData?.comment || '',
        });
    }

    render() {
        return <Window {...this.props}
                       className={`${style.add_role_window}`}
                       title={'Запрос роли'}
                       error={this.props.error}
                       onClose={this.props.onClose}>
            {this.state.rolesIsLoading ? <Spin size={'l'}/>
                : <>
                    {
                        this.state.error
                            ? (this.state.showSimpleError
                                ? <Collapse collapsed={true}
                                            title={<span className={style.error}>
                                                Error: {this.state.selectedRoles[0]}
                                            </span>}>
                                    <SimpleError error={this.state.error}/>
                                </Collapse>
                                : <span className={style.error}>{this.state.error}</span>)
                            : null
                    }
                    <Select options={this.state.rolesIdm}
                            placeholder={'Список ролей...'}
                            onSelect={this.onSelect.bind(this)}
                            multiSelect/>
                    <FormConstructor schema={schema}
                                     onChange={this.onFormChange.bind(this)}
                                     hideChanges/>
                    {
                        this.state?.approvers.length
                            ? <>
                                <span>Подтверждающие: </span>
                                {
                                    this.state.approvers.map(user => {
                                        return <Link key={user.username}
                                                     className={style.approveLink}
                                                     href={`https://staff.yandex-team.ru/${user.username}`}
                                                     target="_blank">
                                            {user.full_name}
                                        </Link>;
                                    })
                                }
                            </>
                            : null
                    }
                    <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                        <Button basic
                                colorType={ButtonTypes.negative}
                                onClick={this.props.onClose.bind(this)}>Отмена</Button>
                        <Button isLoading={this.props.isWorking}
                                colorType={ButtonTypes.positive}
                                onClick={this.props.action.bind(this, this.state)}
                                disabled={!this.state.selectedRoles.length || !!this.state.error}>Запросить</Button>
                    </div>
                </>
            }
        </Window>;
    }
}
