import * as React from 'react';

import { ISearchControls, WithSearchControls } from '../../../../../decorators/SearchControls';
import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import FormatDate from '../../../../ui/FormatDate';
import { LabelStatus, TLabel } from '../../../../ui/Table';
import styleTable from '../../../../ui/Table/index.css';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from '../request';

interface IRolesHistoryProps extends ISearchControls {
    user_id: string;
}

interface IRolesHistoryState {
    isLoading: boolean;
    error: Error | null;
    data: Dict<any>;
}

@WithSearchControls()
export class RolesHistory extends React.PureComponent<IRolesHistoryProps, IRolesHistoryState> {
    state: IRolesHistoryState = {
        isLoading: false,
        error: null,
        data: {},
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IRolesHistoryProps>): void {
        if (prevProps.since !== this.props.since || prevProps.until !== this.props.until
            || prevProps.user_id !== this.props.user_id) {
            this.getData();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {

            const { user_id, since, until } = this.props;
            this.request.exec(REQUESTS.GET_ROLES_HISTORY, { queryParams: { user_id, since, until } })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        data: response || {},
                        error: null,
                    });
                })
                .catch((error) => {
                    this.setState({
                        isLoading: false,
                        data: {},
                        error,
                    });
                });
        });

    }

    render() {
        return this.state.isLoading
            ? <Spin/>
            : this.state.error
                ? <SimpleError error={this.state.error} data={{ label: 'История ролей' }}/>
                : <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Время</th>
                            <th>Действие</th>
                            <th className={'center'}>Активность</th>
                            <th>ID роли</th>
                            <th>Дедлайн</th>
                            <th>Имя пользователя</th>
                        </tr>
                    </thead>
                    <tbody>
                        {
                            this.state.data
                        && this.state.data.records
                        && this.state.data.records.reverse().map((el, index: number) => {
                            return <tr key={index}>
                                <td>{index + 1}</td>
                                <td><FormatDate value={el.timestamp * ONE_SECOND} withSecond={true}/></td>
                                <td>{el.action}</td>
                                <td>
                                    <TLabel text={el.active && 'ВКЛ' || 'ВЫКЛ'}
                                            status={el.active && LabelStatus.POSITIVE || LabelStatus.DEFAULT}/>
                                </td>
                                <td className={'strong'}>{el.role_id}</td>
                                <td className={'half-hide'}><FormatDate value={el.deadline * ONE_SECOND}
                                                                        withSecond={true}/>
                                </td>
                                <td>
                                    <a href={`#/clients/${el.user_id}/info`}>
                                        {this.state.data.users && this.state.data.users[el.user_id].username}
                                    </a>
                                </td>
                            </tr>;
                        })
                        }
                    </tbody>
                </table>;
    }
}
