import moment from 'moment';
import * as React from 'react';

import { ISearchControls, WithSearchControls } from '../../../../decorators/SearchControls';
import { ONE_SECOND } from '../../../constants';
import { AbortError, Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import { HistoryTags } from './historyTags';
import { PerformTags } from './performTags';

interface IUserTagsViewProps extends ISearchControls {
    userId: string;
}

interface IUserTagsViewState {
    errorPerform: Error | null;
    isLoadingPerform: boolean;
    errorHistory: Error | null;
    isLoadingHistory: boolean;
    history: any[];
    user_tags_history: any[];
    perform: any[];
    user_tags: any[];
}

const SINCE_SUBTRACT = 3;

@WithSearchControls({ since: moment(new Date()).subtract(SINCE_SUBTRACT, 'd'), forceUrlParams: true })
export class UserTagsView extends React.Component<IUserTagsViewProps, IUserTagsViewState> {
    state: IUserTagsViewState = {
        errorPerform: null,
        isLoadingPerform: false,
        errorHistory: null,
        isLoadingHistory: false,
        history: [],
        user_tags_history: [],
        perform: [],
        user_tags: [],
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        this.request.abort();
        const { userId, since, until } = this.props;
        this.setState({
            isLoadingPerform: true,
            isLoadingHistory: true,
            errorPerform: null,
            errorHistory: null,
        }, async () => {
            try {
                const perform = await this.request.exec(REQUESTS.GET_USER_TAGS_BY_PERFORMER, {
                    queryParams: {
                        performer: userId,
                    },
                });

                const { records, user_tags } = perform;

                this.setState({
                    perform: records,
                    user_tags,
                    isLoadingPerform: false,
                });
            } catch (errorPerform) {
                if (errorPerform.name !== AbortError) {
                    this.setState({
                        errorPerform,
                        isLoadingPerform: false,
                    });
                }
            }

            try {
                const history = await this.request.exec(REQUESTS.GET_USER_TAGS_HISTORY, {
                    queryParams: {
                        user_id: userId,
                        since: since,
                        until: until,
                    },
                });

                const { records, user_tags } = history;
                this.setState({
                    history: records || [],
                    user_tags_history: user_tags || [],
                    isLoadingHistory: false,
                });
            } catch (errorHistory) {
                if (errorHistory.name !== AbortError) {
                    this.setState({
                        errorHistory,
                        isLoadingHistory: false,
                    });
                }
            }
        });
    }

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IUserTagsViewProps>) {
        if (prevProps.userId !== this.props.userId) {
            this.getData();
        }

        if ((prevProps.since !== this.props.since) || (prevProps.until !== this.props.until)) {
            this.getData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            errorHistory,
            isLoadingHistory,
            errorPerform,
            isLoadingPerform,
            perform,
            user_tags,
            history,
            user_tags_history,
        } = this.state;
        const { since, until } = this.props;

        return (
            <>
                {isLoadingPerform ? <Spin/> : errorPerform ?
                    <SimpleError error={errorPerform}/> : (
                        <>
                            <PerformTags label={'авто'}
                                         perform={perform}/>
                            <PerformTags label={'клиента'}
                                         perform={user_tags}/>
                        </>
                    )}

                {isLoadingHistory ? <Spin/> : errorHistory ?
                    <SimpleError error={errorHistory}/> : (
                        <>
                            <HistoryTags label={'авто'}
                                         history={history}
                                         timelapse={`${moment(since * ONE_SECOND).locale('ru').format('D MMMM')}`
                                + ` по ${moment(until * ONE_SECOND).locale('ru').format('D MMMM')}`}/>

                            <HistoryTags label={'клиента'}
                                         history={user_tags_history}
                                         timelapse={`${moment(since * ONE_SECOND).locale('ru').format('D MMMM')}`
                                + ` по ${moment(until * ONE_SECOND).locale('ru').format('D MMMM')}`}/>
                        </>
                    )}
            </>
        );
    }
}
