import { RouterState } from 'connected-react-router';
import * as React from 'react';
import { Redirect, Route, RouteChildrenProps, Switch } from 'react-router';

import { Dict, LSSettingItems } from '../../../types';
import { AdminUserReducerState, CASTOM_RESPONSE_STATUSES } from '../../reducers/adminUserReducer';
import { LocalizationReducerState } from '../../reducers/localizationReducer';
import { Link } from '../../ui/Link';
import { createRoutes } from '../../utils/createRoutes';
import { isLocalhost } from '../../utils/frontendEnvironment';
import { hasWebphonePanel } from '../../utils/hasWebphonePanel';
import LS from '../../utils/localStorage/localStorage';
import { ytLogs } from '../../utils/sendLogs';
import { EVENT_TYPES } from '../../utils/sendLogs/eventTypes/eventTypes';
import { Translate } from '../../utils/translate';
import { IStore } from '../App/store';
import { Copy } from '../Copy';
import { ErrorsWidget } from '../ErrorLogger/ErrorsWidget';
import { Favourites } from '../Favourites';
import { FavouritesWidget } from '../Favourites/FavouritesWidget';
import { ErrorView } from '../GlobalErrorCatchHandler';
import { RESPONSE_STATUSES, SimpleError } from '../SimpleError';
import { CenteredSpin } from '../Spin';
import { Top10Widget } from '../Top10Widget';
import { ChangeEnv, LOGIN_URL } from '../UserMenu/component';
import { VisitedLog } from './db';
import { getFavIcons } from './getFavIcons';
import * as style from './index.css';
import { ErrorSource, initErrorCounter, logError } from './initErrorCounter';
import { VISITED_TYPES, wrapContentId } from './types';

const favIconTesting = require(`./../../../images/favicon_carsharing-testing.png`);

interface IProps extends IStore, RouterState {
}

export class Content extends React.Component<IProps, any> {
    state = {
        error: null,
    };
    ls = new LS();
    log = ytLogs.getInstance();
    favChat;
    favIcon;

    constructor(props: IProps) {
        super(props);
        const icons = getFavIcons();
        this.favIcon = icons.favIcon;
        this.favChat = icons.favChat;
    }

    componentDidMount(): void {
        this.changeFav();
        this.redirectOldUri();
    }

    componentDidUpdate(prevProps) {
        if (this.props.location !== prevProps.location) {
            this.changeFav();
            this.redirectOldUri();
        }

        if (this.props?.AdminUser?.userId !== prevProps?.AdminUser?.userId) {
            this.initErrorCounter();
        }
    }

    initErrorCounter() {
        if (!isLocalhost() && this.props?.AdminUser?.userId) {
            initErrorCounter(this.props.AdminUser.userId);
        }
    }

    async redirectOldUri() {
        const session = location.hash?.split('#/?redirect=/map/order-session/')?.[1]?.split('?')?.[0] ?? null;

        if (session) {
            const carIdIndex = location.hash.indexOf('car_id');
            const carIdUriShift = 7;
            const carIdLength = 36;
            const carId = carIdIndex > -1
                ? location.hash.substr(carIdIndex + carIdUriShift, carIdLength)
                : null;

            const isFake = location.hash.includes('fake');

            await this.log.send({
                data: {
                    event_type: EVENT_TYPES.SHOW_TAB,
                    tab_name: 'deprecated/map/order-session',
                },
            });

            location.href = `#/tracks?status=`
                + `${(carId && isFake) ? `&car_id=${carId}` : ''}`
                + `${(session && !isFake) ? `&session_id=${session}` : ''}`;
        } else {
            const userId = location.hash.split('#/?redirect=/docs/')?.[1]?.split('/')?.[0];

            if (userId) {
                await this.log.send({
                    data: {
                        event_type: EVENT_TYPES.SHOW_TAB,
                        tab_name: 'deprecated/docs/',
                    },
                });

                location.href = `#/clients/${userId}/docs`;
            }
        }
    }

    changeFav() {
        const fav: any = document.querySelector('link[rel=icon]');

        if (location.hash.includes('chats')) {
            fav.href = this.favChat;
            document.title = 'Чаты.Драйв';
        } else {
            fav.href = this.favIcon;
            document.title = 'Я.Драйв';
        }
    }

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo): void {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);

        this.setState({
            error,
        });
    }

    render() {
        const { AdminUser = {} as AdminUserReducerState, Lang = {} as LocalizationReducerState, location } = this.props;
        const t = new Translate(Lang);

        return this.state.error ? <ErrorView error={this.state.error}/> :
            <div className={style.component}>
                {
                    AdminUser.roleIsLoading
                        ? <CenteredSpin size={'l'}/>
                        : AdminUser.error
                            ? <ErrorUser data={AdminUser}/>
                            : AdminUser.rules && <WrapContent blockRules={this.props.AdminUser?.blockRules}>
                                <Switch>
                                    <Route exact
                                           path={'/'}
                                           render={(props: RouteChildrenProps) => {
                                               const queries = new URLSearchParams(location?.search);
                                               const tab_name = queries.has('redirect') ? 'redirect' : 'home';

                                               this.log.send({
                                                   data: {
                                                       event_type: EVENT_TYPES.SHOW_TAB,
                                                       tab_name,
                                                   },
                                               });

                                               return <RootContent {...AdminUser}
                                                                   Lang={Lang}
                                                                   {...props}/>;
                                           }}/>

                                    {
                                        createRoutes(AdminUser.rules, t)
                                    }

                                    <Route render={(props) => {
                                        return <Redirect to={{
                                            pathname: '/',
                                            search: `redirect=${props.location.pathname}${props.location.search}`,
                                        }}/>;
                                    }}/>
                                </Switch>
                            </WrapContent>
                }
            </div>;
    }
}

interface IRootContent extends AdminUserReducerState, RouteChildrenProps, IStore {
}

export class RootContent extends React.Component<IRootContent> {
    state: any = {
        sessions: [],
        cars: [],
        clients: [],
    };

    visitedLog = new VisitedLog();

    componentDidMount(): void {
        this.getLog();
    }

    getLog() {
        this.visitedLog.getTopVisitedCard('session/').then(sessions => {
            this.setState({ sessions });
        });

        this.visitedLog.getTopVisitedCard('cars/').then(cars => {
            this.setState({ cars });
        });

        this.visitedLog.getTopVisitedCard('clients/').then(clients => {
            this.setState({ clients });
        });

        this.getFavourites();
    }

    getFavourites() {
        const favourites = new Favourites().getFormattedData();
        this.setState({ favourites });
    }

    render() {
        const redirectUri = this.props.location
            && this.props.location.search
            && new URLSearchParams(this.props.location.search).get('redirect');
        const fullPath = this.props.location.pathname + this.props.location.search;

        return <div className={style.dashboard}>
            <h2>Привет, <Copy>{this.props.yandex_login}</Copy>!</h2>
            {
                !this.props?.rawSettings?.length
                    ? <SimpleError error={new Error('Нет данных в GVars')}/>
                    : null
            }
            {
                !Object.keys(this.props?.Lang?.localizations || {}).length
                    ? <SimpleError error={new Error('Нет переводов')}/>
                    : null
            }
            <h3>Инструкция по работе с админкой на <Link href={'https://wiki.yandex-team.ru/eva/drive/toall/crmInterface/'}
                                                         target={'_blank'}>
                wiki
            </Link>
            </h3>
            <div className={style.widgets}>
                {this.state.favourites?.length
                    ? <FavouritesWidget path={fullPath}
                                        label={'Избранное'}
                                        update={this.getFavourites.bind(this)}/>
                    : null}
                {this.state.sessions.length
                    ? <Top10Widget withEnv
                                   data={this.state.sessions}
                                   label={'TOP-10 сессий:'}
                                   type={VISITED_TYPES.session}/>
                    : null}
                {this.state.cars.length
                    ? <Top10Widget withEnv data={this.state.cars} label={'TOP-10 авто:'} type={VISITED_TYPES.cars}/>
                    : null}
                {this.state.clients.length
                    ? <Top10Widget withEnv
                                   data={this.state.clients}
                                   label={'TOP-10 пользователей:'}
                                   type={VISITED_TYPES.clients}/>
                    : null}
                <ErrorsWidget/>
            </div>
            {redirectUri && <div className={style.redirect}>
                Ну а такой странички, возможно, и нет: <strong>{redirectUri}</strong>
            </div>}
        </div>;
    }
}

interface IErrorUser {
    data: AdminUserReducerState;
}

export const ErrorUser = React.memo((props: IErrorUser) => {
    const errorStatus = props.data.error
        && props.data.error.response
        && props.data.error.response.status;

    return <WrapContent>
        {
            errorStatus === RESPONSE_STATUSES.CODE_401
                ? <Error401 error={props.data.error}/>
                : errorStatus === CASTOM_RESPONSE_STATUSES['403']
                    ? <ErrorRoles/>
                    : errorStatus === RESPONSE_STATUSES.CODE_418
                        ? <ErrorTeapot error={props.data.error}/>
                        : <ErrorElse error={props.data.error}/>
        }
    </WrapContent>;
});

const ErrorElse = React.memo((props: { error: any }) => {
    return <div className={style.error_401}>
        <h1>Что-то пошло не так :(</h1>
        <SimpleError error={props.error} data={{ label: 'Доступы' }}/>
    </div>;
});

const ErrorTeapot = React.memo((props: { error: any }) => {
    const ls = new LS();
    const allowUrl = `https://prestable.carsharing.yandex-team.ru/#/`;

    return <div className={style.error_418}>
        <div>
            По текущему адресу админки
            <div><img src={`${location.origin}${location.pathname}${getFavIcons().favIcon}`}/><i>{location.origin}</i>
            </div>
            <span className={style.warning}>запрещено</span> ходить в выбранное
            окружение <strong>{ls.get(LSSettingItems.env)}</strong>
            <div>разрешённые окружения:
                <ChangeEnv className={style.env_list}/>
            </div>
        </div>
        <div>
            Доступ к произольному<sup>*</sup> бекенду разрешён по адресу:
            <div><img src={`${location.origin}${location.pathname}${favIconTesting}`}/>
                <Link href={allowUrl}>{allowUrl}</Link>
            </div>
            <div className={style.allow_api}>
                <div><sup>*</sup><span>.(search|carsharing|gencfg-c|yp-c|taxi|taxi-test).yandex.net</span></div>
            </div>
        </div>
    </div>;
});

const Error401 = React.memo((props: { error: any }) => {
    return <div className={style.error_401}>
        <h1>Пользователь не авторизован</h1>
        <SimpleError error={props.error} data={{ label: 'Доступы' }}/>
        <div><a href={LOGIN_URL}>Войти</a></div>
    </div>;
});

const ErrorRoles = React.memo(() => {
    return <div>
        <h1>Нет активных ролей</h1>
    </div>;
});

const WrapContent = React.memo((props: { blockRules?: Dict<boolean>; children }) => {
    const webphoneStyles = hasWebphonePanel(location.href, props.blockRules)
        ? style.webphone_is_open
        : '';

    return <div className={`${style.wrap_content} ${webphoneStyles}`}
                id={wrapContentId}>
        {props.children}
    </div>;
});
