import Dexie from 'dexie';
import moment from 'moment';

import { Dict, ENVIRONMENT, LSSettingItems } from '../../../types';
import { ONE_SECOND } from '../../constants';
import LS from '../../utils/localStorage/localStorage';
import { VISITED_TYPES } from './types';

const STORAGE_COUNT = 1;
const STORAGE_STEP_MONTH = 'month';

export interface IPageVisited {
    id?: number;
    date: number;
    version: string;
    env: ENVIRONMENT;
    hash: string;
    pathname: string;
    display_name: string;
    search: string;
    response: Dict<object>;
}

interface IGetVisitedProps {
    since: number;
    until: number;
}

const VERSION_2 = 2;
const VERSION_3 = 3;
const VERSION_4 = 4;

class VisitedDatabase extends Dexie {
    visited: Dexie.Table<IPageVisited, number>;

    constructor() {
        super('visitedDB');

        this.version(1).stores({
            visited: '++id, date, raw, version, env, hash, pathname, search',
        });

        this.version(VERSION_2).stores({
            visited: '++id, date, raw, version, env, hash, pathname, display_name, search',
        }).upgrade((trans: any) => {
            return trans.visited.toCollection().modify(visited => {
                visited.display_name = '';
            });
        });

        this.version(VERSION_3).stores({
            visited: '++id, date, response, version, env, hash, pathname, display_name, search',
        }).upgrade((trans: any) => {
            trans.visited.clear();

            return trans.visited.toCollection().modify(visited => {
                visited.response = null;
                delete visited.raw;
            });
        });

        this.version(VERSION_4).stores({
            visited: '++id, date, response, version, env, hash, pathname, display_name, search',
        }).upgrade((trans: any) => {
            trans.visited.clear();

            return trans.visited.toCollection().modify(visited => {
                visited.response = null;
            });
        });

        this.visited = this.table('visited');
    }
}

export class VisitedLog {
    db: VisitedDatabase;
    ls = new LS();

    constructor() {
        this.db = new VisitedDatabase();
        this.clearOutdated();
    }

    private _logVisited(data) {
        const CARD_DISPLAY_NAME = 1;
        const CARD_DISPLAY_NAME_INDEX = 2;
        const card = data.pathname && data.pathname.split('/') || [];

        let response = {};
        if (card[CARD_DISPLAY_NAME] === VISITED_TYPES.clients) {
            response = { email: data.response?.setup?.email?.address, id: data.response?.id };
        }

        if (card[CARD_DISPLAY_NAME] === VISITED_TYPES.cars) {
            response = {
                number: data.response?.number,
                model: data.response?.models?.[0]?.name,
                id: data.response?.id,
            };
        }

        if (card[CARD_DISPLAY_NAME] === VISITED_TYPES.session) {
            const session = data.response?.sessions?.[0] || {};
            response = {
                device_diff: session.device_diff,
                number: session.car?.number,
            };
        }

        const obj: IPageVisited = {
            date: new Date().getTime(),
            version: process.env.VERSION || '',
            env: this.ls.get(LSSettingItems.env),
            hash: data.hash || '',
            pathname: data.pathname || '',
            display_name: card[CARD_DISPLAY_NAME_INDEX] || '',
            search: data.search || '',
            response,
        };

        this.db && this.db.visited && !['', '/'].includes(data.pathname) && this.db.visited.put(obj);
    }

    saveLog(response, props) {
        const data = {
            response,
            hash: props.location.hash || '',
            pathname: props.location.pathname || '',
            search: props.location.search || '',
        };

        this._logVisited(data);
    }

    getTopVisitedCard(card: string) {
        const TEN_ENTRIES = 10;

        return this.db.visited
            .toArray()
            .then(response => {
                const obj = response.reduce((_p, _c) => {
                    if (_c.pathname.includes(`/${card}`) && _c.display_name) {
                        if (!_p.hasOwnProperty(_c.display_name)) {
                            _p[_c.display_name] = [_c];
                        } else {
                            _p[_c.display_name].push(_c);
                        }
                    }

                    return _p;
                }, {});

                return Object.entries(obj)
                    .sort((a: any[], b: any[]) => b[1].length - a[1].length)
                    .slice(0, TEN_ENTRIES);
            });
    }

    getTopIds() {
        const ids = {};

        return this.getTopVisitedCard('cars/')
            .then(response => {
                response?.forEach((car) => {
                    ids[car?.[0]] = 'car';
                });
            })

            .then(() => {
                this.getTopVisitedCard('clients/')
                    .then(response => {
                        response?.forEach((user) => {
                            ids[user?.[0]] = 'client';
                        });
                    });

                return ids;
            });
    }

    getVisited(props?: IGetVisitedProps) {
        const { since = 0, until = 0 } = props || {};

        return this.db.visited
            .where('date')
            .between(since * ONE_SECOND, until * ONE_SECOND)
            .toArray()
            .then(response => response.reverse());
    }

    deleteItem(id) {
        return this.db.visited
            .where('id')
            .equals(id)
            .delete();
    }

    deleteAll() {
        return this.db.visited.clear();
    }

    clearOutdated() {
        const UNTIL = +moment().subtract(STORAGE_COUNT, STORAGE_STEP_MONTH);

        return this.db.visited
            .where('date')
            .between(0, UNTIL)
            .delete();
    }
}
