import diff, { Diff } from 'deep-diff';
import React from 'react';

import { Window } from '../../ui/FullModal';
import * as style from './index.css';

type _Diff = Diff<any, any>[] | undefined;

interface IDiffLabelProps {
    parent: any;
    child: any;
    exclude?: string[];
    className?: string;
    maxCount?: number;
}

interface IDiffLabelState {
    diffData: _Diff;
    showModal: boolean;
}

const PATH_MAX_LENGTH = 3;
const KIND = {
    E: 'edit',
    N: 'new',
    D: 'delete',
};

export class DiffLabel extends React.Component<IDiffLabelProps> {
    state: IDiffLabelState = {
        diffData: [] as _Diff,
        showModal: false,
    };

    componentDidMount(): void {
        this.compare();
    }

    compare() {
        const diffData = diff.diff(this.props.parent, this.props.child, (path, key) => {
            return this.props.exclude && this.props.exclude.includes(key) || false;
        });

        this.setState({
            diffData: diffData && diffData.filter(el => el.kind !== 'A') || [],
        });
    }

    showModal(state) {
        this.setState({
            showModal: state,
        });
    }

    render() {
        return <div className={`${style.labels} ${this.props.className || ''}`}>
            {
                this.state.showModal
                    ? <ModalDiff diffData={this.state.diffData} onClose={this.showModal.bind(this, false)}/>
                    : null
            }
            {this.state.diffData && this.state.diffData.length &&
            <span className={style.diff_length}>{this.state.diffData.length}: </span>}
            <Labels onClick={this.showModal.bind(this, true)}
                    diffData={this.props.maxCount
                        ? this.state.diffData?.slice(0, this.props.maxCount)
                        : this.state.diffData}/>
        </div>;
    }
}

const substr = (value) => {
    const SUBSTR_LIMIT = 50;

    return typeof value === 'object'
        ? `${JSON.stringify(value).substr(0, SUBSTR_LIMIT)}...` : value;
};

const ModalDiff = (props: any) => {
    return <Window title={'Diff'} onClose={props.onClose}>
        <div className={style.modal_diff}>
            <table>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>kind</th>
                        <th>path</th>
                        <th>parent</th>
                        <th>child</th>
                    </tr>
                </thead>
                <tbody>
                    {
                        props.diffData.map((diff: any, index) => {
                            const parent = substr(diff.lhs);
                            const child = substr(diff.rhs);

                            return <tr className={style.diff_text} key={index}>
                                <td>{index + 1}</td>
                                <td className={style[`kind_${diff.kind}`]}>{KIND[diff.kind] || diff.kind}</td>
                                <td>{diff.path && diff.path.join('.')}</td>
                                <td><strong>{parent}</strong></td>
                                <td><strong>{child}</strong></td>
                            </tr>;
                        })
                    }
                </tbody>
            </table>
        </div>
    </Window>;
};

const Labels = React.memo((props: any) => {
    return props.diffData && props.diffData
        .map((el: Diff<any, any>, index) => {
            const PENULTIMATE_PATH_INDEX = 2;

            let path: any = el.path || [];
            if (path.length > PATH_MAX_LENGTH) {
                path = `${path[0]}...${path[path.length - PENULTIMATE_PATH_INDEX]}.${path[path.length - 1]}`;
            } else {
                path = path.join('.');
            }

            return <span key={index}
                         className={style.labels_content}
                         title={KIND[el.kind] || el.kind}
                         onClick={props.onClick}>
                <span className={style[`kind_${el.kind}`]}>{path}</span>
                <span className={style.separate}>; </span>
            </span>;
        });
});
