import * as React from 'react';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import Select, { IOptionInfo } from '../../../ui/Select';
import { isElementHasScroll } from '../../../utils/hasElementScroll';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { FormConstructor } from '../../FormConstructor';
import { controlType } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import DocsQueueView from '../DocsQueue';
import { contentTypeEnum, DOC_FORMAT, DOC_TYPES_DISPLAY_NAME, IDocTemplate, ITemplate } from '../DocsTemplates/types';
import * as style from './index.css';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from './request';

enum DocStorageType {
    mds = 'mds',
    raw = 'raw'
}

interface putDocInQueueData {
    document_name: string;
    storage: DocStorageType;
    document_format: string;
}

interface IDocManager {
    templates: IDocTemplate[];
    document_input: Dict<any>;
    documents: Dict<any>;
    content_types: { content_type: any };
}

interface IDocForQueuePickerProps {
    constants: Promise<{ documents_manager: IDocManager }>;
    onClose: () => void;
    packageData?: any[];
    initValues?: Dict<any>;
    filterFields?: string[];
}

interface IDocForQueuePickerState {
    docManager: IDocManager;
    docTemplates: ITemplate[];
    isLoading: boolean;
    loadingError: Error | null;
    docTemplatesOptions: IOptionInfo[];
    isFormDataValid: boolean;
    selectedDocTemplateName: string;
    docFormData: Dict<any>;
    initialDocFormData: Dict<any> | null;
    isOpenPreviewModal: boolean;
    isPuttingDocInQueue: boolean;
    puttingDocError: Error | null;
}

export default class DocForQueuePicker extends React.Component<IDocForQueuePickerProps, IDocForQueuePickerState> {
    state: IDocForQueuePickerState = {
        docManager: {} as IDocManager,
        docTemplates: [] as ITemplate[],
        isLoading: false,
        loadingError: null,
        docTemplatesOptions: [],
        isFormDataValid: false,
        docFormData: { comment: '' },
        selectedDocTemplateName: '',
        initialDocFormData: null,
        isOpenPreviewModal: false,
        isPuttingDocInQueue: false,
        puttingDocError: null,
    };
    request = new Request2({ requestConfigs });
    modal: React.RefObject<any>;

    constructor(props) {
        super(props);
        this.modal = React.createRef();
    }

    componentDidMount(): void {
        this.getDocTemplatesData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getDocTemplatesData() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            Promise.all([
                this.props?.constants,
                this.request.exec(REQUESTS.GET_DOC_TEMPLATES, { queryParams: { for_queue: true } }),
            ]).then(response => {
                const { filterFields, initValues } = this.props;

                const docManager: IDocManager = response[0]?.documents_manager ?? {};
                let docTemplates: ITemplate[] = response[1]?.documents ?? [];

                //Remove all doc templates without any of filter field
                if (filterFields?.length) {
                    docTemplates = docTemplates.filter(docTemplate => {
                        const inputsKeys = docManager?.document_input?.[docTemplate?.document_name]
                            ? Object.keys(docManager.document_input[docTemplate.document_name])
                            : [];

                        return filterFields?.some(filterField => inputsKeys.includes(filterField));
                    });
                }

                //Option for Select with doc templates
                const docTemplatesOptions = docTemplates.map(doc => {
                    return {
                        text: doc.comment,
                        value: doc.document_name,
                        description: doc.document_name,
                    };
                });

                this.setState({ docManager, docTemplates, docTemplatesOptions, isLoading: false }, () => {
                    if (initValues && Object.keys(initValues).length > 0) {
                        this.setInitValues();
                    }
                });
            }).catch(loadingError => {
                this.setState({ isLoading: false, loadingError });
            });
        });
    }

    setInitValues() {
        const { initValues } = this.props;

        const selectedDocTemplateName: string = initValues?.document_name ?? '';
        this.setState({
            selectedDocTemplateName,
            initialDocFormData: initValues ?? null,
        });
    }

    getSelectedDocTemplateInfo(): ITemplate | null {
        const { selectedDocTemplateName, docTemplates } = this.state;

        return selectedDocTemplateName
            ? docTemplates.filter(template => template.document_name === selectedDocTemplateName)?.[0]
            : null;
    }

    onChangeForm(docFormData: Dict<any>, isFormDataValid: boolean) {
        this.setState({ docFormData, isFormDataValid });
    }

    onDocSelect(selectedDocTemplateName: string) {
        const { packageData } = this.props;
        this.setState({ selectedDocTemplateName }, () => {
            if (packageData) {
                this.setPackageData();
            }
        });
    }

    setPackageData() {
        const { packageData = [] } = this.props;
        const { selectedDocTemplateName } = this.state;
        const docFormData: Dict<any> = deepCopy(this.state.docFormData);
        const initialDocFormData: Dict<any> | null = this.state.initialDocFormData || packageData
            ? deepCopy(this.state.initialDocFormData ?? {})
            : null;

        docFormData[`document_parameters_${selectedDocTemplateName}`] = packageData[0];
        if (initialDocFormData) {
            initialDocFormData[`document_parameters_${selectedDocTemplateName}`] = packageData[0];
        }

        this.setState({ docFormData, initialDocFormData });
    }

    openPreviewModal() {
        this.setState({ isOpenPreviewModal: true });
    }

    closePreviewModal() {
        this.setState({ isOpenPreviewModal: false });
    }

    getDataForPreview() {
        const { docFormData } = this.state;
        const selectedDocTemplateInfo = this.getSelectedDocTemplateInfo();
        const document_name = selectedDocTemplateInfo?.document_name ?? '';

        return Object.assign({}, { document_name }, docFormData);
    }

    putDocInQueue() {
        const { docFormData, docTemplates, selectedDocTemplateName } = this.state;

        let data: putDocInQueueData = {
            storage: DocStorageType.mds,
            document_format: DOC_FORMAT.pdf,
            document_name: '',
        };

        if (docFormData) {
            data = Object.assign({}, data, docFormData);
        }

        const selectedDocTemplateInfo = docTemplates
            .filter(template => template.document_name === selectedDocTemplateName)?.[0];
        data.document_name = selectedDocTemplateInfo?.document_name ?? '';

        this.setState({ isPuttingDocInQueue: true, puttingDocError: null }, () => {
            this.request.exec(REQUESTS.PUT_DOC_IN_QUEUE, { body: data })
                .then(() => {
                    this.setState({ isPuttingDocInQueue: false });
                    this.props.onClose();
                })
                .catch(puttingDocError => {
                    this.setState({ puttingDocError, isPuttingDocInQueue: false });
                });
        });
    }

    render() {
        const { onClose } = this.props;
        const {
            selectedDocTemplateName, isOpenPreviewModal, isLoading, docManager,
            initialDocFormData, isPuttingDocInQueue, loadingError, docTemplatesOptions,
            isFormDataValid, puttingDocError,
        } = this.state;

        const selectedDocTemplateInfo = this.getSelectedDocTemplateInfo();
        const contentType = selectedDocTemplateInfo?.content_type;

        const windowBody = this.modal?.current?.containerRef?.current;
        const formButtons = <div className={coreStyle.button_container}>
            <Button onClick={this.putDocInQueue.bind(this)}
                    disabled={!isFormDataValid || isPuttingDocInQueue
                    || !selectedDocTemplateName}>Поместить в очередь</Button>
            <Button basic
                    onClick={this.openPreviewModal.bind(this)}
                    disabled={!isFormDataValid || isPuttingDocInQueue
                    || !selectedDocTemplateName}>Просмотр</Button>
        </div>;

        return <Window title={'Добавить документ в очередь'}
                       onClose={onClose}
                       ref={this.modal}
                       horizontalAutoScroll>
            <div className={style.doc_picker_container}>
                <div className={`${style.pick_doc} ${!!selectedDocTemplateName ? style.selected : ''}`}>
                    {isLoading
                        ? <Spin/>
                        : <>
                            {loadingError
                                ? <SimpleError error={loadingError}
                                               data={{ label: 'Ошбика при загрузке шаблона документа' }}/>
                                : null}
                            <Select options={docTemplatesOptions}
                                    placeholder={'Документ'}
                                    initialValues={selectedDocTemplateName ? [selectedDocTemplateName] : []}
                                    onSelect={this.onDocSelect.bind(this)}/>
                            {selectedDocTemplateInfo
                                ? <div>
                                    {isElementHasScroll(windowBody)
                                        ? formButtons
                                        : null
                                    }
                                    <div>
                                        <h4>
                                            Тип: {contentType === contentTypeEnum.raw
                                                ? DOC_TYPES_DISPLAY_NAME.raw
                                                : contentType === contentTypeEnum.package
                                                    ? DOC_TYPES_DISPLAY_NAME.package
                                                    : contentType}
                                        </h4>
                                    </div>
                                    <InputsForm onChangeForm={this.onChangeForm.bind(this)}
                                                docManager={docManager}
                                                initialData={initialDocFormData ?? null}
                                                docName={selectedDocTemplateName}/>
                                    {puttingDocError
                                        ? <SimpleError error={puttingDocError}
                                                       data={{ label: 'Ошбика при добавлении документа в очередь' }}/>
                                        : null}
                                    {formButtons}
                                </div>
                                : null}
                        </>}
                </div>
                <div className={style.queue_container}>
                    <DocsQueueView isShort/>
                </div>
            </div>
            {isOpenPreviewModal
                ? <DocPreviewModal data={this.getDataForPreview()} onClose={this.closePreviewModal.bind(this)}/>
                : null}
        </Window>;
    }
}

interface IInputsFormProps {
    docName: string;
    docManager: IDocManager;
    initialData: Dict<any> | null;
    onChangeForm: () => void;
}

const InputsForm = React.memo((props: IInputsFormProps) => {
    const { docName, docManager, initialData, onChangeForm } = props;

    const docInputs = docManager?.document_input?.[docName];
    let schema = Object.assign({}, {
        comment: {
            display_name: 'comment',
            order: Infinity,
            type: controlType.text,
        },
    }, docInputs);

    if (schema?.session_evacuation_address) {
        schema = deepCopy(schema);
        schema.session_evacuation_address.order = -1;
    }

    if (initialData) {
        initialData.docName = docName;
    }

    return <div>
        {docName
            ? <FormConstructor onChange={onChangeForm.bind(null)}
                               initialData={initialData ?? null}
                               schema={schema}/>
            : null}
    </div>;
});

const MAX_PREVIEW_LENGTH = 15000;

class DocPreviewModal extends React.Component<any, any> {
    state = {
        isLoading: false,
        previewText: '',
        error: null,
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        const { data } = this.props;

        this.setState({ isLoading: true }, () => {

            this.request.exec(REQUESTS.GET_DOC_PREVIEW, { body: data })
                .then(response => {
                    if (response && response.preview && response.preview.length
                        && response.preview.length > MAX_PREVIEW_LENGTH) {
                        throw new Error('Слишком большая длина превью');
                    }

                    this.setState({ previewText: response.preview, isLoading: false });
                })
                .catch((error) => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    render() {
        const { error, isLoading, previewText } = this.state;
        const { onClose } = this.props;

        return <Window title={'Предварительный просмотр'} onClose={onClose}>
            {error
                ? <SimpleError error={error} data={{ label: 'Ошибка при формировании превью' }}/>
                : isLoading
                    ? <Spin/>
                    : <div>
                        <span>{previewText}</span>
                    </div>}
            <div className={coreStyle.button_container}>
                <Button onClick={onClose.bind(this)}>Закрыть</Button>
            </div>
        </Window>;
    }
}
