import React, { ReactElement } from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { CopyIcon } from '../../../../ui/CopyIcon';
import FormatDate from '../../../../ui/FormatDate';
import { Link } from '../../../../ui/Link';
import { LabelStatus, TLabel } from '../../../../ui/Table';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Copy } from '../../../Copy';
import * as style from '../index.css';
import { IDoc } from '../types';

enum DOC_STATUSES {
    NOT_READY = 'NOT_READY',
    DONE = 'DONE',
    ERROR = 'ERROR'
}

enum STORAGE {
    mds = 'mds',
    base64 = 'base_64',
    raw = 'raw',
    json = 'json'
}

const TOOLTIP_TOP_MARGIN = 8;
const TOOLTIP_LEFT_MARGIN = 32;
const JSON_SPACE_MARGIN = 2;
const MAX_ERROR_LENGTH = 50;

export interface IDocsQueueItemProps {
    index: number;
    item: any;
    _style: any;
    isShort: boolean;
    openExistingChooseDocModal;
    cancelDocBuilding: () => void;
    changeQueue: () => void;
    openErrorModal: () => void;
}

export default class DocsQueueItem extends React.Component<IDocsQueueItemProps> {

    getAuthorInfo(queueItem: IDoc) {
        const { author: id, author_details } = queueItem;

        let displayName = 'Автор';

        if (author_details) {
            const { last_name = '', first_name = '', username = '' } = author_details;
            displayName = last_name && first_name
                ? `${first_name} ${last_name}`
                : username;
        }

        return id ? <Link href={`#/clients/${id}/info`} target={'_blank'}>{displayName}</Link> : EMPTY_DATA;
    }

    render() {
        const {
            index,
            item,
            _style,
            isShort,
            openExistingChooseDocModal,
            cancelDocBuilding,
            changeQueue,
            openErrorModal,
        } = this.props;

        const status = item.status;
        let labelStatus = LabelStatus.DEFAULT;
        switch (status) {
        case DOC_STATUSES.NOT_READY:
            labelStatus = LabelStatus.WARNING;
            break;
        case DOC_STATUSES.DONE:
            labelStatus = LabelStatus.POSITIVE;
            break;
        case DOC_STATUSES.ERROR:
            labelStatus = LabelStatus.NEGATIVE;
            break;
        }

        const storage: STORAGE = item?.inputs?.storage;
        let result: ReactElement | null = null;

        if (!item?.error) {
            switch (storage) {
            case STORAGE.mds:
                result = item?.result_output?.link
                    ? <Link target='_blank' href={item?.result_output?.link}>
                            Документ
                    </Link>
                    : <span title={'Отсутствует ссылка'} className={style.no_link_waring}>Отсутствует ссылка</span>;
                break;
            case STORAGE.json:
                result = <Copy internalTooltipId={`${item?.create_time.toString()}_${index}`}
                               tooltipConfig={{
                                   overridePosition: (position) => ({
                                       left: position?.left - TOOLTIP_LEFT_MARGIN,
                                       top: _style?.top + TOOLTIP_TOP_MARGIN,
                                   }),
                                   place: 'left',
                               }}
                               text={JSON.stringify(item?.result_output ?? {}, null, JSON_SPACE_MARGIN)}>
                    <CopyIcon text={''}/>JSON
                </Copy>;
                break;
            case STORAGE.base64:
                result = item?.result_output?.data
                    ? <Copy internalTooltipId={`${item?.create_time.toString()}_${index}`}
                            tooltipConfig={{
                                overridePosition: (position) => ({
                                    left: position?.left - TOOLTIP_LEFT_MARGIN,
                                    top: _style?.top + TOOLTIP_TOP_MARGIN,
                                }),
                                place: 'left',
                            }}
                            text={item?.result_output?.data ?? ''}>
                        <CopyIcon text={''}/>Base 64
                    </Copy>
                    : <span title={'Отсутствует текст Base 64'}
                            className={style.no_link_waring}>
                        Отсутствует текст Base 64
                    </span>;
                break;
            case STORAGE.raw:
                result = item?.result_output?.data
                    ? <Copy internalTooltipId={`${item?.create_time.toString()}_${index}`}
                            tooltipConfig={{
                                overridePosition: (position) => ({
                                    left: position?.left - TOOLTIP_LEFT_MARGIN,
                                    top: _style?.top + TOOLTIP_TOP_MARGIN,
                                }),
                                place: 'left',
                            }}
                            text={item?.result_output?.data ?? ''}>
                        <CopyIcon text={''}/>Текст raw
                    </Copy>
                    : <span title={'Отсутствует текст raw'}
                            className={style.no_link_waring}>Отсутствует текст raw</span>;
                break;
            default:
                result = <span>{EMPTY_DATA}</span>;
                break;
            }
        }

        const queue = item?.inputs?.queue ?? '';

        const errorText = item?.error?.includes('empty_document') ? 'Ошибка: Пустой документ' : item.error;

        const errorJson = isValidJSONString(item.error) ? JSON.parse(item.error ?? '{}') : {};

        const errorDisplayText = errorText && !isShort
            ? errorText.length > MAX_ERROR_LENGTH
                ? `${errorText.slice(0, MAX_ERROR_LENGTH)}...`
                : errorText
            : '';

        return <div style={_style} className={style.row}>
            <div className={`${style.cell} ${style.index}`}>
                {index + 1}
            </div>
            <div className={`${style.cell} ${style.status}`}>
                <div>
                    <TLabel status={labelStatus} text={status || EMPTY_DATA}/>
                </div>
                {status === DOC_STATUSES.NOT_READY
                    ? <Link onClick={cancelDocBuilding.bind(this, item?.id ?? '')}>Отмена</Link>
                    : null}
            </div>
            <div className={`${style.cell} ${style.doc_name}`}>
                <Link className='interactive-label doc-name'
                      onClick={openExistingChooseDocModal.bind(this, item)}
                      title={item?.inputs?.document_name || EMPTY_DATA}>
                    {item?.inputs?.document_name || EMPTY_DATA}
                </Link>
            </div>
            <div className={`${style.cell} ${style.queue}`}>
                <div className={style.queue_name}>{queue}</div>
                <Link onClick={changeQueue.bind(this, item)} className={style.change_queue_link}>Сменить</Link>
            </div>
            <div className={`${style.cell} ${style.time}`}>
                <div className='create-time'>
                    {item.create_time
                        ? <FormatDate withSecond={true} value={item.create_time * ONE_SECOND}/>
                        : null
                    }
                </div>
                <div>
                    {item.last_update
                        ? <FormatDate withSecond={true} value={item.last_update * ONE_SECOND}/>
                        : null
                    }
                </div>
            </div>
            {!isShort
                ? <div className={`${style.cell} ${style.author}`}>
                    {item.author && this.getAuthorInfo(item)}
                </div>
                : null}
            <div className={`${style.cell} ${style.comment}`}>{item?.inputs?.comment ?? EMPTY_DATA}</div>
            <div className={`${style.cell} ${style.result}`}>
                {result ?? null}
            </div>
            {!isShort
                ? <div className={`${style.cell} ${style.error}`}>
                    {errorJson?.ui_message
                        ? <div className={style.ui_message_error}>
                            {errorJson?.ui_message?.map((uiMessageItem, errorIndex) => {
                                return <div key={`${index}_error_${errorIndex}`} onClick={openErrorModal.bind(this)}>
                                    {uiMessageItem}
                                </div>;
                            })}
                        </div>
                        : null}
                    <span className={style.full_error} onClick={openErrorModal.bind(this)}>
                        {errorDisplayText}
                    </span>
                </div>
                : null}
        </div>;
    }
}
