import moment from 'moment';
import React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import { Confirm } from '../../../ui/FullModal';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Input } from '../../../ui/Input';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { IStore } from '../../App/store';
import { ExcelUploader } from '../../ExcelUploader';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import DocForQueuePicker from '../DocForQueuePicker';
import ChangeDocQueueModal from './ChangeDocQueueModal/component';
import DocsQueueItem, { IDocsQueueItemProps } from './DocsQueueItem/component';
import * as style from './index.css';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from './request';
import { IDoc } from './types';

enum DISPLAY_TYPES {
    my = 'my',
    all = 'all',
    history = 'history',
}

const DISPLAY_TYPES_TABS: TabItem[] = [
    { link: DISPLAY_TYPES.my, name: 'Мои' },
    { link: DISPLAY_TYPES.all, name: 'Все' },
    { link: DISPLAY_TYPES.history, name: 'История' },
];
const UPDATE_TIME = 5000;
const LIST_ITEM_SIZE = 45;

interface IDocsQueueViewProps extends IStore {
    isShort?: boolean;
}

interface IDocsQueueViewState {
    isLoading: boolean;

    currentDisplayMode: string;
    queueDocuments: IDoc[];
    queueDocumentsError: Error | null;
    lastUpdate: moment.Moment | null;
    historySearch: {
        start_ts: number;
        document_name: string;
        user: string;
    };
    isChooseDocModalOpen: boolean;
    currentPreviewDoc: IDoc | null;
    isPackageChooseDocModalOpen: boolean;
    currentPackageData: any[];
    isOpenCancelDocBuildingConfirm: boolean;
    cancellingId: string | null;
    cancellingError: Error | null;
    isOpenChangeQueue: boolean;
    changingItem: IDoc | null;

    isShowErrorModal: boolean;
    errorModalDoc: IDoc | null;
}

export class DocsQueueView extends React.Component<IDocsQueueViewProps, IDocsQueueViewState> {
    state: IDocsQueueViewState = {
        isLoading: false,
        currentDisplayMode: DISPLAY_TYPES.my,
        queueDocuments: [],
        queueDocumentsError: null,
        lastUpdate: null,
        historySearch: {
            start_ts: 0,
            document_name: '',
            user: '',
        },
        isChooseDocModalOpen: false,
        currentPreviewDoc: null,
        isPackageChooseDocModalOpen: false,
        currentPackageData: [],
        isOpenCancelDocBuildingConfirm: false,
        cancellingId: null,
        cancellingError: null,
        isOpenChangeQueue: false,
        changingItem: null,

        isShowErrorModal: false,
        errorModalDoc: null,
    };
    queueUpdateTimer: any = null;
    request = new Request2({ requestConfigs });
    listRef: React.RefObject<HTMLDivElement> | null = null;

    constructor(props: IDocsQueueItemProps) {
        super(props);
        this.listRef = React.createRef();
    }

    componentDidMount(): void {
        this.getData(true);
    }

    componentWillUnmount(): void {
        this.queueUpdateTimer && clearTimeout(this.queueUpdateTimer);
        this.request.abort();
    }

    getListHeight() {
        const BOTTOM_MARGIN = 112;
        const ITEM_SIZE = 10;
        if (this.listRef?.current?.offsetTop) {
            const windowHeight = window.innerHeight;
            const elementTopPosition = this.listRef.current.offsetTop;

            return windowHeight - elementTopPosition - BOTTOM_MARGIN;
        }

        return LIST_ITEM_SIZE * ITEM_SIZE;
    }

    getData(isInitial = false) {
        this.queueUpdateTimer && clearTimeout(this.queueUpdateTimer);
        this.setState({ isLoading: isInitial, queueDocumentsError: null }, () => {
            this.getQueueDocs()
                .catch(queueDocumentsError => {
                    this.setState({ queueDocumentsError });
                })
                .then(() => {
                    this.queueUpdateTimer = setTimeout(() => {
                        this.getData();
                    }, UPDATE_TIME);
                });
        });
    }

    getQueueDocs() {
        const { currentDisplayMode } = this.state;

        if (currentDisplayMode !== DISPLAY_TYPES.history) {
            return this.request.exec(REQUESTS.GET_DOCS_QUEUE,
                { queryParams: { all: currentDisplayMode === DISPLAY_TYPES.all } })
                .then(response => {
                    const queueDocuments: IDoc[] = response?.documents ?? [];
                    queueDocuments.sort((doc1, doc2) => {
                        return doc2.create_time - doc1.create_time;
                    });
                    this.setState({
                        queueDocuments,
                        isLoading: false,
                        queueDocumentsError: null,
                        lastUpdate: moment(),
                    });
                });
        }

        let { start_ts, document_name, user } = this.state.historySearch;
        start_ts /= ONE_SECOND;

        return this.request.exec(REQUESTS.GET_DOCS_HISTORY, { queryParams: { start_ts, document_name, user } })
            .then(response => {
                const queueDocuments: IDoc[] = response?.history ?? [];

                queueDocuments.forEach(queueDocument => {
                    const inputs: any = queueDocument.inputs;
                    const result: any = queueDocument.result_output;

                    if (inputs && isValidJSONString(inputs)) {
                        queueDocument.inputs = JSON.parse(inputs);
                    }

                    queueDocument.result_output = JSON.parse(result);
                });
                queueDocuments.sort((doc1: IDoc, doc2: IDoc) => {
                    return doc2.create_time - doc1.create_time;
                });
                this.setState({
                    queueDocuments,
                    queueDocumentsError: null,
                    isLoading: false,
                });
            });

    }

    openChooseDocModal() {
        this.setState({ isChooseDocModalOpen: true, currentPreviewDoc: null });
    }

    openExistingChooseDocModal(currentPreviewDoc: IDoc) {
        this.setState({ isChooseDocModalOpen: true, currentPreviewDoc });
    }

    closeChooseDocModal() {
        this.setState({ isChooseDocModalOpen: false });
        this.getData(true);
    }

    closePackageChooseDocModal() {
        this.setState({ isPackageChooseDocModalOpen: false });
        this.getData(true);
    }

    onDisplayModeChange(currentDisplayMode: string) {
        this.setState({ currentDisplayMode }, () => {
            this.getData(true);
        });
    }

    onHistorySearchChange(type: string, value: string) {
        const historySearch = this.state.historySearch;
        historySearch[type] = value;
        this.setState({ historySearch });
    }

    packHandler(json: any[]) {
        if (json) {
            const data = json.reduce((res: any[], curr: { [key: string]: any }) => {
                res.push(curr.data);

                return res;
            }, []);

            this.setState({ currentPackageData: data, isPackageChooseDocModalOpen: true });
        }
    }

    openCancelDocBuildingConfirm(cancellingId: string) {
        this.setState({ isOpenCancelDocBuildingConfirm: true, cancellingId });
    }

    closeCancelDocBuildingConfirm() {
        this.setState({ isOpenCancelDocBuildingConfirm: false, cancellingId: null, cancellingError: null });
    }

    cancelDocBuilding() {
        const { cancellingId } = this.state;

        this.request.exec(REQUESTS.CANCEL_DOC_BUILDING, { body: { id: cancellingId } })
            .then(() => {
                this.closeCancelDocBuildingConfirm();
                this.getData();
            })
            .catch(cancellingError => {
                this.setState({ cancellingError });
            });
    }

    openChangeQueue(changingItem: IDoc) {
        this.setState({ isOpenChangeQueue: true, changingItem });
    }

    closeChangeQueue() {
        this.setState({ isOpenChangeQueue: false, changingItem: null });
    }

    openErrorModal(errorModalDoc: IDoc) {
        this.setState({ isShowErrorModal: true, errorModalDoc });
    }

    closeErrorModal() {
        this.setState({ isShowErrorModal: false, errorModalDoc: null });
    }

    render() {
        const { isShort } = this.props;
        const {
            isLoading, cancellingError, changingItem, historySearch, currentDisplayMode,
            lastUpdate, queueDocumentsError,
            queueDocuments, isChooseDocModalOpen, isPackageChooseDocModalOpen,
            isOpenCancelDocBuildingConfirm, isOpenChangeQueue,
            currentPreviewDoc, currentPackageData,
            isShowErrorModal, errorModalDoc,
        } = this.state;
        const { start_ts, document_name, user } = historySearch;

        const listHeight = +Math.floor(this.getListHeight());

        return <div className={style.docs_queue_container}>
            {!isShort
                ? <>
                    <div className={style.button_container}>
                        <Button onClick={this.openChooseDocModal.bind(this)}>Добавить в очередь</Button>
                        <ExcelUploader JSONHandler={this.packHandler.bind(this)}>Загрузить пакет</ExcelUploader>
                    </div>
                    <Tabs className={style.display_mode_tabs}
                          tabs={DISPLAY_TYPES_TABS}
                          selectTab={this.onDisplayModeChange.bind(this)}
                          currentTab={currentDisplayMode}/>
                    {currentDisplayMode === DISPLAY_TYPES.history
                        ? <div className={style.history_settings_container}>
                            <DatePicker value={start_ts}
                                        placeholder={'Дата с'}
                                        onChange={this.onHistorySearchChange.bind(this, 'start_ts')}/>
                            <Input placeholder={'Имя документа'}
                                   value={document_name}
                                   onChange={this.onHistorySearchChange.bind(this, 'document_name')}/>
                            <Input placeholder={'User ID'}
                                   value={user}
                                   onChange={this.onHistorySearchChange.bind(this, 'user')}/>
                            <Button onClick={this.getData.bind(this, true)}>Поиск</Button>
                        </div>
                        : null}
                </>
                : null}
            <div className={style.last_update}>Последнее
                обновление: {lastUpdate
                ? moment(lastUpdate).format('HH:mm:ss')
                : null}</div>
            {queueDocumentsError
                ? <SimpleError error={queueDocumentsError} data={{ label: 'Ошибка при загрузке очереди' }}/>
                : isLoading
                    ? <div className={style.loading_container}>
                        <Spin/>
                    </div>
                    : queueDocuments?.length
                        ? <div className={`${style.doc_queue_list} ${isShort ? style.short : ''}`}>
                            <div ref={this.listRef} className={style.content}>
                                <div className={`${style.row} ${style.header}`}>
                                    <div className={`${style.cell} ${style.index}`}>#</div>
                                    <div className={`${style.cell} ${style.status}`}>Статус</div>
                                    <div className={`${style.cell} ${style.doc_name}`}>Имя документа</div>
                                    <div className={`${style.cell} ${style.queue}`}>Очередь</div>
                                    <div className={`${style.cell} ${style.time}`}>Создание/Обновление</div>
                                    {!isShort ?
                                        <div className={`${style.cell} ${style.author}`}>Автор</div> : null}
                                    <div className={`${style.cell} ${style.comment}`}>Комментарий</div>
                                    <div className={`${style.cell} ${style.result}`}>Результат</div>
                                    {!isShort ? <div className={`${style.cell} ${style.error}`}/> : null}
                                </div>
                                <VirtualList width={'100%'}
                                             height={listHeight}
                                             itemCount={queueDocuments.length}
                                             itemSize={LIST_ITEM_SIZE}
                                             renderItem={({ index, style }) => {
                                                 return <DocsQueueItem key={index}
                                                                       _style={style}
                                                                       index={index}
                                                                       openErrorModal={
                                                                           this.openErrorModal
                                                                               .bind(this, queueDocuments[index])
                                                                       }
                                                                       openExistingChooseDocModal={
                                                                           this.openExistingChooseDocModal
                                                                               .bind(this)
                                                                       }
                                                                       isShort={!!isShort}
                                                                       cancelDocBuilding={
                                                                           this.openCancelDocBuildingConfirm
                                                                               .bind(this)
                                                                       }
                                                                       changeQueue={this.openChangeQueue.bind(this)}
                                                                       item={queueDocuments[index]}/>;
                                             }}/>
                            </div>
                        </div>
                        : <h4>Нету данных в очереди</h4>
            }
            {isChooseDocModalOpen
                ? <DocForQueuePicker initValues={currentPreviewDoc?.inputs ?? {}}
                                     onClose={this.closeChooseDocModal.bind(this)}/>
                : null}
            {isPackageChooseDocModalOpen
                ? <DocForQueuePicker packageData={currentPackageData}
                                     onClose={this.closePackageChooseDocModal.bind(this)}/>
                : null}
            {isOpenCancelDocBuildingConfirm
                ? <Confirm error={cancellingError}
                           question={`Отменить построение документа?`}
                           accept={this.cancelDocBuilding.bind(this)}
                           onClose={this.closeCancelDocBuildingConfirm.bind(this)}/>
                : null}
            {isOpenChangeQueue
                ? <ChangeDocQueueModal item={changingItem}
                                       onClose={this.closeChangeQueue.bind(this)}/>
                : null}
            {isShowErrorModal
                ? <JsonModal onClose={this.closeErrorModal.bind(this)}
                             obj={isValidJSONString(errorModalDoc?.error ?? '')
                                 ? JSON.parse(errorModalDoc?.error ?? '{}') : {}}/>
                : null}
        </div>;
    }
}
