import * as React from 'react';
import { connect } from 'react-redux';

import { ISearchControls } from '../../../../../decorators/SearchControls';
import { Dict } from '../../../../../types';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import { ConstantsKey, fetchConstants } from '../../../../utils/fetchConstants';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import * as style from '../index.css';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { contentTypeEnum, ISchema, ITemplate } from '../types';

interface IDocTemplateSchemaEditorProps {
    templateId?: string;
    location?: Location;
    onClose: () => void;
}

interface IDocTemplateSchemaCreatorState {
    docTemplates: Dict<ITemplate>;
    documentManager: Dict<any>;
    contentTypeInit: string;
    contentType: string;
    documentName: string;
    isLoading: boolean;
    loadingError: Error | null;
    isWorking: boolean;
    warning: string;
    isDeleteConfirmOpen: boolean;
    schemaInit: ISchema | null;
    schema: ISchema;
}

interface Internal extends ISearchControls {
    constants: Promise<any>;
}

const mapDispatchToProps = (dispatch: any) => {
    return {
        constants: fetchConstants([ConstantsKey.DOCUMENTS_MANAGER], dispatch),
    };
};

class DocTemplateSchemaEditorInternal extends
    React.PureComponent<IDocTemplateSchemaEditorProps & Internal, IDocTemplateSchemaCreatorState> {
    state = {
        docTemplates: {},
        documentManager: {},
        contentTypeInit: '',
        contentType: '',
        documentName: '',
        isLoading: false,
        loadingError: null,
        isWorking: false,
        warning: '',
        isDeleteConfirmOpen: false,
        schemaInit: null,
        schema: {} as ISchema,
    };
    contentTA: HTMLTextAreaElement;
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            Promise.all([this.props.constants,
                this.request.exec(REQUESTS.GET_DOC_TEMPLATES, {
                    queryParams: {
                        is_active: 'false',
                        for_queue: 'false',
                    },
                })]).then(response => {

                const documentManager = response[0].documents_manager
                    && response[0].documents_manager || {};

                const docTemplatesArray = response[1].documents || [];
                const docTemplates = docTemplatesArray.reduce((res: { [key: string]: ITemplate }, curr: ITemplate) => {
                    const { document_name = '' } = curr;
                    res[document_name] = curr;

                    return res;
                }, {});

                const currentTemplate = this.props.templateId && docTemplates[this.props.templateId];
                const schemaInit = currentTemplate ?? null;
                const contentTypeInit = currentTemplate && currentTemplate.content_type || '';
                const documentName = currentTemplate && currentTemplate.document_name || '';

                this.setState({
                    isLoading: false,
                    documentManager,
                    schemaInit,
                    contentTypeInit,
                    docTemplates,
                    documentName,
                });
            }).catch(loadingError => {
                this.setState({ isLoading: false, loadingError });
            });
        });
    }

    getDataFromForm() {
        const schema: Dict<any> = this.state.schema;
        const content_type = this.state.contentType;

        const {
            comment = '',
            document_meta = {},
            document_name = '',
            active = false,
        } = schema;

        return {
            content_type,
            comment,
            document_meta,
            document_name,
            active,
        };
    }

    updateTemplate() {
        if (this.isFormValid()) {
            const data = this.getDataFromForm();
            this.setState({ isWorking: true, warning: '' }, () => {
                this.request.exec(REQUESTS.UPSERT_DOC_TEMPLATES, { body: data }).then(() => {
                    this.setState({ isWorking: false });
                    this.props.onClose();
                }).catch(() => {
                    this.setState({ isWorking: false });
                });
            });
        } else {
            this.setState({ warning: 'Не все поля заполнены' });
        }
    }

    deleteTemplate() {
        const schema: Dict<any> = this.state.schema;
        const id: string = schema.document_name;

        this.setState({ isWorking: true }, () => {
            this.request.exec(REQUESTS.REMOVE_DOC_TEMPLATES, { body: { id } }).then(() => {
                this.setState({ isWorking: false });
                this.props.onClose();
            }).catch(() => {
                this.setState({ isWorking: false });
            });
        });
    }

    isTemplateExist(): boolean {
        const docTemplates: { [key: string]: ITemplate } = this.state.docTemplates;

        return !!docTemplates[this.state.documentName];
    }

    isFormValid() {
        const data = this.getDataFromForm();
        const { document_name } = data;

        return !!document_name;
    }

    openDeleteConfirm() {
        this.setState({ isDeleteConfirmOpen: true });
    }

    closeDeleteConfirm() {
        this.setState({ isDeleteConfirmOpen: false });
    }

    onChangeContentType(data: Dict<any>) {
        const contentType = data && data.content_type || contentTypeEnum.raw;
        contentType && this.setState({ contentType });
    }

    onChangeSchema(data: any) {
        this.setState({ schema: data, documentName: data.document_name || '' });
    }

    render() {
        const { loadingError, schemaInit } = this.state;
        const schema = this.state.schema;
        const isTemplateExist = this.isTemplateExist();

        const title = isTemplateExist
            ? 'Обновить'
            : 'Создать';

        const documentManager: Dict<any> = this.state.documentManager;
        const contentTypes = documentManager.content_types;
        const managersDocuments = documentManager.documents;

        return <Window title={`${title} шаблон`} onClose={this.props.onClose.bind(this)} error={loadingError}>
            <div className={`${style.template_editor} ${style.schema}`}>
                {this.state.isLoading
                    ? <Spin/>
                    : <div className={style.template_schema_container}>
                        {contentTypes
                            ? <FormConstructor onChange={this.onChangeContentType.bind(this)}
                                               initialData={this.state.contentTypeInit
                                                   ? { content_type: this.state.contentTypeInit }
                                                   : {}}
                                               schema={contentTypes}/>
                            : null}

                        {managersDocuments?.[this.state?.contentType]
                            ? <FormConstructor onChange={this.onChangeSchema.bind(this)}
                                               initialData={schemaInit ?? null}
                                               schema={managersDocuments[this.state.contentType]}/>
                            : null}

                        <div className={style.template_editor_button_container}>
                            {isTemplateExist ?
                                <Button className={style.template_editor_button}
                                        isLoading={this.state.isWorking}
                                        colorType={ButtonTypes.negative}
                                        onClick={this.openDeleteConfirm.bind(this)}>
                                    Удалить
                                </Button>
                                : null
                            }
                            {this.state.contentType
                            && <Button className={style.template_editor_button}
                                       isLoading={this.state.isWorking}
                                       onClick={this.updateTemplate.bind(this)}>{title}</Button>
                            }
                        </div>
                        {this.state.warning
                        && <p>
                            ⚠️{this.state.warning}
                        </p>}
                    </div>}
            </div>
            {this.state.isDeleteConfirmOpen
            && <Confirm question={`Удалить шаблон ${schema.document_name}?`}
                        accept={this.deleteTemplate.bind(this)}
                        error={null}
                        onClose={this.closeDeleteConfirm.bind(this)}
                        isWorking={this.state.isWorking}/>}
        </Window>;
    }
}

export const DocTemplateSchemaEditor = connect(() => {
    return {};
}, mapDispatchToProps)(DocTemplateSchemaEditorInternal);
