import * as React from 'react';

import { Dict } from '../../../../../types';
import { DeleteButton, SaveButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { controlType, ISchemaItem } from '../../../FormConstructor/types';
import Spin from '../../../Spin';
import * as style from '../index.css';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { contentTypeEnum, DOC_FORMAT, IDocTemplate, ISchema, ITemplate } from '../types';

const POST = 'post';
const THROTTLE_TIMEOUT = 300;

interface IDocTemplateVisualCreatorProps {
    templateId?: string;
    constants: Promise<any>;
    location?: Location;
    onClose: () => void;
}

interface IDocTemplateVisualCreatorState {
    docTemplatesMap: Dict<ITemplate>;
    isLoading: boolean;
    loadingError: Error | null;
    isWorking: boolean;
    schema: ISchema;
    isDeleteConfirmOpen: boolean;
    templatesVariables: IDocTemplate[];
    templateSchema: Dict<ISchemaItem>;
    formData: Dict<any>;
    isFormValid: boolean;
    updateError: Error | null;
}

export default class DocTemplateVisualCreator extends
    React.Component<IDocTemplateVisualCreatorProps, IDocTemplateVisualCreatorState> {
    state: IDocTemplateVisualCreatorState = {
        docTemplatesMap: {},
        templatesVariables: [],
        isLoading: false,
        loadingError: null,
        isWorking: false,
        isDeleteConfirmOpen: false,
        schema: {
            document_name: this.props.templateId || '',
            comment: '',
            groupping_tags: '',
            queue: '',
            active: false,
            use_user_watermark: false,
            document_format: DOC_FORMAT.pdf,
            templates: [],
            raw: {
                content: '',
                additionalParams: [],
            },
            package: {
                documents: [],
                currentDocuments: [],
            },
        },
        templateSchema: {},
        formData: {},
        isFormValid: false,
        updateError: null,
    };
    contentTA: HTMLTextAreaElement;
    timeOutChangeContent: any;
    request = new Request2({
        requestConfigs,
    });
    VISUAL_EDITOR_SCHEMA: Dict<ISchemaItem> = {
        document_name: { type: controlType.string, display_name: 'Имя документ', required: true },
        document_format: {
            type: controlType.variants,
            display_name: 'Формат документа',
            variants: [DOC_FORMAT.pdf, DOC_FORMAT.csv, DOC_FORMAT.json],
            default: DOC_FORMAT.pdf,
            editable: true,
        },
        comment: { type: controlType.string, display_name: 'Комментарий' },
        groupping_tags: {
            display_name: 'Атрибуты группировки',
            type: controlType.string,
        },
        queue: {
            default: 'default',
            display_name: 'Очередь построения',
            type: controlType.string,
        },
        active: { type: controlType.bool, display_name: 'Доступен для построения' },
        use_user_watermark: { type: controlType.bool, display_name: 'Добавить водяной знак' },
    };

    componentDidMount(): void {
        this.getData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        this.setState({ isLoading: true, loadingError: null }, () => {
            Promise.all([
                this.props.constants,
                this.request.exec(REQUESTS.GET_DOC_TEMPLATES, {
                    queryParams: {
                        is_active: 'false',
                        for_queue: 'false',
                    },
                }),
            ])
                .then(response => {
                    const templatesVariables: IDocTemplate[] = response?.[0]?.documents_manager?.templates ?? [];
                    const docTemplatesArray: ITemplate[] = response[1]?.documents ?? [];
                    const schema: ISchema = this.state.schema;

                    const allDocsNames = docTemplatesArray.map((template: ITemplate) => template.document_name);

                    const docTemplatesMap = docTemplatesArray.reduce((res: Dict<ITemplate>, curr: ITemplate) => {
                        const { document_name = '' } = curr;
                        res[document_name] = curr;

                        return res;
                    }, {});

                    const currentTemplate: ITemplate | null = this.props.templateId
                        ? docTemplatesMap[this.props.templateId]
                        : null;

                    if (currentTemplate) {

                        const { document_name, comment, active } = currentTemplate;

                        const useUserWatermark = currentTemplate?.document_meta?.use_user_watermark ?? false;
                        const documentFormat = currentTemplate?.document_meta?.document_format ?? DOC_FORMAT.pdf;
                        const grouppingTags = currentTemplate?.document_meta?.groupping_tags ?? '';
                        const queue = currentTemplate?.document_meta?.queue ?? 'default';
                        const templates = currentTemplate?.document_meta?.templates ?? [];

                        schema.document_name = document_name;
                        schema.comment = comment;
                        schema.groupping_tags = grouppingTags;
                        schema.queue = queue;
                        schema.active = active;
                        schema.use_user_watermark = useUserWatermark;
                        schema.document_format = documentFormat;
                        schema.templates = templates;

                        if (currentTemplate.content_type === contentTypeEnum.raw) {
                            const { content, additional_parameters = [] } = currentTemplate.document_meta;
                            schema.raw.content = content;
                            schema.raw.additionalParams = additional_parameters;
                        } else {
                            const { documents } = currentTemplate.document_meta;
                            schema.package.currentDocuments = documents.map(document => document.document_name);
                        }
                    }

                    schema.package.documents = allDocsNames;

                    const selectForm = {
                        templates: {
                            type: controlType.variants,
                            variants: templatesVariables.map(template => template.name),
                            display_name: 'Необходимые наборы данных',
                            multi_select: true,
                        },
                    };

                    const templateSchema = Object.assign({}, this.VISUAL_EDITOR_SCHEMA, selectForm);

                    this.setState({ templateSchema, docTemplatesMap, templatesVariables, schema, isLoading: false });
                })
                .catch(loadingError => {
                    this.setState({ isLoading: false, loadingError });
                });
        });
    }

    onFormChange(formData: Dict<any>, isFormValid: boolean) {
        this.setState({ formData, isFormValid });
    }

    getDataFromForm() {
        const { formData, schema } = this.state;
        const {
            comment = '', groupping_tags = '', queue = '', document_name = '', active = false,
            use_user_watermark = false, document_format = DOC_FORMAT.pdf, templates = [],
        } = formData;

        const document_meta = {
            content: schema.raw.content || '',
            additional_parameters: schema.raw.additionalParams || [],
            templates,
            document_format,
            groupping_tags,
            queue,
        };

        return {
            content_type: contentTypeEnum.raw,
            comment,
            document_meta,
            document_name,
            active,
            use_user_watermark,
        };
    }

    updateTemplate() {
        const data = this.getDataFromForm();
        this.setState({ isWorking: true, updateError: null }, () => {
            this.request.exec(REQUESTS.UPSERT_DOC_TEMPLATES, { body: data }).then(() => {
                this.setState({ isWorking: false });
                this.props.onClose();
            }).catch(updateError => {
                this.setState({ updateError, isWorking: false });
            });
        });
    }

    deleteTemplate() {
        const { document_name: id } = this.state.schema;

        this.setState({ isWorking: true }, () => {
            this.request.exec(REQUESTS.REMOVE_DOC_TEMPLATES, { body: { id } }).then(() => {
                this.setState({ isWorking: false });
                this.props.onClose();
            }).catch(() => {
                this.setState({ isWorking: false });
            });
        });
    }

    isTemplateExist(): boolean {
        const docTemplatesMap: { [key: string]: ITemplate } = this.state.docTemplatesMap;

        return !!docTemplatesMap[this.state.schema.document_name];
    }

    openDeleteConfirm() {
        this.setState({ isDeleteConfirmOpen: true });
    }

    closeDeleteConfirm() {
        this.setState({ isDeleteConfirmOpen: false });
    }

    changeContent(e: React.FormEvent<HTMLInputElement>) {
        const { value } = e.currentTarget;
        const schema: ISchema = this.state.schema;
        schema && schema.raw && (schema.raw.content = value);

        this.setState({ schema }, () => {
            if (this.timeOutChangeContent) {
                clearTimeout(this.timeOutChangeContent);
            }

            this.timeOutChangeContent = setTimeout(() => {

                const REG_EXP = /\<.*?\>/g;
                const VARIABLE_SPLITTER = ':';

                const matches = value.match(REG_EXP);

                let values = matches && Array.isArray(matches) && matches.length
                    && matches
                        .map((match: string) => {
                            return match.slice(1, match.length - 1);
                        })
                        .filter((match: string) => match.split(VARIABLE_SPLITTER)[0] === POST)
                        .map((match: string) => match.split(VARIABLE_SPLITTER)[1])
                    || [];

                values.length && (values = Array.from(new Set(values)));
                schema && schema.raw && (schema.raw.additionalParams = values);

                this.setState({ schema });
            }, THROTTLE_TIMEOUT);
        });
    }

    addTemplateVarInContent(template: string, variable: string) {
        const { schema, formData } = this.state;

        const { start = 0, end = 0 } = this.getContentSelection(this.contentTA);
        const newContent = formData?.document_format === DOC_FORMAT.pdf
            ? `\\verb|<${template}:${variable}>| `
            : `<${template}:${variable}>`;
        schema.raw.content = schema.raw.content.slice(0, start) + newContent + schema.raw.content.slice(end);
        this.setState({ schema });
    }

    getContentSelection(el: any) {
        const Document: any = document;

        let start = 0, end = 0, normalizedValue, range,
            textInputRange, len, endRange;

        if (typeof el.selectionStart == 'number' && typeof el.selectionEnd == 'number') {
            start = el.selectionStart;
            end = el.selectionEnd;
        } else {
            range = Document.selection.createRange();

            if (range && range.parentElement() == el) {
                len = el.value.length;
                normalizedValue = el.value.replace(/\r\n/g, '\n');

                textInputRange = el.createTextRange();
                textInputRange.moveToBookmark(range.getBookmark());

                endRange = el.createTextRange();
                endRange.collapse(false);

                if (textInputRange.compareEndPoints('StartToEnd', endRange) > -1) {
                    start = end = len;
                } else {
                    start = -textInputRange.moveStart('character', -len);
                    start += normalizedValue.slice(0, start).split('\n').length - 1;

                    if (textInputRange.compareEndPoints('EndToEnd', endRange) > -1) {
                        end = len;
                    } else {
                        end = -textInputRange.moveEnd('character', -len);
                        end += normalizedValue.slice(0, end).split('\n').length - 1;
                    }
                }
            }
        }

        return {
            start: start,
            end: end,
        };
    }

    render() {
        const {
            loadingError, isLoading, isWorking, templatesVariables, schema,
            isDeleteConfirmOpen, templateSchema, isFormValid,
        } = this.state;

        const loadingBlock = <div className={style.template_editor}>
            <Spin/>
        </div>;

        const isTemplateExist = this.isTemplateExist();

        return <Window title={`Визуальный редактор - ${isTemplateExist ? 'Обновить' : 'Создать'} шаблон`}
                       onClose={this.props.onClose.bind(this)}
                       error={loadingError}>
            {isLoading
                ? loadingBlock
                : <div className={style.template_editor}>
                    <div className={style.template_info_container}>
                        <FormConstructor schema={templateSchema}
                                         initialData={schema}
                                         onChange={this.onFormChange.bind(this)}/>
                        <div className={coreStyle.button_container}>
                            <DeleteButton isLoading={isWorking}
                                          disabled={!isTemplateExist || !isFormValid}
                                          onClick={this.openDeleteConfirm.bind(this)}/>
                            <SaveButton isLoading={isWorking}
                                        disabled={!isFormValid}
                                        onClick={this.updateTemplate.bind(this)}/>
                        </div>
                        <TemplatesSuggest templates={templatesVariables}
                                          additionalParams={schema.raw.additionalParams}
                                          addTemplateVarInContent={this.addTemplateVarInContent.bind(this)}/>
                    </div>
                    <div className={style.template_edit_container}>
                        <textarea onChange={this.changeContent.bind(this)}
                                  value={schema.raw.content}
                                  ref={(contentTA: HTMLTextAreaElement) => {
                                      this.contentTA = contentTA;
                                  }}/>
                    </div>
                </div>}
            {isDeleteConfirmOpen
            && <Confirm question={`Удалить шаблон ${schema.document_name}?`}
                        accept={this.deleteTemplate.bind(this)}
                        onClose={this.closeDeleteConfirm.bind(this)}
                        error={null}
                        isWorking={isWorking}/>}
        </Window>;
    }
}

interface ITemplatesSuggestProps {
    templates: IDocTemplate[];
    addTemplateVarInContent: () => {};
    additionalParams: string[];
}

const TemplatesSuggest = (props: ITemplatesSuggestProps) => {
    const { templates = [], addTemplateVarInContent, additionalParams } = props;

    return templates && Array.isArray(templates)
        ? <div className={style.template_inputs}>
            {templates.map((template: IDocTemplate) => {
                return <div key={template.name}>
                    <h4>{template.name}:</h4>
                    <div className={style.template_outputs_container}>
                        {template.name.toLowerCase() === POST
                            ? <>
                                <button key={`add-additional-parameter`}
                                        onClick={addTemplateVarInContent.bind(null, POST, '')}>
                                    +
                                </button>
                                {
                                    additionalParams && Array.isArray(additionalParams) && additionalParams.length
                                        ? additionalParams.map((additionalParam: string, index: number) => {
                                            return <button key={`additionalParam_${index}_${additionalParam}`}
                                                           onClick={addTemplateVarInContent
                                                               .bind(null, template.name, additionalParam)}>
                                                {additionalParam}
                                            </button>;
                                        })
                                        : null
                                }
                            </>
                            : template.outputs && Array.isArray(template.outputs)
                            && template.outputs.map((output: string) => {
                                return <button key={`${template.name}_${output}`}
                                               onClick={addTemplateVarInContent.bind(null, template.name, output)}>
                                    {output}
                                </button>;
                            })

                        }

                    </div>
                </div>;
            })}
        </div>
        : null;
};
