import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import * as styleTable from '../../../../ui/Table/index.css';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { ITemplate } from '../types';
import * as style from './index.css';

const getDiffDisplayValue = (value: string) => {
    return value
        ? (isValidJSONString(value) || Array.isArray(value))
            ? JSON.stringify(value)
            : value.toString()
        : '';
};

interface IHistoryItem {
    document_meta: string;
    history_action: string;
    document_name: string;
    active: string;
    history_timestamp: string;
    content_type: string;
    comment: string;
    history_user_id: string;
    newData?: Dict<any>;
}

interface ITemplateHistoryModalProps {
    currentTemplate: ITemplate;
    onClose: () => void;
}

interface ITemplateHistoryModalState {
    history: IHistoryItem[];
    isLoading: boolean;
    loadingError: Error | null;
}

export class TemplateHistoryModal extends React.Component<ITemplateHistoryModalProps, ITemplateHistoryModalState> {
    state: ITemplateHistoryModalState = {
        history: [],
        isLoading: false,
        loadingError: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        const { currentTemplate } = this.props;
        const { document_name = '' } = currentTemplate;

        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_DOC_TEMPLATE_HISTORY, { queryParams: { document_name } })
                .then(response => {
                    const history: IHistoryItem[] = response?.history ?? [];

                    //Работа с первым элементом, где всё new
                    if (history.length) {
                        Object.entries(history[0])?.forEach(historyEntry => {
                            const [key, value] = historyEntry;

                            if (history[0].newData) {
                                history[0].newData[key] = { new: value };
                            } else {
                                history[0].newData = { [key]: { new: value } };
                            }
                        });
                    }

                    history?.length && history.reduce((prev, curr) => {
                        const prevObject = isValidJSONString(prev?.document_meta)
                            ? JSON.parse(prev.document_meta) : {};
                        const currObject = isValidJSONString(curr?.document_meta)
                            ? JSON.parse(curr.document_meta) : {};

                        Object.keys(currObject).forEach(objKey => {
                            const currValue = currObject[objKey];
                            const prevValue = prevObject[objKey];

                            const prevValueForCompare = getDiffDisplayValue(prevValue);
                            const currValueForCompare = getDiffDisplayValue(currValue);

                            if (prevValueForCompare !== currValueForCompare) {
                                if (curr.newData) {
                                    curr.newData[objKey] = { old: prevValue, new: currValue };
                                } else {
                                    curr.newData = { [objKey]: { old: prevValue, new: currValue } };
                                }
                            }

                        });

                        return curr;
                    });
                    history.reverse();

                    this.setState({ history, isLoading: false });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    render() {
        const { currentTemplate, onClose } = this.props;
        const { loadingError, isLoading, history } = this.state;

        return <Window title={`История изменений документа ${currentTemplate?.document_name ?? ''}`}
                       error={loadingError}
                       onClose={onClose?.bind(this)}>
            <div className={style.template_history}>
                {isLoading
                    ? <Spin/>
                    : history?.length
                        ? <table className={styleTable.table}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Дата/Пользователь</th>
                                    <th>Изменения</th>
                                </tr>
                            </thead>
                            <tbody>
                                {history?.map((historyItem, index) => {
                                    return <HistoryRow key={index} historyItem={historyItem} index={index}/>;
                                })}
                            </tbody>
                        </table>
                        : <h4>Не найдено истории изменений</h4>}
            </div>
        </Window>;
    }
}

const SHORT_SHOW_LENGTH = 3;

interface IHistoryRowProps {
    historyItem: IHistoryItem;
    index: number;
}

interface IHistoryRowState {
    newDataDiffElements: React.ReactElement[];
    newDataLength: number;
    historyItem: IHistoryItem | null;
    ifFullShow: boolean;
}

class HistoryRow extends React.Component<IHistoryRowProps, IHistoryRowState> {
    state: IHistoryRowState = {
        newDataDiffElements: [],
        newDataLength: 0,
        historyItem: null,
        ifFullShow: false,
    };

    componentDidMount() {
        const { historyItem } = this.props;
        const newDataEntries = historyItem.newData ? Object.entries(historyItem.newData) : [];
        const newDataLength = 0;

        const newDataDiffElements = newDataEntries?.reduce((res: React.ReactElement[], newDataEntry) => {
            const [key] = newDataEntry;

            res.push(<DiffElement key={key} diff={newDataEntry}/>);

            return res;
        }, []);

        this.setState({ newDataDiffElements, newDataLength, historyItem });
    }

    changeShowFull() {
        this.setState({ ifFullShow: !this.state.ifFullShow });
    }

    render() {
        const { index } = this.props;
        const { historyItem, newDataLength, ifFullShow, newDataDiffElements } = this.state;

        return historyItem
            ? <tr className={style.diff_row}>
                <td>{index + 1}</td>
                <td>
                    <FormatDate className={style.history_date} value={+historyItem?.history_timestamp * ONE_SECOND}/>
                    <div>
                        <Link href={`#/clients/${historyItem.history_user_id}/info`}>Пользователь</Link>
                    </div>
                    {newDataLength > SHORT_SHOW_LENGTH
                        ? <Link onClick={this.changeShowFull.bind(this)}>
                            {`${ifFullShow ? `Скрыть` : `Развернуть`}(${newDataLength})`}
                        </Link>
                        : null}
                </td>
                <td>
                    {newDataDiffElements
                        ? ifFullShow
                            ? newDataDiffElements
                            : newDataDiffElements.slice(0, SHORT_SHOW_LENGTH)
                        : <div className={style.diff}>
                            <span className={style.no_diff}>Изменений не было</span>
                        </div>}
                </td>
            </tr>
            : null;
    }
}

const DiffElement = (props: { diff: [string, { old: string; new: string }] }) => {
    const { diff } = props;
    const [key, value] = diff;

    const oldValue = value?.old;
    const newValue = value?.new;

    const oldValueDisplay = getDiffDisplayValue(oldValue);
    const newValueDisplay = getDiffDisplayValue(newValue);

    return <div className={style.diff}>
        <span className={style.key}>
            {key}
        </span>:
        {oldValue
            ? <span>
                <span className={`${style.value} ${style.old}`}>
                    {oldValueDisplay}
                </span>
            →
            </span>
            : null}
        <span className={`${style.value} ${style.new}`}>
            {newValueDisplay}
        </span>
    </div>;
};
