import * as React from 'react';
import { Route } from 'react-router';

import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as styleTable from '../../../ui/Table/index.css';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { NAVIGATION } from '../../../utils/navigation';
import { Request2 } from '../../../utils/request';
import { IStore } from '../../App/store';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { DocTemplateSchemaEditor } from './DocTemplateSchemaEditor/component';
import DocTemplateVisualCreator from './DocTemplateVisualCreator/index';
import * as style from './index.css';
import { DOC_TEMPLATE_REQUESTS as requestConfigs, REQUESTS } from './request';
import { TemplateHistoryModal } from './TemplateHistoryModal/component';
import {
    ALL_GROUPING_TAG_TAB,
    contentTypeEnum,
    CURRENT_VISUAL_CONTENT_TYPES,
    DOC_FORMAT,
    DOC_TYPES_DISPLAY_NAME,
    GROUPING_TAG_KEY,
    ISchema,
    ITemplate,
} from './types';

enum EDITOR_TYPE {
    edit = 'edit',
    edit_schema = 'edit_schema'
}

interface IDocsTemplatesViewProps extends IStore {
    templateId?: string;
    location: Location;
}

interface IOption {
    text: string;
    value: string;
}

interface IDocsTemplatesViewState {
    isTemplatesLoading: boolean;
    templatesError: Error | null;
    isTemplateHistoryModalOpen: boolean;
    currentTemplate: ITemplate;
    templates: ITemplate[];
    schemaType: contentTypeEnum;
    isLoading: boolean;
    isWorking: boolean;
    warning: string;
    isDeleteConfirmOpen: boolean;
    schema: ISchema;
    grouppingTagsTabs: TabItem[];
    currentGrouppingTagsTab: string;
    showActiveCheckboxValue: boolean;
    typesOfDocuments: IOption[];
    selectedTypeOfDocument: string;
    filterValue: string;
}

export default class DocsTemplatesView extends React.Component<IDocsTemplatesViewProps, IDocsTemplatesViewState> {
    state: IDocsTemplatesViewState = {
        templates: [],
        schemaType: contentTypeEnum.raw,
        isLoading: false,
        isWorking: false,
        warning: '',
        isDeleteConfirmOpen: false,
        schema: {
            document_name: this.props.templateId || '',
            comment: '',
            groupping_tags: '',
            queue: '',
            active: false,
            use_user_watermark: false,
            document_format: DOC_FORMAT.pdf,
            templates: [],
            raw: {
                content: '',
                additionalParams: [],
            },
            package: {
                documents: [],
                currentDocuments: [],
            },
        },
        isTemplatesLoading: false,
        templatesError: null,
        isTemplateHistoryModalOpen: false,
        currentTemplate: {} as ITemplate,
        grouppingTagsTabs: [],
        currentGrouppingTagsTab: ALL_GROUPING_TAG_TAB,
        showActiveCheckboxValue: false,
        typesOfDocuments: [],
        selectedTypeOfDocument: '',
        filterValue: '',
    };

    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IDocsTemplatesViewProps>): void {
        this.props?.location && prevProps?.location
        && (this.props.location.pathname !== prevProps.location.pathname)
        && (prevProps.location.pathname === `/${NAVIGATION?.ROUTES?.GARAGE?.uri}/docs-templates/`)
        && this.getData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        this.setState({ isTemplatesLoading: true, templatesError: null }, () => {
            this.request.exec(REQUESTS.GET_DOC_TEMPLATES, {
                queryParams: {
                    is_active: 'false',
                    for_queue: 'false',
                },
            })
                .then(response => {
                    const templates = response?.documents ?? [];

                    let grouppingTags = templates.map(template => template?.document_meta?.groupping_tags);
                    grouppingTags = Array.from(new Set(grouppingTags));
                    const allGrouppingTags = [ALL_GROUPING_TAG_TAB, ...grouppingTags];
                    const grouppingTagsTabs = allGrouppingTags.map(attribute => {
                        return {
                            name: attribute,
                            link: attribute,
                            active: true,
                        };
                    });

                    const searchParams = new URLSearchParams(this.props.location?.search ?? '');
                    const currentTab = searchParams.get(GROUPING_TAG_KEY);
                    let currentGrouppingTagsTab = ALL_GROUPING_TAG_TAB;

                    if (currentTab) {
                        if (grouppingTagsTabs?.filter(grouppingTagsTab => grouppingTagsTab.link === currentTab)?.[0]) {
                            currentGrouppingTagsTab = currentTab;
                        }
                    }

                    const typesOfDocuments = new Set();
                    templates.forEach(document => typesOfDocuments.add(document.content_type));
                    const typesOfDocumentsArray = Array.from(typesOfDocuments).map(type => {
                        return { text: this.getDocumentType(type), value: type };
                    });

                    this.setState({
                        templates,
                        typesOfDocuments: typesOfDocumentsArray as IOption[],
                        grouppingTagsTabs,
                        currentGrouppingTagsTab,
                        isTemplatesLoading: false,
                    });
                })
                .catch(templatesError => {
                    this.setState({ templatesError, isTemplatesLoading: false });
                });
        });
    }

    openEditorModal(type: EDITOR_TYPE, template?: ITemplate) {
        const { currentGrouppingTagsTab } = this.state;

        location.href = `#/${NAVIGATION?.ROUTES?.GARAGE?.uri}/docs-templates/${type}/${template && template.document_name || ''}?${GROUPING_TAG_KEY}=${currentGrouppingTagsTab}`;
    }

    openTemplateHistoryModal(currentTemplate: ITemplate) {
        this.setState({ isTemplateHistoryModalOpen: true, currentTemplate });
    }

    closeTemplateHistoryModal() {
        this.setState({ isTemplateHistoryModalOpen: false, currentTemplate: {} as ITemplate });
    }

    getEditorLink(type: EDITOR_TYPE) {
        return `/${NAVIGATION?.ROUTES?.GARAGE?.uri}/docs-templates/${type}/:templateId?`;
    }

    selectTab(currentGrouppingTagsTab: string) {
        this.setState({ currentGrouppingTagsTab });
    }

    onEditorClose() {
        const searchParams = new URLSearchParams(this.props.location?.search ?? '');
        const currentTab = searchParams.get(GROUPING_TAG_KEY) ?? ALL_GROUPING_TAG_TAB;

        location.href = `#/${NAVIGATION?.ROUTES?.GARAGE?.uri}/docs-templates?${GROUPING_TAG_KEY}=${currentTab}`;
    }

    onShowActiveCheckboxValueChange(value: boolean) {
        this.setState({
            showActiveCheckboxValue: value,
        });
    }

    selectTypeOfDocument(type: string) {
        this.setState({
            selectedTypeOfDocument: type,
        });
    }

    getDocumentType(content_type) {
        return content_type === contentTypeEnum.raw
            ? DOC_TYPES_DISPLAY_NAME.raw
            : content_type === contentTypeEnum.package
                ? DOC_TYPES_DISPLAY_NAME.package
                : content_type;
    }

    setFilterInputValue(value: string) {
        this.setState({
            filterValue: value,
        });
    }

    checkIfStringIncludesFilter(target: string) {
        const { filterValue } = this.state;

        return target.toLocaleLowerCase().includes(filterValue.trim().toLocaleLowerCase());
    }

    render() {
        let {
            templatesError, isTemplatesLoading, templates, isTemplateHistoryModalOpen, currentTemplate,
            grouppingTagsTabs, currentGrouppingTagsTab,
        } = this.state;

        if (currentGrouppingTagsTab !== ALL_GROUPING_TAG_TAB) {
            templates = templates
                .filter(template => template?.document_meta?.groupping_tags === currentGrouppingTagsTab);
        }

        if (this.state.showActiveCheckboxValue) {
            templates = templates.filter(template => template.active);
        }

        if (this.state.selectedTypeOfDocument) {
            templates = templates.filter(template => template.content_type === this.state.selectedTypeOfDocument);
        }

        if (this.state.filterValue) {
            templates = templates.filter(template => {
                return this.checkIfStringIncludesFilter(template.document_name)
                    || this.checkIfStringIncludesFilter(template.comment)
                    || this.checkIfStringIncludesFilter(this.getDocumentType(template.content_type));
            });
        }

        return <div className={style.docs_templates_container}>
            <div className={style.controls}>
                <Button basic
                        className={style.button}
                        onClick={this.openEditorModal.bind(this, EDITOR_TYPE.edit)}>
                    Создать в визуальном редакторе
                </Button>
                <Button basic
                        className={style.button}
                        onClick={this.openEditorModal.bind(this, EDITOR_TYPE.edit_schema)}>
                    Создать в редакторе схемы
                </Button>
            </div>
            {templatesError
                ? <SimpleError error={templatesError} data={{ label: 'Ошибка при загрузке шаблонов' }}/>
                : null}
            <div className={style.filterControlsContainer}>
                <Select options={this.state.typesOfDocuments}
                        placeholder={'Выберите тип документа'}
                        onSelect={this.selectTypeOfDocument.bind(this)}
                        containerClassName={style.selectTypeOfDocument}/>
                <Input value={this.state.filterValue}
                       onChange={this.setFilterInputValue.bind(this)}
                       placeholder={'Фильтр'}/>
                <div className={style.checkboxContainer}>
                    <span>Показать активные</span>
                    <Checkbox checked={this.state.showActiveCheckboxValue}
                              onChange={this.onShowActiveCheckboxValueChange.bind(this)}/>
                </div>
            </div>
            <Tabs tabs={grouppingTagsTabs}
                  currentTab={currentGrouppingTagsTab ?? ALL_GROUPING_TAG_TAB}
                  selectTab={this.selectTab.bind(this)}/>
            {isTemplatesLoading
                ? <Spin/>
                : templates?.length
                    ? <table className={`${styleTable.table} ${style.docs_templates_table}`}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Тип документа</th>
                                <th>Статус</th>
                                <th>Идентификатор документа</th>
                                <th>Комментарий</th>
                                <th colSpan={3}/>
                            </tr>
                        </thead>
                        <tbody>
                            {templates.map((template, index) => {
                                const { active, content_type, document_name, comment } = template;

                                return <tr key={document_name}>
                                    <td>{index + 1}</td>
                                    <td>
                                        {this.getDocumentType(content_type)}
                                    </td>
                                    <td>
                                        <TLabel status={active ? LabelStatus.POSITIVE : LabelStatus.DEFAULT}
                                                text={active ? 'Активный' : 'Неактивный'}/>
                                    </td>
                                    <td>
                                        {document_name ?? EMPTY_DATA}
                                    </td>
                                    <td>{comment ?? EMPTY_DATA}</td>
                                    <td>
                                        {CURRENT_VISUAL_CONTENT_TYPES.includes(content_type)
                                            ? <Link onClick={this.openEditorModal
                                                .bind(this, EDITOR_TYPE.edit, template)}>
                                            Визуальный редактор
                                            </Link>
                                            : null}
                                    </td>
                                    <td>
                                        <Link onClick={this.openEditorModal
                                            .bind(this, EDITOR_TYPE.edit_schema, template)}>
                                        Редактор схемы
                                        </Link>
                                    </td>
                                    <td>
                                        <Link onClick={this.openTemplateHistoryModal.bind(this, template)}>
                                        История изменений
                                        </Link>
                                    </td>
                                </tr>;
                            })}
                        </tbody>
                    </table>
                    : <h4>Нет данных о шаблонах</h4>
            }
            <Route path={this.getEditorLink(EDITOR_TYPE.edit)}
                   render={props => {
                       const { templateId } = props.match && props.match.params || '';

                       return <DocTemplateVisualCreator templateId={templateId}
                                                        onClose={this.onEditorClose.bind(this)}/>;
                   }}/>
            <Route path={this.getEditorLink(EDITOR_TYPE.edit_schema)}
                   render={props => {
                       const { templateId } = props.match && props.match.params || '';

                       return <DocTemplateSchemaEditor templateId={templateId}
                                                       onClose={this.onEditorClose.bind(this)}/>;
                   }}/>
            {isTemplateHistoryModalOpen
                ? <TemplateHistoryModal currentTemplate={currentTemplate}
                                        onClose={this.closeTemplateHistoryModal.bind(this)}/>
                : null}
        </div>;
    }
}
