import * as React from 'react';

import { EMPTY_DATA } from '../../constants';
import FormatDate from '../../ui/FormatDate';
import { Link } from '../../ui/Link';
import { NoInformation } from '../../ui/NoInformation';
import { TLabel } from '../../ui/Table';
import { ErrorLog, IError } from './db';
import { ErrorModal } from './ErrorModal';
import { getStatusType } from './getStatusType';
import * as style from './index.css';

enum REDUCED_ENVS {
    'admin' = 'A',
    'testing' = 'T',
    'prestable' = 'P',
    'qa' = 'QA',
    'st' = 'ST',
    'taxi' = 'TA',
    'leasing-cabinet' = 'LC',
    'scooter' = 'SC',
    'scooter_test' = 'SC_T'
}

interface IErrorsWidgetProps {
}

interface IErrorsWidgetState {
    errorInfoOpen: boolean;
    data: IError[];
    currentItem: IError | null;
}

const ERROR_COUNT = 10;
const ERR_PATH = 'settings/error-log';

export class ErrorsWidget extends React.Component<IErrorsWidgetProps, IErrorsWidgetState> {
    state: IErrorsWidgetState = {
        errorInfoOpen: false,
        data: [] as IError[],
        currentItem: null,
    };
    errorLog = new ErrorLog();

    componentDidMount(): void {
        this.getData();
    }

    getData() {

        this.errorLog.getErrors()
            .then(data => {
                this.filterLogs(data);
            });
    }

    filterLogs(data) {
        const filteredItems: IError[] = [];

        data.forEach((item, index) => {
            const prevItem = data[index - 1];

            if (isUniqueItems(item, prevItem, index) && filteredItems.length < ERROR_COUNT) {
                filteredItems.push(item);
            }
        });

        this.setState({ data: filteredItems });
    }

    openErrorInfo(errorInfoOpen, index) {
        this.setState({ errorInfoOpen, currentItem: this.state.data[index] });
    }

    render() {
        const { errorInfoOpen, data, currentItem } = this.state;

        return <div className={style.widget}>
            <div className={style.header}>
                <h4>Последние ошибки</h4>
                <Link href={`${location.href}${ERR_PATH}`}>Все ошибки</Link>
            </div>

            <ul>
                {data?.length
                    ? data.map((item, index) => {
                        return <li key={index}>
                            <div className={style.env}>{REDUCED_ENVS[item?.env] || EMPTY_DATA}</div>
                            <div className={style.date}>
                                <FormatDate value={item?.date} withSecond/>
                            </div>
                            <div>
                                {item?.env
                                    ? <TLabel status={getStatusType(item)}
                                              text={item?.status.toString()}
                                              className={style.status}/>
                                    : EMPTY_DATA
                                }
                            </div>
                            <div className={style.endpoint}>
                                {
                                    reduceEndpoint(decodeURIComponent(item?.endpoint))
                                }
                            </div>
                            <div>
                                <Link onClick={this.openErrorInfo.bind(this, true, index)}>info</Link>
                            </div>
                        </li>;
                    })
                    : <NoInformation/>
                }
            </ul>

            {
                errorInfoOpen
                    ? <ErrorModal onClose={this.openErrorInfo.bind(this)} item={currentItem}/>
                    : null
            }
        </div>;
    }
}

function reduceEndpoint(str) {
    const EXCESS_STR = '/saasnew/api';

    return str.replace(EXCESS_STR, '');
}

export function isUniqueItems(item, prevItem, index) {
    return index === 0
        || item.env !== prevItem.env
        || item.code !== prevItem.code
        || item.endpoint !== prevItem.endpoint;
}
