import produce from 'immer';
import moment from 'moment';
import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { ISearchControls, WithSearchControls } from '../../../decorators/SearchControls';
import { ENVIRONMENT } from '../../../types';
import { Button, ButtonTypes } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Confirm } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import Select from '../../ui/Select';
import { deepCopy } from '../../utils/utils';
import { ErrorLog, IError } from './db';
import { ErrorLoggerTableItem } from './ErrorLoggerTableItem';
import { ErrorModal } from './ErrorModal';
import { isUniqueItems } from './ErrorsWidget';
import * as style from './index.css';

type IProps = ISearchControls

const ENV_OPTIONS = [
    { value: ENVIRONMENT.TESTING },
    { value: ENVIRONMENT.PRESTABLE },
    { value: ENVIRONMENT.ADMIN },
    { value: ENVIRONMENT.CUSTOM },
    { value: ENVIRONMENT.QA },
    { value: ENVIRONMENT.ST },
];

enum ErrorsFilterTypes {
    env = 'env',
    status = 'status',
    endpoint = 'endpoint',
    repeats = 'repeats'
}

const SINCE_SUBTRACT = 7;

@WithSearchControls({ since: moment(new Date()).subtract(SINCE_SUBTRACT, 'd') })
export default class ErrorLoggerView extends React.Component<IProps> {
    state = {
        data: [] as IError[],
        showInfo: false,
        selectedId: 0,
        confirmIsOpen: false,
        error: null,
        isWorking: false,
        envFilter: [] as ENVIRONMENT[],
        statusFilter: [] as number[],
        endpointFilter: '',
        hiddenRepeats: true,
        filteredData: [] as IError[],
    };
    errorLog = new ErrorLog();
    errorStatuses: { value: number }[] = [];

    getData() {
        this.errorLog.getErrors({ since: +this.props.since, until: +this.props.until })
            .then(data => {
                this.setState({
                    data,
                }, () => this.getErrorStatuses());
            });
    }

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IProps>, prevState: Readonly<{}>, snapshot?: any): void {
        if (prevProps.since !== this.props.since
            || prevProps.until !== this.props.until || prevProps.user_id !== this.props.user_id) {
            this.getData();
        }
    }

    showInfo(selectedId) {
        this.setState(produce(this.state, draft => {
            draft.showInfo = true;
            draft.selectedId = selectedId;
        }));
    }

    deleteItem(id) {
        this.errorLog.deleteItem(id).then(() => {
            this.getData();
        });
    }

    closeInfo() {
        this.setState(produce(this.state, draft => {
            draft.showInfo = false;
        }));
    }

    removeAll() {
        this.setState(produce(this.state, draft => {
            draft.error = null;
            draft.isWorking = true;
        }), () => {
            this.errorLog.deleteAll()
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.error = null;
                        draft.confirmIsOpen = false;
                        draft.isWorking = false;
                    }), () => {
                        this.getData();
                    });
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.error = error;
                        draft.confirmIsOpen = false;
                        draft.isWorking = false;
                    }));
                });
        });

    }

    openConfirm(state) {
        this.setState(produce(this.state, draft => {
            draft.confirmIsOpen = state;
        }));
    }

    getErrorStatuses() {
        const statuses: number[] = [];

        this.state.data.forEach(log => {
            if (!statuses.includes(log.status)) {
                statuses.push(log.status);
            }
        });

        this.errorStatuses = statuses.sort((a, b) => a - b).map(s => {
            return { value: s };
        });

        this.filterLogs();
    }

    onFilterChange(selectType, data) {
        if (selectType === ErrorsFilterTypes.env) {
            this.setState({ envFilter: data }, () => this.filterLogs());
        }

        if (selectType === ErrorsFilterTypes.status) {
            this.setState({ statusFilter: data }, () => this.filterLogs());
        }

        if (selectType === ErrorsFilterTypes.endpoint) {
            this.setState({ endpointFilter: data }, () => this.filterLogs());
        }

        if (selectType === ErrorsFilterTypes.repeats) {
            this.setState({ hiddenRepeats: data }, () => this.filterLogs());
        }
    }

    filterLogs() {
        const { envFilter, statusFilter, endpointFilter, hiddenRepeats } = this.state;
        let filteredData = deepCopy(this.state.data);

        filteredData = this.check(ErrorsFilterTypes.env, envFilter, filteredData);
        filteredData = this.check(ErrorsFilterTypes.status, statusFilter, filteredData);
        filteredData = this.check(ErrorsFilterTypes.endpoint, endpointFilter, filteredData);

        if (hiddenRepeats) {
            filteredData = this.checkHidden(filteredData);
        }

        this.setState({ filteredData });
    }

    check(filter, values, data) {
        if (filter === ErrorsFilterTypes.endpoint) {
            return data.filter(log => log[filter].includes(values));
        }

        if (values?.length) {
            return data.filter(log => values.includes(log[filter]));
        }

        return data;
    }

    checkHidden(data) {
        const filteredItems: any[] = [];
        data.forEach((item, index) => {
            const prevItem = data[index - 1];
            if (isUniqueItems(item, prevItem, index)) {
                item.repeated = 1;
                filteredItems.push(item);
            } else {
                filteredItems[filteredItems.length - 1].repeated++;
            }
        });

        return filteredItems;
    }

    render() {
        const obj = this.state.data?.find(el => el.id === this.state.selectedId) || {};

        return <div className={style.component}>
            {
                this.state.showInfo
                    ? <ErrorModal onClose={this.closeInfo.bind(this)} item={obj}/>
                    : null
            }
            {this.state.confirmIsOpen && <Confirm onClose={this.openConfirm.bind(this, false)}
                                                  error={this.state.error}
                                                  isWorking={this.state.isWorking}
                                                  accept={this.removeAll.bind(this)}
                                                  question={'Удалить всё?'}/>}
            <div className={style.controls}>
                <Button colorType={ButtonTypes.negative} onClick={this.openConfirm.bind(this, true)}>
                    Удалить
                </Button>

                <div className={style.filters}>
                    <Select options={ENV_OPTIONS}
                            onSelect={this.onFilterChange.bind(this, ErrorsFilterTypes.env)}
                            multiSelect={true}
                            placeholder={'Окружение'}
                            className={style.selects}/>
                    <Select options={this.errorStatuses}
                            onSelect={this.onFilterChange.bind(this, ErrorsFilterTypes.status)}
                            multiSelect={true}
                            placeholder={'Код ошибки'}
                            className={style.selects}/>
                    <Input onChange={this.onFilterChange.bind(this, ErrorsFilterTypes.endpoint)}
                           value={this.state.endpointFilter}
                           placeholder={'endpoint'}
                           className={style.input}/>
                    <div className={style.filter_checkbox}>
                        <span className={style.placeholder}>Скрыть повторяющиеся:</span>
                        <Checkbox checked={this.state.hiddenRepeats}
                                  onChange={this.onFilterChange.bind(this, ErrorsFilterTypes.repeats)}/>
                    </div>
                </div>
            </div>
            <div className={style.container}>
                {this.state.filteredData.length > 0 ?
                    <ErrorLoggerTable data={this.state.filteredData}
                                      showInfo={this.showInfo.bind(this)}
                                      deleteItem={this.deleteItem.bind(this)}/> : <div>Нет данных для отображения</div>
                }

            </div>
        </div>;
    }
}

interface IErrorLoggerTableProps {
    data: any;
    showInfo: (index: number) => void;
    deleteItem: (index: number) => void;
}

const TABLE_PADDING = 365;
const LIST_ITEM_SIZE = 50;

const ErrorLoggerTable = React.memo((props: IErrorLoggerTableProps) => {
    const { data, showInfo, deleteItem } = props;

    const [windowSize, setWindowSize] = React.useState(() => {
        return { width: window.innerWidth, height: window.innerHeight };
    });
    React.useEffect(() => {
        const onResize = () => setWindowSize({ width: window.innerWidth, height: window.innerHeight });
        window.addEventListener('resize', onResize);

        return () => window.removeEventListener('resize', onResize);
    });

    const listHeight = windowSize.height - TABLE_PADDING;

    return <div className={style.error_logger_list}>
        <div className={style.content}>
            <div className={style.header}>
                <span className={style.cell}>#</span>
                <span className={style.cell}>date</span>
                <span className={style.cell}>env</span>
                <span className={style.cell}>ver.</span>
                <span className={style.cell}>code</span>
                <span className={style.cell}>error</span>
                <span className={style.cell}/>
                <span className={style.cell}/>
            </div>

            <VirtualList width={'100%'}
                         height={listHeight}
                         itemCount={data.length}
                         itemSize={LIST_ITEM_SIZE}
                         renderItem={({ index, style }) => {
                             const item = data[index];

                             return <ErrorLoggerTableItem key={item.id}
                                                          style={style}
                                                          index={index}
                                                          item={item}
                                                          showInfo={showInfo.bind(null, item.id)}
                                                          deleteItem={deleteItem.bind(null, item.id)}/>;
                         }}/>
        </div>
    </div>;
});
