import Dexie from 'dexie';
import moment from 'moment';

import { ENVIRONMENT } from '../../../types';
import { ONE_HUNDRED_ITEMS, ONE_SECOND } from '../../constants';

export interface IError {
    date: number;
    id: number;
    data: any;
    endpoint: string;
    string: string;
    status: number;
    version: string;
    env: ENVIRONMENT;
}

interface IGetErrorsProps {
    since: number;
    until: number;
}

export const STORAGE_STEP_DAYS = 'days';
const STORAGE_COUNT = 4;
const VERSION_1 = 1;
const VERSION_2 = 2;
const VERSION_3 = 3;
const VERSION_4 = 4;

class ErrorDatabase extends Dexie {
    errors: Dexie.Table<IError, number>;

    constructor() {
        super('errorDB');
        this.version(VERSION_1).stores({
            errors: '++id, date, data, endpoint, message, string, status, raw',
        });

        this.version(VERSION_2).stores({
            errors: '++id, date, data, endpoint, string, status, raw, version, env',
        }).upgrade((trans: any) => {
            return trans.errors.toCollection().modify(errors => {
                errors.version = '';
                errors.env = '';
                delete errors.message;
            });
        });

        this.version(VERSION_3).stores({
            errors: '++id, date, data, endpoint, string, status, version, env',
        }).upgrade((trans: any) => {
            trans.errors.clear();

            return trans.errors.toCollection().modify(errors => {
                delete errors.raw;
            });
        });

        this.version(VERSION_4).stores({
            errors: '++id, date, data, requestData, endpoint, string, status, version, env',
        }).upgrade((trans: any) => {
            trans.errors.clear();

            return trans.errors.toCollection().modify(errors => {
                errors.requestData = '{}';
            });
        });

        this.errors = this.table('errors');
    }
}

export class ErrorLog {
    db: ErrorDatabase;

    constructor() {
        try {
            this.db = new ErrorDatabase();
            this.clearOutdated();
        } catch (e) {
            this.db = {} as ErrorDatabase;
        }
    }

    logError(error) {
        const errorObj: any = {
            date: new Date().getTime(),
            endpoint: error && error.config && error.config.endpoint,
            method: error && error.config && error.config.method,
            message: JSON.stringify(error && error.response && error.response.data || {}),
            requestData: JSON.stringify(error?.request || {}),
            data: error.data || null,
            status: error && error.response && error.response.status || 0,
            version: process.env.VERSION,
            env: error.env,
        };
        this.db && this.db.errors && this.db.errors.put(errorObj);
    }

    getErrors(props?: IGetErrorsProps) {
        const { since = 0, until = new Date().getTime() } = props || {};

        return this.db.errors
            .where('date')
            .between(since * ONE_SECOND, until * ONE_SECOND)
            .limit(ONE_HUNDRED_ITEMS)
            .toArray()
            .then(response => response.reverse());
    }

    deleteItem(id) {
        return this.db.errors
            .where('id')
            .equals(id)
            .delete();
    }

    deleteAll() {
        return this.db.errors.clear();
    }

    clearOutdated() {
        const UNTIL = +moment().subtract(STORAGE_COUNT, STORAGE_STEP_DAYS);

        return this.db.errors
            .where('date')
            .between(0, UNTIL)
            .delete();
    }
}
