import * as React from 'react';
import XLSX, { WorkBook, WorkSheet } from 'xlsx';

import { Dict } from '../../../types';
import { ALPHABET_LOWER_CASE, EMPTY_DATA } from '../../constants';
import { Button } from '../../ui/Button';
import { generateRange } from '../../utils/utils';

//26-ричная система исчисления
const alphabetNumberSystem = ALPHABET_LOWER_CASE.length;

interface IBorders {
    [key: string]: { row: string; col: string };

    start: { row: string; col: string };
    end: { row: string; col: string };
}

export interface IJSONData {
    sheetName: string;
    data: Dict<string>[];
}

interface IExcelUploaderProps {
    JSONHandler: (JSON: IJSONData[], workbook: XLSX.WorkBook) => void;
    disabled?: boolean;
    title?: string;
}

interface IExcelUploaderState {
    isProcessingFile: boolean;
}

export class ExcelUploader extends React.Component <IExcelUploaderProps, IExcelUploaderState> {
    state = {
        isProcessingFile: false,
    };

    getSheetBorders(sheet: { [key: string]: any }): IBorders {
        const cellsIndexes = Object.keys(sheet).filter(key => /^[A-Za-z]+[0-9]+$/ig.test(key));
        let maxLetterIndex = 'A';
        let maxNumberIndex = 0;
        cellsIndexes.forEach((cellsIndex: string) => {
            const letterIndex = cellsIndex.replace(/[0-9]+/ig, '');
            const numberIndex = +cellsIndex.replace(/[A-Za-z]+/ig, '');

            maxLetterIndex = letterIndex.localeCompare(maxLetterIndex) ? letterIndex : maxLetterIndex;
            maxNumberIndex = Math.max(maxNumberIndex, numberIndex);
        });

        return {
            start: { row: '1', col: 'A' },
            end: { row: maxNumberIndex.toString(), col: maxLetterIndex },
        };
    }

    getColumnIndexInDecimal(columnIndex: string): number {
        return columnIndex.split('')
            .reduce((result: number, letter: string, index: number, arr: string[]) => {
                const letterInNumber: number = ALPHABET_LOWER_CASE.indexOf(letter.toLowerCase()) + 1;
                const power = arr.length - (index + 1);
                result += letterInNumber * Math.pow(alphabetNumberSystem, power);

                return result;
            }, 0);
    }

    getColumnIndexInLetters(columnIndex: number): string {
        const subtractNextPower = (array: number[], startIndex: number) => {
            if (array[startIndex + 1] === 0) {
                subtractNextPower(array, startIndex + 1);
            }

            array[startIndex] = alphabetNumberSystem;
            array[startIndex + 1]--;
        };

        const letterIndexes: any[] = [];
        while (columnIndex > alphabetNumberSystem) {
            const modulo = columnIndex % alphabetNumberSystem;
            letterIndexes.unshift(modulo);
            columnIndex = Math.floor(columnIndex / alphabetNumberSystem);
        }

        letterIndexes.unshift(columnIndex);

        return letterIndexes.reverse()
            .map((letterIndex: number, index: number, arr: number[]) => {
                if (letterIndex === 0) {
                    subtractNextPower(arr, index);
                    letterIndex = arr[index];
                }

                return ALPHABET_LOWER_CASE[letterIndex - 1].toUpperCase();
            })
            .reverse()
            .join('');
    }

    generateColsLetterIndexes(startCol: string, endCol: string) {
        const startDecimal = this.getColumnIndexInDecimal(startCol);
        const endDecimal = this.getColumnIndexInDecimal(endCol);

        const range = generateRange(startDecimal, endDecimal, 1);

        return range.map((index: number) => this.getColumnIndexInLetters(index));
    }

    getObjectsArray(sheet: WorkSheet) {
        const borders = this.getSheetBorders(sheet);
        const colStartIndex: string = borders && borders.start
            && borders.start.col && borders.start.col.toLowerCase() || EMPTY_DATA;
        const colEndIndex: string = borders && borders.end
            && borders.end.col && borders.end.col.toLowerCase() || EMPTY_DATA;
        const cols = this.generateColsLetterIndexes(colStartIndex, colEndIndex);

        const rowStartIndex = borders && borders.start && +borders.start.row;
        const rowEndIndex = borders && borders.end && +borders.end.row;
        const rows = generateRange(rowStartIndex, rowEndIndex, 1);

        const rowsMap = rows.reduce((result: string[][], rowIndex: number) => {
            const row = cols.map((colLetter: string) => `${colLetter.toUpperCase()}${rowIndex}`);
            result.push(row);

            return result;
        }, []);

        const keys = rowsMap?.[0]?.map((cellAddress: string) => {
            return sheet?.[cellAddress]?.v?.toLowerCase?.() ?? EMPTY_DATA;
        }) || [];

        return rowsMap.slice(1)
            .filter((rowAddresses: string[]) => rowAddresses.some((rowAddress: string) => sheet[rowAddress]))
            .map((rowAddresses: string[]) => {
                return rowAddresses.reduce((result: { [key: string]: string }, curr: string, index: number) => {
                    const key = keys[index];
                    sheet?.[curr]?.v !== null || sheet?.[curr]?.v !== undefined
                        ? result[key] = sheet?.[curr]?.v?.toString?.()
                        : null;

                    return result;
                }, {});
            });
    }

    getWorkbookData(workbook: WorkBook) {
        const { SheetNames: sheetNames, Sheets: sheets } = workbook;

        return sheetNames.map((sheetName: string) => {
            return Object.assign({ sheetName }, sheets[sheetName]);
        }).filter((sheet: any) => sheet['!ref'])
            .map((sheet: any) => {
                return {
                    sheetName: sheet.sheetName,
                    data: this.getObjectsArray(sheet),
                };
            });
    }

    uploadModelsFile() {
        const rABS = true;

        const file: HTMLInputElement = document.createElement('input');
        file.type = 'file';
        file.style.display = 'none';
        document.body.appendChild(file);
        file.click();
        file.onchange = (e: any) => {
            this.setState({ isProcessingFile: true }, () => {
                e.stopPropagation();
                e.preventDefault();
                const files = e.target.files, f = files[0];
                const reader = new FileReader();
                reader.onload = (e: any) => {
                    let data = e.target.result;
                    if (!rABS) {
                        data = new Uint8Array(data);
                    }

                    const workbook = XLSX.read(data, { type: rABS ? 'binary' : 'array' });
                    const JSONData = workbook && this.getWorkbookData(workbook);
                    this.setState({ isProcessingFile: false }, () => {
                        this.props && this.props.JSONHandler && this.props.JSONHandler(JSONData, workbook);
                    });
                };

                if (rABS) {
                    reader.readAsBinaryString(f);
                } else {
                    reader.readAsArrayBuffer(f);
                }
            });
        };
    }

    render() {
        return <Button basic
                       disabled={this.props.disabled}
                       isLoading={this.state.isProcessingFile}
                       onClick={this.uploadModelsFile.bind(this)}>
            {this.props.title || 'Загрузить пакет'}
        </Button>;
    }
}

export interface ISheetData {
    sheetName: string;
    columnNames: string[];
    rows: Dict<any>[];
}

interface IDownloadData {
    fileName: string;
    sheets: ISheetData[];
}

export class ExcelDownloader {

    static downloadFile(props: IDownloadData) {
        const { fileName, sheets } = props;

        const workbook = XLSX.utils.book_new();
        workbook.Props = {
            CreatedDate: new Date(),
        };

        sheets.forEach(sheet => {
            const { sheetName, columnNames, rows } = sheet;

            workbook.SheetNames.push(sheetName);
            const worksheetData = [columnNames];

            const orderedRows = rows.map(row => {
                return columnNames.map(columnName => {
                    return row[columnName] ?? '';
                });
            });

            worksheetData.push(...orderedRows);

            workbook.Sheets[sheetName] = XLSX.utils.aoa_to_sheet(worksheetData);

        });

        XLSX.writeFile(workbook, fileName);
    }
}
