import React from 'react';

import { LocalizationReducerState } from '../../reducers/localizationReducer';
import { Button, ButtonTypes } from '../../ui/Button';
import { FileListView } from '../../ui/FileListView';
import { Window } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import { Translate } from '../../utils/translate';
import { generateuuid4 } from '../../utils/utils';
import { IStore } from '../App/store';
import { ProgressBar } from '../ProgressBar';
import { IProgressBarProps } from '../ProgressBar/types';
import * as style from './index.css';

export enum localizationNamespace {
    Modal = 'modal'
}

export interface IFileChooserDialogProps extends IStore {
    onClose: () => void;
    error: Error | null;
    upload: (files: File[], inputNameArr: string[]) => void;
    isUploading: boolean;
    onlyPDF?: boolean;
    fileCountLimit: number;
    withInput?: boolean;
    progressBarObj?: IProgressBarProps;
}

interface IFileChooserDialogState {
    files: File[];
    pdfData: {
        link: string;
        name: string;
    };
    inputNameArr: string[];
}

export default class FileChooserDialog extends React.Component<IFileChooserDialogProps, IFileChooserDialogState> {
    t = new Translate(this.props.Lang && this.props.Lang || {} as LocalizationReducerState);
    state: IFileChooserDialogState = {
        files: [],
        pdfData: {
            link: '',
            name: '',
        },
        inputNameArr: [],
    };

    componentDidMount(): void {
        this.t.setNamespace(localizationNamespace.Modal);
    }

    openFiles(data: FileList) {
        this.setState({
            inputNameArr: [],
        });
        const renamedFiles = [...data].map((file, index) => {
            const blob = file.slice(0, file.size, file.type);
            const newFile = new File([blob], generateuuid4(), { type: file.type });
            this.state.inputNameArr.push(file.name);

            return newFile;
        });

        this.setState({
            files: renamedFiles || [],
        });
    }

    replaceInputNameArr(index, newValue) {
        const localInputNameArr = this.state.inputNameArr;
        localInputNameArr.splice(index, 1, newValue);
        this.setState({
            inputNameArr: localInputNameArr,
        });
    }

    removeItem(index) {
        this.setState((prev: any) => {
            const files = [...prev.files];
            const inputNameArr = [...prev.inputNameArr];
            inputNameArr.splice(index, 1);
            files.splice(index, 1);
            prev.inputNameArr = inputNameArr;
            prev.files = files;

            return {
                ...prev,
            };
        });
    }

    setPdfLink(type, value) {
        this.setState({
            pdfData: {
                ...this.state.pdfData,
                [type]: value,
            },
        });
    }

    render() {
        return <Window onClose={this.props?.onClose?.bind(this)}
                       title={'Загрузка файлов'}
                       error={this.props.error}>
            {this.props.onlyPDF
                ? <>
                    <Input value={this.state.pdfData.link}
                           placeholder={'Ссылка на PDF'}
                           onChange={this.setPdfLink.bind(this, 'link')} />

                    <Input value={this.state.pdfData.name}
                           placeholder={'Имя документа'}
                           onChange={this.setPdfLink.bind(this, 'name')} />

                    <div className={style.right}>
                        <Button colorType={ButtonTypes.positive}
                                disabled={!this.state.pdfData.link || !this.state.pdfData.name}
                                className={style.upload_button}
                                onClick={this.props?.upload?.bind(this, this.state.pdfData)}>
                            Загрузить
                        </Button>
                    </div>
                </>
                : <>
                    <div className={style.files_container}>
                        <Button file
                                onClick={this.openFiles.bind(this)}
                                basic>Выбрать файл</Button>
                        <FileListView files={this.state.files}
                                      removeItem={this.removeItem.bind(this)}
                                      withInput={this.props.withInput}
                                      inputNameArr={this.state.inputNameArr}
                                      replaceInputNameArr={this.replaceInputNameArr.bind(this)}/>
                    </div>
                    <div className={style.right}>
                        <Button colorType={ButtonTypes.positive}
                                disabled={this.state.files.length > this.props.fileCountLimit
                                    || !this.state.files.length}
                                onClick={() => {
                                    this.props?.upload(this.state.files, this.state.inputNameArr);
                                }
                                }
                                isLoading={this.props.isUploading}>Загрузить</Button>
                    </div>
                    { this.props.progressBarObj &&
                        <ProgressBar allLength={this.props.progressBarObj?.allLength}
                                     successLength={this.props.progressBarObj?.successLength}
                                     errors={this.props.progressBarObj?.errors}/>
                    }
                </>
            }
        </Window>;
    }
}
