import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { Button, CancelButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { rubs } from '../../../../utils/rubs';
import { FINE_REQUEST as requestConfigs, FINE_REQUEST_ENUM as REQUESTS } from '../../request';
import * as style from '../index.css';
import { ShortFineInfoBlock } from '../ShortFineInfo';

const KOAP_MAX_LENGTH = 85;

interface IFineDialogProps {
    sendMessage: (item: Dict<any> ) => void;
    onClose: () => void;
    userId: string;
}

interface IFineDialogState {
    error: Error | null;
    isFinesLoading: boolean;
    fines: Dict<any> [];
    chosenItem: Dict<any> | null;
    selectTitles: IOptionInfo[];
}

export class FineDialogModal extends React.Component<IFineDialogProps, IFineDialogState> {
    state: IFineDialogState = {
        error: null,
        isFinesLoading: false,
        fines: [],
        chosenItem: null,
        selectTitles: [],
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getFines();
    }

    getFines() {
        this.setState({ isFinesLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_DATA, { queryParams: { user_id: this.props.userId } })
                .then(response => {
                    const selectTitles: IOptionInfo[] = this.getSelectTitles(response || []);

                    this.setState({
                        isFinesLoading: false,
                        fines: response || [],
                        chosenItem: response?.[0],
                        selectTitles,
                    });
                })
                .catch(error => {
                    this.setState({ error, isFinesLoading: false });
                });
        });
    }

    getSelectTitles(data) {
        return data?.map(el => {
            const koap = el?.article_koap.length > KOAP_MAX_LENGTH
                ? el?.article_koap.substring(0, KOAP_MAX_LENGTH) + '...'
                : el?.article_koap;
            const description = `${koap} | ${el?.car?.model?.name}`;
            const value = el?.id;
            const text = `${moment(el?.violation_time * ONE_SECOND)
                .format('DD.MM.YYYY HH:MM')} | ${rubs(el?.sum_to_pay)}`;

            return {
                description,
                value,
                text,
            };
        });
    }

    onSelectChange(fine_id) {
        const chosenItem: Dict<any> | null = this.state.fines.find(fine => fine?.id === fine_id) || null;
        this.setState({ chosenItem });
    }

    send() {
        this.props.sendMessage && this.state.chosenItem && this.props.sendMessage(this.state.chosenItem?.id);
        this.props.onClose();
    }

    render() {
        const { error, selectTitles, chosenItem } = this.state;
        const initialValue = selectTitles.find(t => t.value === chosenItem?.id);

        return <Window onClose={this.props.onClose.bind(this)}
                       title={'Отправка информации о штрафе'}
                       className={style.modal}
                       error={error}>

            <Select onSelect={this.onSelectChange.bind(this)}
                    options={selectTitles}
                    initialValues={initialValue?.value ? [initialValue?.value] : []}
                    className={style.select}
                    placeholder={'Штраф'}/>

            {chosenItem
                ? <>
                    <ShortFineInfoBlock fineInfo={chosenItem}/>

                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={this.props.onClose.bind(this)}/>
                        <Button onClick={this.send.bind(this)}>Отправить</Button>
                    </div>
                </>
                : null
            }

        </Window>;
    }
}
