import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../types';
import { COINS_PER_RUBLE, EMPTY_DATA, ONE_SECOND } from '../../constants';
import { Button } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Cross } from '../../ui/Cross';
import DatePicker from '../../ui/DatePicker';
import { FormatDateInString } from '../../ui/FormatDate';
import { Window } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import Select, { IOptionInfo } from '../../ui/Select';
import { Request2 } from '../../utils/request';
import Spin from '../Spin';
import * as styles from './index.css';
import { FINES_DIALOG_REQUESTS, REQUESTS } from './request';

interface IFinesDownPaymentsDialogProps {
    onClose: () => void;
    userId: string;
    finesTags: string;
}

interface IFinesDownPaymentsDialogState {
    [x: number]: any;
    currentTag: string[];
    error: Error | null;
    isLoading: boolean;
    st: string;
    tagsOptions: IOptionInfo[];
    addTags: Dict<number>[];
    divideByDate: boolean;
    divideBySum: boolean;
    isUploading: boolean;
    errorUpload: Dict<string>[];
    divideEqualAmount: number;
    divideEqualPeriodQuantity: number;
    divideEqualPeriod: ('y' | 'M' | 'd')[];
    divideEqualStart: number;
    divideEqualMonthAmount: number;
}

const timeKeys = [
    { text: 'День', value: 'd' },
    { text: 'Неделя', value: 'w' },
    { text: 'Месяц', value: 'M' },
];

const FIXED_SUM_TAG = 'fixed_sum_tag';

export class FinesDownPaymentsDialog extends
    React.Component<IFinesDownPaymentsDialogProps, IFinesDownPaymentsDialogState> {
    state: IFinesDownPaymentsDialogState = {
        currentTag: [],
        error: null,
        isLoading: false,
        st: '',
        tagsOptions: [],
        addTags: [{ date: 0, amount: 0 }],
        divideByDate: false,
        divideBySum: false,
        isUploading: false,
        errorUpload: [],
        divideEqualAmount: 0,
        divideEqualPeriodQuantity: 1,
        divideEqualPeriod: ['M'],
        divideEqualStart: Date.now(),
        divideEqualMonthAmount: 0,
    };

    request = new Request2({ requestConfigs: FINES_DIALOG_REQUESTS });

    getTags() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.TAG_DESCRIPTIONS)
                .then((response) => {
                    const { records } = response;
                    const { finesTags } = this.props;
                    const tagsOptions: IOptionInfo[] = [];
                    records.forEach((tag) => {
                        const tagName = tag && tag.name || EMPTY_DATA;
                        const tagType = tag && tag.type;
                        const tagDisplayName = tag && tag.display_name || EMPTY_DATA;
                        const finesTagsArray = (finesTags && finesTags.split(',')) || [];

                        if (tagName && finesTagsArray.includes(tagName) && tagType !== FIXED_SUM_TAG) {
                            tagsOptions.push({ text: tagDisplayName, value: tagName });
                        }
                    });

                    this.setState({
                        tagsOptions,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    addTag() {
        this.setState((prev) => ({
            addTags: [...prev.addTags, { date: 0, amount: 0 }],
        }));
    }

    onDeleteTag(index) {
        this.setState((prev) => {
            const { addTags } = prev;
            addTags.splice(index, 1);

            return {
                addTags,
            };
        });
    }

    onSelect(key, value) {
        this.setState({
            [key]: value ? [value] : [],
        });
    }

    onChange(key, value) {
        const numberKeys = [
            'divideEqualAmount',
            'divideEqualPeriodQuantity',
            'divideEqualStart',
            'divideEqualMonthAmount',
        ];

        if (numberKeys.includes(key)) {
            value = +value;
        }

        this.setState({
            [key]: value,
        });
    }

    onChangeCheckbox(key, value) {
        if (key === 'divideBySum') {
            this.setState({
                divideBySum: value,
                divideByDate: false,
            });
        }

        if (key === 'divideByDate') {
            this.setState({
                divideByDate: value,
                divideBySum: false,
            });
        }
    }

    onChangeTag(index, key, value) {
        this.setState((prev) => {
            const { addTags } = prev;
            addTags.splice(index, 1, {
                ...addTags[index],
                [key]: +value,
            });

            return {
                addTags,
            };
        });
    }

    divideEqual(quantity, divideEqualStart, divideEqualAmount, divideEqualPeriod?) {
        let currentDate = moment(new Date(divideEqualStart));
        const addTags: any[] = [];

        const remainderAmount = divideEqualAmount % quantity;
        const amount = divideEqualPeriod
            ? Math.floor(divideEqualAmount / quantity)
            : quantity;
        const payments_count = Math.floor(divideEqualAmount / amount);

        for (let i = 0; i < payments_count; i++) {
            addTags.push({
                date: +currentDate,
                amount: i + 1 === payments_count ? amount + remainderAmount : amount,
            });

            currentDate = currentDate.add(1, divideEqualPeriod ? divideEqualPeriod[0] : 'M' );
        }

        this.setState({
            addTags,
        });
    }

    uploadData() {
        this.setState({
            isUploading: true,
            errorUpload: [],
        }, () => {
            const { currentTag, st, addTags } = this.state;
            const { userId, onClose } = this.props;

            addTags.forEach((tag) => {
                this.request.exec(REQUESTS.ATTACH_USER_TAGS, {
                    body: {
                        tag: currentTag[0],
                        links: [{ uri: st, type: 'st' }],
                        priority: 0,
                        active_since: Math.round(tag.date / ONE_SECOND),
                        amount: tag.amount * COINS_PER_RUBLE,
                        object_id: userId,
                    },
                })
                    .then(() => {
                        this.setState({
                            isUploading: false,
                        });
                        onClose();
                    })
                    .catch((error) => {
                        this.setState((prev) => ({
                            errorUpload: [
                                ...prev.errorUpload,
                                {
                                    main: `${FormatDateInString({ value: tag.date })}`
                                        + ` — ${tag.amount}₽: списание не создано`,
                                    hint: `(код ошибки: ${error.data.error_details?.http_code || 0},`
                                        + ` сообщение: ${error.data.error_details?.ui_message || EMPTY_DATA})`,
                                },
                            ],
                            isUploading: false,
                        }));
                    });
            });
        });
    }

    componentDidMount() {
        this.getTags();
    }

    render() {
        const {
            currentTag,
            st,
            tagsOptions,
            addTags,
            divideByDate,
            divideBySum,
            isUploading,
            isLoading,
            error,
            errorUpload,
            divideEqualAmount,
            divideEqualPeriod,
            divideEqualPeriodQuantity,
            divideEqualStart,
            divideEqualMonthAmount,
        } = this.state;

        const { onClose } = this.props;

        const amount = addTags.reduce((amount, tag) => {
            return amount + tag.amount;
        }, 0);

        const disabled = !(currentTag && currentTag.length) || !addTags.every((tag) => {
            return !!tag.amount && !!tag.date;
        });

        return (
            <Window className={styles.modal}
                    error={error}
                    onClose={onClose.bind(this)}
                    title={'Добавить рассрочку штрафа'}>
                {isLoading ? <Spin /> :
                    <>
                        <Select options={tagsOptions}
                                onSelect={this.onSelect.bind(this, 'currentTag')}
                                initialValues={currentTag}
                                placeholder={'Тег для списания штрафа'}/>
                        <Input value={st}
                               onChange={this.onChange.bind(this, 'st')}
                               placeholder={'Ссылка на st.yandex-team.ru'}/>
                        <div className={styles.checkbox}>
                            <span>Разделить списания по периодичности</span>
                            <Checkbox checked={divideByDate}
                                      onChange={this.onChangeCheckbox.bind(this, 'divideByDate')}/>
                        </div>
                        {divideByDate && (
                            <div className={styles.divideEqual}>
                                <Input value={divideEqualAmount}
                                       onChange={this.onChange.bind(this, 'divideEqualAmount')}
                                       placeholder={'Общ. сумма'}/>
                                <p>Регулярность списания:</p>
                                <div className={styles.divideEqualPeriod}>
                                    <Input className={styles.input}
                                           value={divideEqualPeriodQuantity}
                                           onChange={this.onChange.bind(this, 'divideEqualPeriodQuantity')}
                                           placeholder={'Количество'}/>
                                    <Select options={timeKeys}
                                            onSelect={this.onSelect.bind(this, 'divideEqualPeriod')}
                                            initialValues={divideEqualPeriod}
                                            placeholder={'Временной период'}/>
                                </div>
                                <div className={styles.divideEqualPeriod}>
                                    <DatePicker className={styles.input}
                                                value={divideEqualStart}
                                                onChange={this.onChange.bind(this, 'divideEqualStart')}
                                                placeholder={'Дата первого платежа'}/>
                                    <Button basic
                                            onClick={this.divideEqual
                                                .bind(
                                                    this,
                                                    divideEqualPeriodQuantity,
                                                    divideEqualStart,
                                                    divideEqualAmount,
                                                    divideEqualPeriod,
                                                )}>
                                        Применить
                                    </Button>
                                </div>
                            </div>
                        )}
                        <div className={styles.checkbox}>
                            <span>Разделить списания на ежемесячные платежи</span>
                            <Checkbox checked={divideBySum}
                                      onChange={this.onChangeCheckbox.bind(this, 'divideBySum')}/>
                        </div>
                        {divideBySum && (
                            <div className={styles.divideEqual}>
                                <div className={styles.divideEqualSum}>
                                    <Input className={styles.input}
                                           value={divideEqualAmount}
                                           onChange={this.onChange.bind(this, 'divideEqualAmount')}
                                           placeholder={'Общ. сумма'}/>
                                    <Input className={styles.input}
                                           value={divideEqualMonthAmount}
                                           onChange={this.onChange.bind(this, 'divideEqualMonthAmount')}
                                           placeholder={'Сумма одного платежа'}/>
                                </div>
                                <div className={styles.divideEqualPeriod}>
                                    <DatePicker className={styles.input}
                                                value={divideEqualStart}
                                                onChange={this.onChange.bind(this, 'divideEqualStart')}
                                                placeholder={'Дата первого платежа'}/>
                                    <Button basic
                                            onClick={this.divideEqual.bind(this, divideEqualMonthAmount,
                                                divideEqualStart, divideEqualAmount, null)}>Применить</Button>
                                </div>
                            </div>
                        )}
                        {addTags.length
                            ? addTags.map((tag, index) => {
                                return (
                                    <AddTag date={tag.date}
                                            amount={tag.amount}
                                            onChange={this.onChangeTag.bind(this, index)}
                                            onDelete={this.onDeleteTag.bind(this, index)}
                                            index={index}
                                            key={index}/>
                                );
                            })
                            : <div className={styles.noData}>Нет списаний</div>
                        }
                        <Button basic
                                className={styles.addTagButton}
                                onClick={this.addTag.bind(this)}>Добавить списание</Button>
                        <p className={styles.finalSum}>Итоговая сумма: {amount}₽</p>
                        {errorUpload.length
                            ?
                            <CustomError error={errorUpload}/>
                            : undefined
                        }
                        <div className={styles.buttonContainer}>
                            <Button onClick={onClose.bind(this)}
                                    basic>Отмена</Button>
                            <Button disabled={disabled}
                                    isLoading={isUploading}
                                    onClick={this.uploadData.bind(this)}>Отправить</Button>
                        </div>
                    </>
                }
            </Window>
        );
    }
}

const AddTag = (props: {date: number; amount: number; onChange: () => void; onDelete: () => void; index: number}) => {
    const { onChange, date, amount, onDelete, index } = props;

    return (
        <div className={styles.addTag}>
            <span className={styles.addTagHint}>
                Списание {index + 1}
            </span>
            <div className={styles.addTagContent}>
                <Cross className={styles.delete}
                       onClick={onDelete.bind(null)}/>
                <DatePicker className={styles.input}
                            value={date}
                            onChange={onChange.bind(null, 'date')}
                            placeholder={'Дата активации'}/>
                <Input value={amount}
                       onChange={onChange.bind(null, 'amount')}
                       placeholder={'Сумма списания (руб.)'}/>
            </div>
        </div>
    );
};

const CustomError = (props: {error: Dict<string>[]}) => {
    const { error } = props;

    return (
        <div className={styles.errorUploadBox}>
            {error.map((error) => {
                return (
                    <>
                        <p className={styles.errorUpload}>
                            {error.main}
                        </p>
                        <p className={`${styles.errorUpload} ${styles.errorUploadHint}`}>
                            {error.hint}
                        </p>
                    </>
                );
            })}
        </div>
    );
};
