import * as React from 'react';
import ReactJson from 'react-json-view';

import { Dict } from '../../../../types';
import { Window } from '../../../ui/FullModal';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as styleTable from '../../../ui/Table/index.css';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { deepCopy, isObject } from '../../../utils/utils';
import { controlType, ISchemaItem } from '../types';
import * as style from './index.css';

enum IDifferenceKinds {
    NEW = 'N',
    DELETE = 'D',
    EDIT = 'E',
    ARRAY = 'A',
}

const IDifferenceKindsMap = {
    [IDifferenceKinds.NEW]: 'Добавление',
    [IDifferenceKinds.DELETE]: 'Удаление',
    [IDifferenceKinds.EDIT]: 'Редактирование',
    [IDifferenceKinds.ARRAY]: 'Изменение в массиве',
};
const PATH_SEPARATOR = '.';

export interface IDifference {
    kind: IDifferenceKinds;
    path: string[];
    lhs?: Dict<any> ;
    rhs?: Dict<any> ;
    index?: number;
    item?: Dict<any>;
}

interface IDifferenceFormatted {
    kind: IDifferenceKinds;
    path: string[];
    pathSchema: string[];
    lhs?: Dict<any> ;
    rhs?: Dict<any> ;
    index?: number;
    item?: Dict<any>;
}

interface IFormChangesModalProps {
    onClose: () => {};
    diffData: IDifference[] | null;
    schema: Dict<ISchemaItem>;
}

interface IFormChangesModalState {
    hash: Dict<string[]>;
    diffData: IDifferenceFormatted[];
}

export default class FormChangesModal extends React.Component<IFormChangesModalProps, IFormChangesModalState> {
    state: IFormChangesModalState = {
        hash: {},
        diffData: [],
    };

    constructor(props: IFormChangesModalProps) {
        super(props);
        const { hash, diffData } = this.formatDiffData();
        this.state = { hash, diffData };
    }

    componentDidUpdate(prevProps: Readonly<IFormChangesModalProps>): void {
        if (!isObjectEqual(this.props.diffData, prevProps.diffData)) {
            const { hash, diffData } = this.formatDiffData();
            this.setState({ hash, diffData });
        }
    }

    formatDiffData() {
        const diffDataProps = this.props.diffData ?? [];
        const diffData = this.formatDiffPath(diffDataProps);
        const hash = this.getDisplayNames(diffData);

        return { hash, diffData };
    }

    formatDiffPath(diffData: IDifference[]): IDifferenceFormatted[] {
        return diffData.map(diffDataItem => {
            const formattedDiffItem: IDifferenceFormatted = deepCopy(diffDataItem);

            const pathSchema = [...diffDataItem.path];
            if (diffDataItem.kind === IDifferenceKinds.ARRAY) {
                const index = diffDataItem.index as number;
                pathSchema.push(index.toString());
            }

            formattedDiffItem.pathSchema = pathSchema;

            return formattedDiffItem;
        })
            .sort((diffDataItem1, diffDataItem2) => {
                return diffDataItem1.path.join(PATH_SEPARATOR)
                    .localeCompare(diffDataItem2.path.join(PATH_SEPARATOR));
            });
    }

    getDisplayNames(diffData: IDifferenceFormatted[]) {
        const { schema } = this.props;
        const hash = {};

        diffData
            .forEach(diffDataItem => {
                const path = diffDataItem.pathSchema ?? [];
                const prevPathString: string[] = [];

                for (let pathIndex = 1; pathIndex < path.length + 1; pathIndex++) {
                    const currentPath = path.slice(0, pathIndex);
                    const currentPathString = currentPath.join(PATH_SEPARATOR);

                    const key = currentPath.slice(-1)[0];
                    const parents = currentPath.length > 1 ? currentPath.slice(0, currentPath.length - 1) : [];

                    let schemaItem = parents && parents.length
                        ? parents.reduce((resultValue: Dict<any> , parentItem: string) => {
                            const parent = /^\d+$/ig.test(parentItem) ? 'array_type' : parentItem;

                            return resultValue[parent] && resultValue[parent].type
                            && resultValue[parent].type === controlType.structure
                                ? resultValue[parent].structure
                                : resultValue?.[parent];
                        }, schema)
                        : schema;
                    schemaItem = /^\d+$/ig.test(key)
                        ? schemaItem.type === controlType.variants
                            ? schemaItem
                            : schemaItem.array_type
                        : schemaItem[key];

                    hash[currentPathString] = [...prevPathString, schemaItem?.display_name ?? key];
                    prevPathString.push(schemaItem?.display_name ?? key);
                }
            });

        return hash;
    }

    render() {
        const { onClose } = this.props;
        const { diffData, hash } = this.state;

        return <Window onClose={onClose?.bind(this)} title={'Изменения в форме'}>
            <div className={style.form_changes_modal}>
                <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Тип изменения</th>
                            <th>Путь</th>
                            <th>Старое значение</th>
                            <th>Новое значение</th>
                        </tr>
                    </thead>
                    <tbody>
                        {diffData.map((diffDataItem, index) => {
                            return <FormChangesRow key={diffDataItem.pathSchema.join(PATH_SEPARATOR)}
                                                   diffDataItem={diffDataItem}
                                                   index={index}
                                                   hash={hash}/>;
                        })}
                    </tbody>
                </table>
            </div>

        </Window>;
    }
}

const FormChangesRow = React.memo((props: any) => {
    const getPathString = (path: string[]) => {
        const { hash } = props;
        const displayPath: string[] = [];

        for (let pathIndex = 0; pathIndex < path.length; pathIndex++) {
            const currentPath = path.slice(0, pathIndex + 1);

            const currentPosition = path[pathIndex];
            const hashArrPath = hash[currentPath.join(PATH_SEPARATOR)];
            let currentPathString = hashArrPath?.[hashArrPath.length - 1] ?? currentPath.join(PATH_SEPARATOR);

            if (!isNaN(+currentPosition)) {
                currentPathString += `[${currentPosition}]`;
            }

            displayPath.push(currentPathString);
        }

        return displayPath.join(' => ');
    };

    const getValue = (key: string) => {
        const value = diffDataItem.hasOwnProperty(key)
            ? diffDataItem[key]
            : diffDataItem?.item?.hasOwnProperty(key)
                ? diffDataItem.item?.[key]
                : '';

        return isObject(value) ? <ReactJson src={value} enableClipboard={false}/> : value?.toString();
    };

    const { diffDataItem, index } = props;

    const { kind, pathSchema } = diffDataItem;

    const pathString = getPathString(pathSchema);

    let status = LabelStatus.DEFAULT;
    switch (kind) {
    case IDifferenceKinds.NEW:
        status = LabelStatus.POSITIVE;
        break;
    case IDifferenceKinds.DELETE:
        status = LabelStatus.NEGATIVE;
        break;
    case IDifferenceKinds.EDIT:
        status = LabelStatus.INFO;
        break;
    case IDifferenceKinds.ARRAY:
        status = LabelStatus.INFO;

        break;
    }

    return <tr>
        <td>{index + 1}</td>
        <td>
            <TLabel status={status} text={IDifferenceKindsMap[kind] ?? kind}/>
        </td>
        <td>{pathString}</td>
        <td>{getValue('lhs')}</td>
        <td>{getValue('rhs')}</td>
    </tr>;
});
