import diff from 'deep-diff';
import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../types';
import { COINS_LENGTH, COINS_PER_RUBLE, ONE_SECOND } from '../../constants';
import { UIStatusTypes } from '../../ui';
import { Button } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Cross } from '../../ui/Cross';
import DatePicker from '../../ui/DatePicker';
import FileInput, { FileInputST } from '../../ui/FileInput';
import IDSelect from '../../ui/IDSelect/component';
import { GLOBAL_SEARCH_OBJECTS } from '../../ui/IDSelect/constants';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import Select, { IOptionInfo } from '../../ui/Select';
import SelectConstants from '../../ui/Select/SelectConstants';
import { SeparatorLine } from '../../ui/SeparatorLine';
import { Tabs } from '../../ui/Tabs';
import TextArea from '../../ui/TextArea';
import { chunksErrorHandling } from '../../utils/chunksErrorHandling';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { deepCopy, isDateValid } from '../../utils/utils';
import { JSON_TAB_SIZE, STRING_VECTOR_SPLITTER } from './constants';
import FormChangesModal, { IDifference } from './FormChangesModal';
import { FormConstructorSchemaWorker } from './FormConstructorSchemaWorker';
import { ArrayAction, DEFAULT_TAB_NAME, FormConstructorTabsWorker, IFormTabs } from './FormConstructorTabsWorker';
import { FormConstructorValuesWorker, PATH_JOINER } from './FormConstructorValuesWorker';
import FormStructure from './FormStructure';
import * as style from './index.css';
import TableConfigModal from './TableConfigModal';
import { controlType, IFormConstructorProps, ISchemaItem, SchemaItemVisual } from './types';
import { isValueExist } from './utils/utils';

interface IFormConstructorState {
    schema: Dict<ISchemaItem>;
    initValues: Dict<any>;
    values: Dict<any>;
    validValues: Dict<Dict<boolean | null> | boolean | null>;
    tabs: Dict<IFormTabs>;
    isTableModalOpen: boolean;
    isOpenDiffModal: boolean;
    currentTable: { schema: ISchemaItem; parents: string[] } | null;
    formChanges: IDifference[] | null;
    excludedStructures: Dict<any>; //object with paths, joined by dot
    excludedStructuresInitial: Dict<any>;
}

const ButtonUploadST = React.lazy(() => chunksErrorHandling(() => import('./ButtonUploadST')));

export class FormConstructorInputs extends React.Component<IFormConstructorProps, IFormConstructorState> {
    state: IFormConstructorState = {
        schema: {},
        values: {},
        initValues: {},
        validValues: {},
        tabs: {} as Dict<IFormTabs>,
        isTableModalOpen: false,
        isOpenDiffModal: false,
        currentTable: null,
        formChanges: null,
        excludedStructures: {},
        excludedStructuresInitial: {},
    };

    constructor(props: IFormConstructorProps) {
        super(props);

        const { schema } = this.props;
        const values = FormConstructorSchemaWorker.constructValuesBySchema(schema);
        const tabs = FormConstructorTabsWorker.constructTabsBySchema({ schemaInit: schema, values });

        this.state = {
            schema: schema ?? {},
            initValues: deepCopy(values),
            values,
            validValues: values,
            tabs: tabs,
            isTableModalOpen: false,
            isOpenDiffModal: false,
            currentTable: null,
            formChanges: null,
            excludedStructures: {},
            excludedStructuresInitial: {},
        };
    }

    componentDidMount(): void {
        const { initialData } = this.props;
        const { values, schema } = this.state;

        if (initialData) {
            const initializedValues = this.setInitialData(values, initialData, []);
            const tabs = FormConstructorTabsWorker.constructTabsBySchema({
                schemaInit: schema,
                values: initializedValues,
            });

            this.setState({
                values: initializedValues,
                tabs,
                initValues: deepCopy(initializedValues),
            }, () => {
                this.onFormValuesChange();
            });
        }

        this.onFormValuesChange();
    }

    componentDidUpdate(prevProps: Readonly<IFormConstructorProps>, prevState: Readonly<IFormConstructorState>): void {
        const { schema = {}, initialData } = this.props;
        const { schema: prevSchema = {}, initialData: prevInitialData } = prevProps;
        let { values } = this.state;

        if (!isObjectEqual(initialData, prevInitialData)
            || !isObjectEqual(schema, prevSchema)) {

            if (!isObjectEqual(schema, prevSchema)) {

                this.setState({ schema }, () => {
                    const { schema } = this.state;

                    values = FormConstructorSchemaWorker.constructValuesBySchema(schema);
                    const tabs = FormConstructorTabsWorker.constructTabsBySchema({ schemaInit: schema, values });
                    values = initialData ? this.setInitialData(values, initialData, []) : values;

                    this.setState({ values, initValues: deepCopy(values), tabs }, () => {
                        this.onFormValuesChange();
                    });
                });

            } else if (!isObjectEqual(initialData, prevInitialData)) {
                values = FormConstructorSchemaWorker.constructValuesBySchema(schema);
                values = initialData ? this.setInitialData(values, initialData, []) : values;
                const tabs = FormConstructorTabsWorker.constructTabsBySchema({ schemaInit: schema, values });

                this.setState({ values, initValues: deepCopy(values), tabs }, () => {
                    this.onFormValuesChange();
                });
            }
        }
    }

    setInitialData(values: Dict<any>, initialData: Dict<any>, parents) {
        const { schema } = this.state;

        //If values or initialData are not objects it is no sense to go further
        if ((typeof values !== 'object' || values === null)
            || (typeof initialData !== 'object' || initialData === null)) {
            return values;
        }

        return Object.entries(values).reduce((result: Dict<any>, entry: [string, any]) => {
            const [key, value] = entry;

            const schemaItem = FormConstructorSchemaWorker.getSchemaItem({ values, key, parents, schema });

            //If initialData exist and array is empty, exclude it
            if (!schemaItem?.required && schemaItem?.type && schemaItem.type === controlType.array_types
                && (initialData?.[key] === null || initialData?.[key] === undefined)) {
                const excludedStructures = this.state.excludedStructures;
                excludedStructures[[...parents, key].join(PATH_JOINER)] = true;
                this.setState({ excludedStructuresInitial: excludedStructures, excludedStructures }, () => {
                    this.onFormValuesChange();
                });
            }

            switch (schemaItem?.type) {
            case controlType.string:
                result[key] = initialData[key] !== undefined && initialData[key] !== null
                    ? schemaItem?.visual === SchemaItemVisual.COLOR
                        ? initialData[key]?.includes('#')
                            ? initialData[key]
                            : `#${initialData[key]}`
                        : initialData[key]
                    : value;
                break;
            case controlType.json:
                result[key] = initialData[key] !== undefined && initialData[key] !== null
                    ? JSON.stringify(initialData[key], null, JSON_TAB_SIZE)
                    : value;
                break;
            case controlType.numeric:
                //RUBS and MONEY are the same entities with different names, need to clear one of them
                if ((schemaItem.visual === SchemaItemVisual.TIMESTAMP
                        || schemaItem.visual === SchemaItemVisual.RUBS
                        || schemaItem.visual === SchemaItemVisual.MONEY)) {

                    if (schemaItem.visual === SchemaItemVisual.TIMESTAMP) {

                        if (isDateValid(initialData[key])) {
                            result[key] = initialData[key];
                        } else {
                            console.warn('Initial timestamp value is not valid');
                            result[key] = value;
                        }
                    }

                    if (schemaItem.visual === SchemaItemVisual.RUBS
                            || schemaItem.visual === SchemaItemVisual.MONEY) {

                        result[key] = typeof initialData[key] === 'number'
                            ? +((initialData[key] / COINS_PER_RUBLE).toFixed(COINS_LENGTH))
                            : value;
                    }

                } else {
                    result[key] = initialData[key] !== undefined && initialData[key] !== null
                        ? initialData[key]
                        : value;
                }

                break;
            case controlType.structure:
                const isStructureRequired = schemaItem?.type === controlType.structure && schemaItem?.required;

                //If initialData exist and structure is empty, exclude it
                if (!isStructureRequired && (initialData[key] === null || initialData[key] === undefined)) {
                    const excludedStructures = this.state.excludedStructures;
                    excludedStructures[[...parents, key].join(PATH_JOINER)] = true;
                    this.setState({ excludedStructuresInitial: excludedStructures, excludedStructures }, () => {
                        this.onFormValuesChange();
                    });
                }

                result[key] = this.setInitialData(value,
                    initialData && initialData[key]
                        ? initialData[key]
                        : {},
                    [...parents, key]);
                break;
            case controlType.variable:
                const { variants_fields, default_fields } = schemaItem;
                const variableValue = values?.[key];

                const controlFieldKey = Object.keys(schemaItem?.control_field ?? {})[0];
                const controlFieldValue = variableValue[controlFieldKey] ?? null;
                const controlFieldValueInitialData = initialData[controlFieldKey] ?? null;

                if (initialData.hasOwnProperty(controlFieldKey)
                        && variableValue[controlFieldKey] !== controlFieldValueInitialData) {
                    //if control_field was changed, clear all variants/defaults values
                    result[key] = { [controlFieldKey]: controlFieldValueInitialData };
                } else {
                    result[key] = { [controlFieldKey]: controlFieldValue };
                }

                if (controlFieldValueInitialData) {
                    const variantsFieldValue = controlFieldValueInitialData
                        ? variants_fields?.[controlFieldValueInitialData]
                            ? FormConstructorSchemaWorker
                                .constructValuesBySchema(variants_fields[controlFieldValueInitialData] ?? {})
                            : default_fields
                                ? FormConstructorSchemaWorker.constructValuesBySchema(default_fields ?? {})
                                : {}
                        : {};

                    Object.entries(variantsFieldValue).forEach((variantEntry: [string, Dict<any>]) => {
                        const [variantKey, variantValue] = variantEntry;
                        result[key][variantKey] = variantValue;
                    });
                }

                //If variants or default fields is changed, change it in result value
                const controlFieldResultValue = result[key][controlFieldKey];
                const variantsFields = controlFieldResultValue
                    ? variants_fields?.[controlFieldResultValue]
                        ? variants_fields?.[controlFieldResultValue]
                        : default_fields ?? {}
                    : {};

                Object.keys(variantsFields).forEach(variantsField => {

                    if (initialData[variantsField]) {
                        result[key][variantsField] = initialData[variantsField];
                    } else {
                        result[key][variantsField] = result[key].hasOwnProperty(variantsField)
                            ? result[key][variantsField]
                            : values[key][variantsField];
                    }
                });
                break;
            case controlType.string_vector:
                //Deprecated type of values, must use array of strings instead
                result[key] = initialData[key] && initialData[key].join
                    ? initialData[key].join(STRING_VECTOR_SPLITTER)
                    : value[key];
                break;
            default:
                result[key] = initialData[key] !== undefined && initialData[key] !== null
                    ? initialData[key]
                    : value;
                break;
            }

            return result;
        }, {});
    }

    onFormValuesChange() {
        const { rawResult } = this.props;
        const { values: valuesState, schema, excludedStructures } = this.state;

        const values = deepCopy(valuesState);

        const formattedValues = FormConstructorValuesWorker.formatValues({
            valuesFull: values,
            schemaFull: schema,
            parents: [],
            rawResult,
            excludedStructures,
        });

        const validValues = FormConstructorValuesWorker.validateValues({ valuesFull: values, schemaFull: schema });

        const formChanges = this.getFormChanges(formattedValues);
        const isFormChanged = !!formChanges;

        this.setState({ validValues, formChanges }, () => {
            const { onChange } = this.props;
            const { validValues } = this.state;

            const isFormValid = FormConstructorValuesWorker.isValuesValid(validValues);
            onChange && onChange(formattedValues, isFormValid, isFormChanged);
        });
    }

    getFormChanges(resultValues: Dict<any>): IDifference[] | null {
        const { rawResult } = this.props;
        const { initValues, schema, excludedStructures, excludedStructuresInitial } = this.state;

        const formattedInitValues = FormConstructorValuesWorker.formatValues({
            valuesFull: initValues,
            schemaFull: schema,
            parents: [],
            rawResult,
            excludedStructures,
        });
        let resultDIff: IDifference[] = diff.diff(formattedInitValues, resultValues) as IDifference[];
        resultDIff = resultDIff ?? [];

        let resultDiffExcluded: IDifference[] = diff
            .diff(excludedStructuresInitial, excludedStructures) as IDifference[];
        resultDiffExcluded = (resultDiffExcluded ?? []).map(resultDiffExcludedItem => {
            const copy = deepCopy(resultDiffExcludedItem);
            copy.kind = 'E';
            copy.lhs = !!resultDiffExcludedItem.lhs ? 'Структура исключена' : 'Структура включена';
            copy.rhs = !!resultDiffExcludedItem.rhs ? 'Структура исключена' : 'Структура включена';

            return copy;
        });

        return resultDIff.length || resultDiffExcluded.length ? [...resultDIff, ...resultDiffExcluded] : null;
    }

    onIncludeStatusChange(fullPath: string[]) {
        const excludedStructures = deepCopy(this.state.excludedStructures);
        const fullPathString = fullPath.join(PATH_JOINER);

        if (Object.keys(excludedStructures).some(excludedStructure => excludedStructure === fullPathString)) {
            delete excludedStructures[fullPathString];
        } else {
            excludedStructures[fullPathString] = true;
        }

        this.setState({ excludedStructures }, () => {
            this.onFormValuesChange();
        });
    }

    buildControls(schema: ISchemaItem, parents: string[] = [], isVariableControlField = false) {
        const { tabs: tabsState, values, validValues } = this.state;

        const tabsInfo = FormConstructorTabsWorker.getTabsInfoByPath({
            tabsInfo: tabsState,
            parents: isVariableControlField ? parents.slice(0, parents.length - 1) : parents,
        });

        const tabsItems = tabsInfo?.tabs?.map(tab => {
            return { name: `${tab.name} (${tab.countItems})`, link: tab.name };
        });

        const currentTab = tabsInfo?.currentTab;

        const controls = Object.entries(schema)
            .sort((entry1: [string, ISchemaItem], entry2: [string, ISchemaItem]) => {
                const schema1 = entry1[1];
                const schema2 = entry2[1];

                const order1 = !!schema1.deprecated
                    ? Infinity
                    : schema1.order ?? Infinity;
                const order2 = !!schema2.deprecated
                    ? Infinity
                    : schema2.order ?? Infinity;

                return order1 - order2;
            })
            .map((entry: [string, ISchemaItem]) => {
                const [element, schema] = entry;

                const currentItemTab = schema && schema.tab_name || DEFAULT_TAB_NAME;

                if (!isVariableControlField
                    && currentTab !== null
                    && currentTab !== undefined
                    && currentItemTab !== currentTab) {
                    return null;
                }

                let control: any;

                const value = parents && parents.length
                    ? parents.reduce((resultValue: any, parentItem: string) => {
                        return resultValue?.[parentItem] ?? {};
                    }, values)
                    : values;

                let placeholder = element;

                if (schema.display_name) {
                    placeholder = schema.display_name;
                } else if (!isNaN(+element)) {
                    placeholder = parents[parents.length - 1] && `${parents[parents.length - 1]} [${+element + 1}]`;
                }

                const disable = !!schema.read_only;
                const deprecated = !!schema.deprecated;
                const required = !!schema.required;
                const maxLength = schema.maxLength;
                if (required) {
                    placeholder += ' (Обязательное поле)';
                }

                const collapsed = !!schema.collapsed;

                let status = parents && parents.length
                    ? parents.reduce((resultValue: Dict<any>, parentItem: string) => {
                        return /^\d+$/ig.test(parentItem) ? resultValue : resultValue && resultValue[parentItem] || {};
                    }, validValues)
                    : validValues;

                status = status && status[element] || true;

                let statusObj: any = null;

                const negativeStatusesTexts: string[] = [];
                const warningStatusesTexts: string[] = [];

                if (status === false) {
                    negativeStatusesTexts.push('Неверный формат данных');
                }

                if (disable) {
                    warningStatusesTexts.push('Только для чтения');
                }

                if (deprecated) {
                    warningStatusesTexts.push('Deprecated');
                }

                if (schema.visual === SchemaItemVisual.RUBS || schema.visual === SchemaItemVisual.MONEY) {
                    negativeStatusesTexts.push('Внимание! Вводите сумму в рублях');
                }

                if (schema?.type === controlType.numeric && (value?.[element] && isNaN(value?.[element]))) {
                    negativeStatusesTexts.push('Неверное числовое значение');
                }

                const min = schema?.min ?? -Infinity;
                const max = schema?.max ?? Infinity;

                if (schema?.type === controlType.numeric
                    && (value?.[element] > max || value?.[element] < min)) {
                    negativeStatusesTexts.push(`[min:max]: [${min} : ${max}]`);
                }

                if ((schema.type === controlType.string
                    || schema.type === controlType.text
                    || schema.type === controlType.json) && value?.[element]) {

                    if (value?.[element]?.[0] === ' '
                        && value?.[element]?.substr?.(value?.[element]?.length - 1) === ' ') {
                        if (value?.[element]?.length === 1) {
                            warningStatusesTexts.push('Строка состоит из пробела');
                        } else {
                            warningStatusesTexts.push('Пробелы в начале и конце строки');
                        }
                    } else if (value?.[element]?.substr?.(value?.[element]?.length - 1) === ' ') {
                        warningStatusesTexts.push('Пробелы в конце строки');
                    } else if (value?.[element]?.[0] === ' ') {
                        warningStatusesTexts.push('Пробелы в начале строки');
                    }
                }

                if (negativeStatusesTexts.length) {
                    statusObj = {
                        type: UIStatusTypes.negative,
                        text: negativeStatusesTexts.join(', '),
                    };
                } else if (warningStatusesTexts.length) {
                    statusObj = {
                        type: UIStatusTypes.warning,
                        text: warningStatusesTexts.join(', '),
                    };
                }

                const description = schema.description || schema.display_name;

                switch (schema.type) {
                case controlType.string :
                    if (schema.visual === SchemaItemVisual.ID_SELECT) {
                        control =
                                <IDSelect onSelect={this.onValueChange.bind(this, element, parents)}
                                          multiSelect={schema.multi_select}
                                          placeholder={placeholder}
                                          description={description}
                                          initialValues={value?.[element] || []}
                                          disabled={disable}
                                          addingOddVariants={schema.editable}
                                          required={required}
                                          object={schema.object || GLOBAL_SEARCH_OBJECTS.all}
                                          key={element}/>;
                    } else if (schema.visual === SchemaItemVisual.FILE) {
                        control = <FileInput required={required}
                                             disabled={disable}
                                             key={element}
                                             description={description}
                                             value={value?.[element] ?? ''}
                                             onChange={this.onValueChange.bind(this, element, parents)}
                                             placeholder={placeholder}/>;
                    } else if (schema.visual === SchemaItemVisual.FILE_ST) {
                        control = control = <FileInputST required={required}
                                                         disabled={disable}
                                                         description={description}
                                                         key={element}
                                                         value={value?.[element] ?? ''}
                                                         onChange={this.onValueChange.bind(this, element, parents)}
                                                         placeholder={placeholder}/>;
                    } else if (schema.visual === SchemaItemVisual.DATE_ISO) {
                        control =
                                <DatePicker status={statusObj}
                                            required={required}
                                            placeholder={placeholder}
                                            disabled={disable}
                                            key={element}
                                            description={description}
                                            onlyDate
                                            value={value && value[element] && moment(value[element]) || ''}
                                            onChange={this.onValueChange.bind(this, element, parents)}/>;
                    } else {
                        let inputValue = value?.[element] ?? '';

                        if (schema.visual === SchemaItemVisual.COLOR) {
                            inputValue = inputValue.includes('#') ? inputValue : `#${inputValue}`;
                        }

                        control =
                                <Input status={statusObj}
                                       required={required}
                                       placeholder={placeholder}
                                       disabled={disable}
                                       maxLength={maxLength}
                                       description={description}
                                       key={element}
                                       value={inputValue}
                                       templateList={schema.templateList ?? null}
                                       type={schema.visual === SchemaItemVisual.COLOR ? 'color' : ''}
                                       onChange={this.onValueChange.bind(this, element, parents)}/>;
                    }

                    break;
                case controlType.text :
                    control =
                            <TextArea status={statusObj}
                                      required={required}
                                      placeholder={placeholder}
                                      disabled={disable}
                                      key={element}
                                      description={description}
                                      value={value?.[element] ?? ''}
                                      withoutCross={schema.withoutCross}
                                      onChange={this.onValueChange.bind(this, element, parents)}/>;
                    break;
                case controlType.separator :
                    control =
                            <SeparatorLine key={element} title={schema?.display_name ?? null}/>;
                    break;
                case controlType.json :
                    let jsonValue = value?.[element];
                    if (jsonValue && typeof jsonValue === 'object') {
                        jsonValue = JSON.stringify(jsonValue, null, JSON_TAB_SIZE);
                    }

                    control =
                            <TextArea status={statusObj}
                                      required={required}
                                      json
                                      placeholder={placeholder}
                                      disabled={disable}
                                      description={description}
                                      key={element}
                                      value={jsonValue ?? ''}
                                      onChange={this.onValueChange.bind(this, element, parents)}/>;
                    break;
                case controlType.numeric :
                    if (schema.visual === SchemaItemVisual.TIMESTAMP) {
                        control =
                                <DatePicker onlyDate={!!schema._only_date}
                                            status={statusObj}
                                            required={required}
                                            placeholder={placeholder}
                                            disabled={disable}
                                            description={description}
                                            key={element}
                                            value={value && value[element] ? value[element] * ONE_SECOND : null}
                                            onChange={this.onValueChange.bind(this, element, parents)}/>;
                    } else {
                        const numericDescription = `${value?.[element] && isNaN(value?.[element])
                            ? 'Значение нельзя привести к формату числа, форма вернёт пустое значение этого поля. '
                            : ''}${description ?? ''}`;

                        control = <Input status={statusObj}
                                         required={required}
                                         placeholder={schema.visual === SchemaItemVisual.RUBS
                                                    || schema.visual === SchemaItemVisual.MONEY
                                             ? `${placeholder}, ₽`
                                             : placeholder}
                                         disabled={disable}
                                         min={min}
                                         max={max}
                                         description={numericDescription}
                                         key={element}
                                         value={value[element] ?? ''}
                                         onChange={this.onValueChange.bind(this, element, parents)}/>;
                    }

                    break;
                case controlType.bool :
                    control = <div className={`${style.bool_element} ${required ? style.required : ''}`}
                                   key={element}>
                        <span>{placeholder}{statusObj?.text ? ` (${statusObj.text})` : ''}:</span>
                        <Checkbox description={description}
                                  status={statusObj}
                                  disabled={disable}
                                  checked={value && value[element] || false}
                                  onChange={this.onValueChange.bind(this, element, parents)}/>
                    </div>;
                    break;
                case controlType.variants :

                    if (!schema.reference) {
                        const options: IOptionInfo[] = schema.variants && Array.isArray(schema.variants)
                            ? schema.variants.map(variant => {
                                if (variant.value) {
                                    return variant;
                                }

                                return {
                                    value: variant,
                                    text: variant,
                                };
                            })
                            : [];

                        control =
                                <Select status={statusObj}
                                        required={required}
                                        multiSelect={schema.multi_select}
                                        disabled={disable}
                                        key={element}
                                        description={description}
                                        placeholder={placeholder}
                                        initialValues={value && value[element] || []}
                                        options={options}
                                        onSelect={this.onValueChange.bind(this, element, parents)}
                                        addingOddVariants={schema.editable}/>;

                    } else {
                        control = <SelectConstants reference={schema.reference}
                                                   status={statusObj}
                                                   required={required}
                                                   multiSelect={schema.multi_select}
                                                   disabled={disable}
                                                   key={element}
                                                   placeholder={placeholder}
                                                   initialValues={value && value[element] || []}
                                                   onSelect={this.onValueChange.bind(this, element, parents)}
                                                   addingOddVariants={schema.editable}/>;
                    }

                    break;
                case controlType.structure:
                    const newParents = [...parents, element];
                    const fullPathString = newParents.join(PATH_JOINER);
                    const { excludedStructures = {} } = this.state;

                    const isExcluded = excludedStructures[fullPathString] ?? false;

                    control =
                            <FormStructure collapsed={collapsed}
                                           deprecated={deprecated}
                                           placeholder={placeholder}
                                           key={element}
                                           description={description ?? ''}
                                           onIncludeStatusChange={this.onIncludeStatusChange.bind(this, newParents)}
                                           isExcluded={isExcluded}
                                           schema={schema}
                                           element={element}>
                                {this.buildControls(schema.structure || {}, newParents)}
                                {schema.visual === SchemaItemVisual.TABLE
                                    ? <div className={style.table_open_button}>
                                        <Button basic
                                                onClick={this.openTableModal.bind(this, schema, newParents)}>
                                            Настройка таблицы
                                        </Button>
                                    </div>
                                    : null}
                            </FormStructure>;
                    break;
                case controlType.variable:
                    const values = this.state.values;
                    const newParentsVariable = [...parents, element];

                    const { control_field, variants_fields, default_fields } = schema;
                    let controlFieldKey = control_field ? Object.keys(control_field)[0] : '';
                    controlFieldKey = controlFieldKey ? controlFieldKey?.toString() : '';

                    const valueItemControl = FormConstructorValuesWorker
                        .getValuesItem(values, [...newParentsVariable, controlFieldKey]);
                    if (!isValueExist(valueItemControl)) {

                        parents.reduce((result = {}, parent, index, arr) => {
                            if (index === arr.length - 1) {

                                if (!result[parent]) {
                                    result[parent] = {};
                                }

                                result = result[parent];

                                const newControlValues = FormConstructorSchemaWorker
                                    .constructValuesBySchema(control_field ?? {});

                                Object.entries(newControlValues).forEach(newControlValueEntry => {
                                    const [key, value] = newControlValueEntry;

                                    if (!result[key]) {
                                        result[key] = value;
                                    }
                                });
                            } else {
                                if (!result[parent]) {
                                    result[parent] = {};
                                } else {
                                    return result[parent];
                                }
                            }

                        }, values);
                    }

                    if (variants_fields?.[value?.[element]?.[controlFieldKey]]) {
                        newParentsVariable.reduce((result, parent, index, arr) => {

                            if (index === arr.length - 1) {
                                if (!result[parent]) {
                                    result[parent] = {};
                                }

                                result = result[parent];

                                const newVariantsValues = FormConstructorSchemaWorker
                                    .constructValuesBySchema(
                                        variants_fields?.[value?.[element]?.[controlFieldKey]]
                                        ?? {},
                                    );
                                Object.entries(newVariantsValues).forEach(newVariantsValueEntry => {
                                    const [key, value] = newVariantsValueEntry;
                                    if (!isValueExist(result[key])) {
                                        result[key] = value;
                                    }
                                });

                            } else {
                                if (!result[parent]) {
                                    result[parent] = {};
                                } else {
                                    return result[parent];
                                }
                            }

                        }, values);
                    } else if (default_fields) {
                        newParentsVariable.reduce((result, parent, index, arr) => {

                            if (index === arr.length - 1) {
                                if (!result[parent]) {
                                    result[parent] = {};
                                }

                                result = result[parent];

                                const newDefaultValues = FormConstructorSchemaWorker
                                    .constructValuesBySchema(default_fields ?? {});
                                Object.entries(newDefaultValues).forEach(newDefaultValueEntry => {
                                    const [key, value] = newDefaultValueEntry;
                                    if (!isValueExist(result[key])) {
                                        result[key] = value;
                                    }
                                });

                            } else {
                                if (!result[parent]) {
                                    result[parent] = {};
                                } else {
                                    return result[parent];
                                }
                            }

                        }, values);
                    }

                    control = <div key={`controlFieldKey_${element}`}>
                        {this.buildControls(control_field ?? {}, newParentsVariable, true)}
                        {isValueExist(value?.[element]?.[controlFieldKey])
                            ? variants_fields?.[value?.[element]?.[controlFieldKey]]
                                ? this.buildControls(
                                    variants_fields[value?.[element]?.[controlFieldKey]] ?? {},
                                    newParentsVariable)
                                : default_fields
                                    ? this.buildControls(default_fields ?? {}, newParentsVariable)
                                    : null
                            : null}
                    </div>;
                    break;
                case controlType.array_types :
                    control = this.buildArrayControl(element, schema, parents);
                    break;
                case controlType.string_vector :
                    const status = {
                        type: UIStatusTypes.warning,
                        text: 'Формат string_vector больше не поддерживается',
                    };
                    control = <TextArea status={status}
                                        required={required}
                                        placeholder={placeholder}
                                        disabled={disable}
                                        key={element}
                                        description={description}
                                        value={value && value[element] || ''}
                                        onChange={this.onValueChange.bind(this, element, parents)}/>;
                    break;

                case controlType.ignore:
                    break;
                default:
                    console.error('Неизвестный компонент:\n', schema.type, schema);
                    control = <Input status={statusObj}
                                     required={required}
                                     placeholder={`${placeholder} (Неизвестный компонент ${schema.type})`}
                                     disabled={disable}
                                     key={element}
                                     description={description}
                                     value={value && value[element] || ''}
                                     type={schema.visual === SchemaItemVisual.COLOR ? 'color' : ''}
                                     onChange={this.onValueChange.bind(this, element, parents)}/>;
                    break;
                }

                return control;
            });

        return <>
            {!isVariableControlField && tabsItems && tabsItems.length && tabsItems.length > 1
                ? <Tabs selectTab={this.selectTab.bind(this, parents)}
                        tabs={tabsItems ?? []}
                        currentTab={currentTab ?? DEFAULT_TAB_NAME}/>
                : null}
            {controls}
        </>;
    }

    openTableModal(schema: ISchemaItem, parents: string[]) {
        this.setState({ isTableModalOpen: true, currentTable: { schema, parents } });
    }

    closeTableModal() {
        this.setState({ isTableModalOpen: false, currentTable: null });
    }

    onTableDataSave(tableData: { cols: string[]; rows: string[]; values: string[] }) {
        const { currentTable, values } = this.state;

        const parents: string[] = currentTable?.parents || [];

        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);

        valuesItem.cols = tableData.cols;
        valuesItem.rows = tableData.rows;
        valuesItem.values = tableData.values;

        this.setState({ values }, () => {
            this.onFormValuesChange();
            this.closeTableModal();
        });
    }

    selectTab(parents: string[], selectedTab: string) {
        const { tabs } = this.state;

        const tabsInfo = FormConstructorTabsWorker.changeCurrentTabByPath({
            tabsInfo: deepCopy(tabs),
            parents,
            currentTab: selectedTab,
        });

        this.setState({ tabs: tabsInfo });
    }

    buildArrayControl(element: string, schema: Dict<any>, parents: string[]) {

        const values = deepCopy(this.state.values);
        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);
        const items = valuesItem?.[element];

        const placeholder = schema.display_name
            || (isNaN(+element) ? element : parents[parents.length - 1] && `${parents[parents.length - 1]} [${+element + 1}]`)
            || element;
        const collapsed = !!schema.collapsed;

        const arrayItems = items && Array.isArray(items) && items.length
            ? items.map((item: any, index: number) => {

                const fullPathString = [...parents, element, index.toString()].join(PATH_JOINER);
                const { excludedStructures = {} } = this.state;

                const isExcluded = excludedStructures?.[fullPathString] ?? false;

                return schema.array_type.hasOwnProperty('type') && (typeof schema.array_type.type === 'string')
                    // Передаём в schema index, который уже глубже добавляется к parents
                    // schema передаётся так для общей схемы {key: value}
                    ? this.buildControls({ [index.toString()]: schema.array_type }, [...parents, element])
                    : <FormStructure isExcluded={isExcluded}
                                     onIncludeStatusChange={this.onIncludeStatusChange
                                         .bind(this, [...parents, element, index.toString()])}
                                     collapsed={collapsed}
                                     key={placeholder + `[${(index + 1).toString()}]`}
                                     placeholder={placeholder + `[${(index + 1).toString()}]`}
                                     element={element}
                                     schema={schema}>
                        {this.buildControls(schema.array_type, [...parents, element, index.toString()])}
                    </FormStructure>;
            })
            : null;

        const fullPathString = [...parents, element].join(PATH_JOINER);
        const { excludedStructures = {} } = this.state;

        const isExcluded = excludedStructures?.[fullPathString] ?? false;

        return <div className={style.array_container} key={[...parents, element].join('.')}>
            <FormStructure collapsed={collapsed}
                           placeholder={placeholder}
                           element={element}
                           schema={schema}
                           isExcluded={isExcluded}
                           onIncludeStatusChange={this.onIncludeStatusChange.bind(this, [...parents, element])}>
                {arrayItems && arrayItems.length
                    ? arrayItems.map((arrayControlItem: JSX.Element, index: number, arr: JSX.Element[]) => {
                        return <div key={index} className={style.array_item_container}>
                            <div className={style.array_item}>
                                {arrayControlItem}
                            </div>
                            <div className={style.array_item_controls}>
                                <div className={style.arrows_container}>
                                    {index !== 0
                                        ? <div className={`${style.arrow} ${style.arrow_up}`}
                                               onClick={this.swipeArrayItems
                                                   .bind(this, element, parents, index - 1, index)}/> :
                                        null}
                                    {index !== arr.length - 1
                                        ? <div className={style.arrow}
                                               onClick={this.swipeArrayItems
                                                   .bind(this, element, parents, index + 1, index)}/>
                                        : null}
                                </div>
                            </div>
                            <div className={style.delete_button_container}>
                                <Cross className={style.delete_button}
                                       onClick={this.deleteArrayItem.bind(this, element, parents, index)}/>
                            </div>
                        </div>;
                    })
                    : null}
                <Button className={style.add_button}
                        basic
                        onClick={this.addArrayItem.bind(this, { element, parents, schema })}>
                    Добавить
                </Button>
                {
                    schema?.array_type?.visual === SchemaItemVisual.FILE_ST
                        ? <ButtonUploadST onClick={this.uploadFilesAndArrayItems
                            .bind(this, { element, parents, schema })}/>
                        : null
                }
            </FormStructure>
        </div>;
    }

    onValueChange(element: string, parents: string[], value: any) {
        const { schema, values, tabs } = this.state;
        const lastParent = parents[parents.length - 1];

        let tabsInfo = deepCopy(tabs);

        const schemaItem = FormConstructorSchemaWorker.getSchemaItem({ schema, key: element, parents, values });
        const parentSchemaItem = lastParent
            ? FormConstructorSchemaWorker.getSchemaItem({
                schema,
                key: parents[parents.length - 1],
                parents: parents.slice(0, parents.length - 1),
                values,
            })
            : null;
        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);

        if (parentSchemaItem?.type === controlType.variable) {

            const controlFieldKey = Object.keys(parentSchemaItem?.control_field ?? {})?.[0] ?? null;
            const variantsFieldsKeys = Object.keys(parentSchemaItem?.variants_fields ?? {}) ?? null;

            if (element === controlFieldKey) {
                const parentValuesItem = FormConstructorValuesWorker
                    .getValuesItem(values, parents.slice(0, parents.length - 1));
                parentValuesItem[lastParent] = {};
                parentValuesItem[lastParent][element] = this.formatValuesOnChange({ schemaItem, value });

                if (variantsFieldsKeys.includes(value)) {
                    const newVariantsValues = FormConstructorSchemaWorker
                        .constructValuesBySchema(parentSchemaItem?.variants_fields?.[value] ?? {});

                    Object.entries(newVariantsValues).forEach(newVariantsValueEntry => {
                        const [key, value] = newVariantsValueEntry;
                        if (!isValueExist(parentValuesItem[lastParent][key])) {
                            parentValuesItem[lastParent][key] = value;
                        }
                    });
                } else {
                    const newDefaultValues = FormConstructorSchemaWorker
                        .constructValuesBySchema(parentSchemaItem?.default_fields ?? {});

                    Object.entries(newDefaultValues).forEach(newVariantsValueEntry => {
                        const [key, value] = newVariantsValueEntry;
                        if (!isValueExist(parentValuesItem[lastParent][key])) {
                            parentValuesItem[lastParent][key] = value;
                        }
                    });
                }

                tabsInfo = FormConstructorTabsWorker.changeControlVariableTabInfo({
                    tabsInfo: deepCopy(tabs),
                    parents,
                    schema,
                    values,
                });
            } else {
                valuesItem[element] = this.formatValuesOnChange({ schemaItem, value });
            }
        } else {
            valuesItem[element] = this.formatValuesOnChange({ schemaItem, value });
        }

        this.setState({ values, tabs: tabsInfo }, () => {
            this.onFormValuesChange();
        });
    }

    formatValuesOnChange(props: { schemaItem; value }) {
        const { schemaItem, value } = props;

        const visual = schemaItem?.visual ?? '';
        const isNotTimestamp = !/^[+-]?\d*(\.\d*)?$/ig.test(value) && visual !== SchemaItemVisual.TIMESTAMP;

        if (schemaItem && schemaItem.type === controlType.numeric && isNotTimestamp) {
            if (value === '') {
                return null;
            }

            return value;

        }

        if (visual === SchemaItemVisual.TIMESTAMP) {
            if (value) {
                return typeof value === 'string'
                    ? value
                    : +moment(+value / ONE_SECOND);
            }

            return null;

        }

        //a-kulinchik #drivesup-7788
        if (visual === SchemaItemVisual.DATE_ISO) {
            return value
                ? moment(value)?.format?.('YYYY-MM-DDT00:00:00.000Z')
                : null;
        }

        if (schemaItem?.type && [controlType.string, controlType.text,
            controlType.string_vector, controlType.variants]
            .includes(schemaItem.type)) {

            return isValueExist(value)
                ? value
                : null;
        }

        return value;

    }

    addArrayItem(data: { element: string; parents: string[]; schema: Dict<any>; initialValue?: string }) {
        const { values, tabs, schema: schemaInit } = this.state;
        const { element, parents, schema, initialValue } = data;
        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);

        let newEmptyItem = schema.array_type.hasOwnProperty('type')
        && (typeof schema.array_type.type === 'string')
            ? FormConstructorSchemaWorker.constructValueBySchema(schema.array_type)
            : FormConstructorSchemaWorker.constructValuesBySchema(schema.array_type);

        //for File_ST only
        if (initialValue && schema?.array_type?.visual === SchemaItemVisual.FILE_ST) {
            newEmptyItem = initialValue;
        }

        const tabsInfo = FormConstructorTabsWorker.changeArrayTabInfo({
            schema: schemaInit,
            tabsInfo: deepCopy(tabs),
            action: ArrayAction.ADD,
            parents: [...parents, element],
        });

        if (valuesItem?.[element]) {
            valuesItem[element].push(newEmptyItem);
        } else {
            console.error('No valuesItem');
        }

        this.setState({ values, tabs: tabsInfo }, () => {
            this.onFormValuesChange();
        });
    }

    uploadFilesAndArrayItems(data: { element: string; parents: string[]; schema: Dict<any> }, files: any[]) {
        files.map(el => {
            this.addArrayItem({ ...data, initialValue: el });
        });
    }

    deleteArrayItem(element: string, parents: string[], index: number) {
        const { values, tabs } = this.state;

        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);

        valuesItem[element].splice(index, 1);

        const tabsInfo = FormConstructorTabsWorker.changeArrayTabInfo({
            tabsInfo: deepCopy(tabs),
            action: ArrayAction.REMOVE,
            parents: [...parents, element],
            index,
        });

        this.setState({ values, tabs: tabsInfo }, () => {
            this.onFormValuesChange();
        });
    }

    swipeArrayItems(element: string, parents: string[], index1: number, index2: number) {
        const { values, tabs } = this.state;

        const valuesItem = FormConstructorValuesWorker.getValuesItem(values, parents);

        const tmp = valuesItem[element][index1];
        valuesItem[element][index1] = valuesItem[element][index2];
        valuesItem[element][index2] = tmp;

        const tabsInfo = FormConstructorTabsWorker.changeArrayTabInfo({
            tabsInfo: deepCopy(tabs),
            action: index1 > index2 ? ArrayAction.MOVE_DOWN : ArrayAction.MOVE_UP,
            index: index1,
            parents: [...parents, element],
        });

        this.setState({ values, tabs: tabsInfo }, () => {
            this.onFormValuesChange();
        });
    }

    openDiffModal() {
        this.setState({ isOpenDiffModal: true });
    }

    closeDiffModal() {
        this.setState({ isOpenDiffModal: false });
    }

    render() {
        const { hideChanges } = this.props;
        const { isTableModalOpen, currentTable, values, isOpenDiffModal, formChanges, schema } = this.state;

        return <div className={`${style.component} ${this.props.className || ''}`}>
            {this.buildControls(schema)}
            {isTableModalOpen
                ? <TableConfigModal values={values}
                                    tableData={currentTable}
                                    onClose={this.closeTableModal.bind(this)}
                                    onSave={this.onTableDataSave.bind(this)}/>
                : null}
            {formChanges && !hideChanges
                ? <Link className={style.changes_link_container}
                        onClick={this.openDiffModal.bind(this)}>
                    Изменения в форме
                </Link>
                : null}
            {isOpenDiffModal
                ? <FormChangesModal onClose={this.closeDiffModal.bind(this)} diffData={formChanges} schema={schema}/>
                : null}
        </div>;
    }
}
