import { Dict } from '../../../../types';
import { COINS_LENGTH, COINS_PER_RUBLE } from '../../../constants';
import { deepCopy, isDateValid } from '../../../utils/utils';
import { JSON_TAB_SIZE, STRING_VECTOR_SPLITTER } from '../constants';
import { FormConstructorValuesWorker } from '../FormConstructorValuesWorker';
import { controlType, ISchemaItem, SchemaItemVisual } from '../types';
import { isValueExist } from '../utils/utils';

export class FormConstructorSchemaWorker {
    static constructValuesBySchema(schemaInit: ISchemaItem): Dict<any> {
        const schema = deepCopy(schemaInit);

        return Object.entries(schema)
            .reduce((result: Dict<any>, entry: [string, ISchemaItem]) => {
                const [element, schema] = entry;
                if (schema.type !== controlType.separator) {
                    result[element] = this.constructValueBySchema(schema);
                }

                return result;
            }, {});
    }

    static constructValueBySchema(schema: ISchemaItem): any {
        const defaultValue = schema.hasOwnProperty('default') ? schema.default : null;
        const { type, visual } = schema;
        let value: any;

        switch (type) {
        case controlType.structure:
            value = this.constructValuesBySchema(schema.structure || {});
            break;
        case controlType.array_types:
            value = defaultValue || [];
            break;
        case controlType.string_vector:
            value = defaultValue && Array.isArray(defaultValue) && defaultValue.join(STRING_VECTOR_SPLITTER) || '';
            break;
        case controlType.bool:
            value = defaultValue || false;
            break;
        case controlType.numeric:
            if (visual === SchemaItemVisual.TIMESTAMP && defaultValue) {
                if (isDateValid(defaultValue)) {
                    value = defaultValue;
                } else {
                    console.error('Default timestamp value is not valid');
                    value = null;
                }
            } else if (visual === SchemaItemVisual.RUBS || visual === SchemaItemVisual.MONEY) {
                value = typeof defaultValue === 'number'
                    ? +((defaultValue / COINS_PER_RUBLE).toFixed(COINS_LENGTH))
                    : null;
            } else {
                value = typeof defaultValue === 'number' ? defaultValue : null;
            }

            break;
        case controlType.json:
            if (typeof defaultValue !== 'object') {
                console.warn(`Data for control with JSON type isn't Object or Array`);
            }

            value = defaultValue ? JSON.stringify(defaultValue, null, JSON_TAB_SIZE) : null;
            break;
        case controlType.variable:
            const { control_field, variants_fields, default_fields } = schema;
            if (!control_field || !variants_fields) {
                console.error(`Отсутствуют поля control_field, variants_fields или default_fields  в схеме с типом ${controlType.variable}`);
            }

            const controlFieldValue = this.constructValuesBySchema(control_field ?? {});
            const controlFieldCurrentValue: string = Object.values(controlFieldValue)[0];

            const variantsFieldValue = isValueExist(controlFieldCurrentValue)
                ? variants_fields?.[controlFieldCurrentValue]
                    ? this.constructValuesBySchema(variants_fields[controlFieldCurrentValue] ?? {})
                    : default_fields
                        ? this.constructValuesBySchema(default_fields ?? {})
                        : {}
                : {};

            value = Object.assign(controlFieldValue, variantsFieldValue);
            break;
        default:
            if (visual === SchemaItemVisual.COLOR) {
                value = defaultValue?.includes('#') ? defaultValue : `#${defaultValue}`;
            } else {
                value = defaultValue;
            }

            break;
        }

        return value;
    }

    static getSchemaItem(props: {
        schema: Dict<any>;
        key: string;
        parents?: string[];
        controlsValues?: string[];
        values?: Dict<any>;
    }): ISchemaItem | null {
        const { schema = {}, key = '', parents = [], controlsValues = [], values } = props;
        const ARRAY_TYPE_FIELD = 'array_type';
        const numberRegExp = new RegExp(/^\d+$/ig);
        let controlValuesIndex = 0;

        const isArraySchema = (schemaItemKey: string, schemaParentItem?: ISchemaItem) => {
            return numberRegExp.test(schemaItemKey)
                || (schemaParentItem && Object.keys(schemaParentItem)?.includes(ARRAY_TYPE_FIELD));
        };

        const getVariableItem = (schemaItem: ISchemaItem, parents: string[]) => {
            const isControlField = Object.keys(schemaItem?.control_field ?? {}).includes(key);
            if (isControlField) {
                return schemaItem.control_field;
            }

            if (values) {
                const valuesItem = FormConstructorValuesWorker.getValuesItem(values ?? {}, parents);
                const controlFieldKey = schemaItem.control_field && Object.keys(schemaItem.control_field)[0];
                const controlFieldValue = valuesItem[controlFieldKey ?? ''];

                if (Object.keys(schemaItem?.variants_fields ?? {}).includes(controlFieldValue)) {
                    return schemaItem?.variants_fields?.[controlFieldValue];
                }

                return schemaItem?.default_fields ? schemaItem?.default_fields : null;

            }

            if (Object.keys(schemaItem?.variants_fields ?? {}).includes(controlsValues[controlValuesIndex])
                    && (controlsValues[controlValuesIndex] !== null
                    && controlsValues[controlValuesIndex] !== undefined)) {

                let resultSchemaItem = schemaItem?.variants_fields;

                Object.entries(schemaItem?.variants_fields ?? {}).forEach((variantEntry: [string, Dict<any>]) => {
                    const [variantKey, variantValue] = variantEntry;
                    if (variantKey === controlsValues[controlValuesIndex]) {
                        controlValuesIndex++;
                        resultSchemaItem = variantValue;
                    }
                });

                return resultSchemaItem;
            }

            return schemaItem?.default_fields ? schemaItem?.default_fields : null;

        };

        const schemaParentItem = parents?.length
            ? parents.reduce((resultValue: Dict<any>, parentItem: string, index: number) => {
                let parent = parentItem;
                if (isArraySchema(parentItem)) {
                    parent = ARRAY_TYPE_FIELD;
                }

                if (resultValue?.type === 'variable') {
                    const schemaVariableParentItem = getVariableItem(resultValue, parents.slice(0, index));

                    return schemaVariableParentItem?.[parent]?.type
                    && schemaVariableParentItem[parent].type === controlType.structure
                        ? schemaVariableParentItem[parent].structure
                        : schemaVariableParentItem?.[parent] || {};
                }

                return resultValue?.[parent]?.type
                && resultValue[parent].type === controlType.structure
                    ? resultValue[parent].structure
                    : resultValue?.[parent] || {};
            }, schema)
            : schema;

        if (isArraySchema(key, schemaParentItem)) {
            return schemaParentItem && schemaParentItem.type === controlType.variants
                ? schemaParentItem
                : schemaParentItem?.[ARRAY_TYPE_FIELD];
        }

        if (schemaParentItem?.type === 'variable') {
            const schemaVariableParentItem = getVariableItem(schemaParentItem, parents);

            return schemaVariableParentItem?.[key]
                ? schemaVariableParentItem[key]
                : null;
        }

        return schemaParentItem?.[key]
            ? schemaParentItem[key]
            : null;

    }
}
