import { Dict } from '../../../../types';
import { deepCopy } from '../../../utils/utils';
import { FormConstructorSchemaWorker } from '../FormConstructorSchemaWorker';
import { FormConstructorValuesWorker } from '../FormConstructorValuesWorker';
import { controlType, ISchemaItem } from '../types';

const JOIN_SYMBOL = '.';
export const ROOT_KEY = '__root';
export const DEFAULT_TAB_NAME = 'default';
export const ARRAY_KEY = '__array_key';

export enum ArrayAction {
    ADD = 'add',
    REMOVE = 'remove',
    MOVE_DOWN = 'moveDown',
    MOVE_UP = 'moveUp',
}

export interface IFormTabs {
    tabs: { name: string; isRequired?: boolean; countItems: number }[];
    currentTab: string;
}

export class FormConstructorTabsWorker {
    //Tabs info is collecting with PARENTS path, because every element is below this path
    static constructTabsBySchema(props: {
        schemaInit: Dict<ISchemaItem>;
        parents?: string[];
        values?: Dict<any>;
    }): Dict<IFormTabs> {
        const { schemaInit = {}, parents = [], values } = props;
        const pathKey = parents.length ? parents.join(JOIN_SYMBOL) : ROOT_KEY;
        let schema: any = parents.length
            ? FormConstructorSchemaWorker.getSchemaItem({
                schema: schemaInit,
                key: parents[parents.length - 1],
                parents: parents.slice(0, parents.length - 1),
                values,
            })
            : schemaInit;

        if (schema?.type === controlType.structure) {
            schema = schema?.structure;
        }

        if (schema?.type === controlType.array_types) {
            if (typeof schema?.array_type?.type === 'string') {
                schema = { [ARRAY_KEY]: schema?.array_type };
            } else {
                schema = schema?.array_type;
            }
        }

        if (schema?.type === controlType.variable) {
            const variableValueItem = FormConstructorValuesWorker.getValuesItem(values ?? {}, [...parents]);
            const controlFieldKey = schema.control_field && Object.keys(schema.control_field)[0];
            const controlFieldValue = variableValueItem?.[controlFieldKey ?? ''];

            if (controlFieldValue) {
                let schemaVariantItem: Dict<ISchemaItem> | null = null;

                if (Object.keys(schema?.variants_fields ?? {}).includes(controlFieldValue)) {
                    schemaVariantItem = schema?.variants_fields?.[controlFieldValue] ?? null;
                } else {
                    schemaVariantItem = schema?.default_fields ? schema?.default_fields : null;
                }

                schema = schemaVariantItem;
            }
        }

        return Object.entries(schema ?? {}).reduce((result: Dict<IFormTabs>, entry: [string, ISchemaItem]) => {
            const [element, schemaItem] = entry;
            const { tab_name = null, type } = schemaItem;

            const tabName = tab_name !== null ? tab_name : DEFAULT_TAB_NAME;

            //get tabs of current level
            if (result[pathKey]) {
                if (result[pathKey].tabs.some(pathTab => pathTab.name === tabName)) {
                    const foundedTab = result[pathKey].tabs?.filter(pathTab => pathTab.name === tabName)?.[0] ?? null;
                    if (foundedTab && 'countItems' in foundedTab) {
                        foundedTab.countItems++;
                    }

                } else {
                    result[pathKey].tabs.push({
                        name: tabName,
                        countItems: 1,
                    });
                }
            } else {
                result[pathKey] = {
                    tabs: [{
                        name: tabName,
                        countItems: 1,
                    }],
                    currentTab: tabName,
                };
            }

            //sorting tabs
            result[pathKey].tabs?.sort((tab1, tab2) => {
                const name1 = tab1?.name?.toString() ?? '';
                const name2 = tab2?.name?.toString() ?? '';

                return name1.localeCompare(name2);
            });

            //init first of sorted tabs as currentTab
            result[pathKey].currentTab = result[pathKey].tabs[0]?.name;

            //get tabs for structure
            if (type === controlType.structure) {
                const nestedTabsInfo = FormConstructorTabsWorker.constructTabsBySchema({
                    schemaInit,
                    parents: [...parents, element],
                    values,
                });
                Object.entries(nestedTabsInfo).forEach((nestedTabsEntry: [string, IFormTabs]) => {
                    const [key, tabsInfo] = nestedTabsEntry;
                    result[key] = tabsInfo;
                });
            }

            //get tabs for primitive array
            if (type === controlType.array_types) {
                if (values) {
                    const valueItem = FormConstructorValuesWorker.getValuesItem(values, [...parents, element]);

                    const nestedTabsInfo = FormConstructorTabsWorker.constructTabsBySchema({
                        schemaInit,
                        parents: [...parents, element],
                    });

                    for (let i = 0; i < valueItem?.length; i++) {
                        Object.entries(nestedTabsInfo).forEach((nestedTabsEntry: [string, IFormTabs]) => {
                            const [key, tabsInfo] = nestedTabsEntry;
                            result[[key, i].join(JOIN_SYMBOL)] = tabsInfo;
                        });
                    }
                }
            }

            //get tabs for variable
            if (type === controlType.variable) {
                const variableValueItem = FormConstructorValuesWorker
                    .getValuesItem(values ?? {}, [...parents, element]);
                const controlFieldKey = schemaItem.control_field && Object.keys(schemaItem.control_field)[0];
                const controlFieldValue = variableValueItem?.[controlFieldKey ?? ''];

                if (controlFieldValue) {
                    let schemaVariantItem: Dict<ISchemaItem> | null = null;

                    if (Object.keys(schemaItem?.variants_fields ?? {}).includes(controlFieldValue)) {
                        schemaVariantItem = schemaItem?.variants_fields?.[controlFieldValue] ?? null;
                    } else {
                        schemaVariantItem = schemaItem?.default_fields ? schemaItem?.default_fields : null;
                    }

                    if (schemaVariantItem !== null) {
                        const schemaVariantItemTabs = FormConstructorTabsWorker.constructTabsBySchema({
                            schemaInit,
                            parents: [...parents, element],
                            values,
                        });

                        Object.entries(schemaVariantItemTabs).forEach((nestedTabsEntry: [string, IFormTabs]) => {
                            const [key, tabsInfo] = nestedTabsEntry;
                            result[key] = tabsInfo;
                        });
                    }
                }
            }

            return result;
        }, {});
    }

    static getTabsInfoByPath(props: { tabsInfo: Dict<IFormTabs>; parents?: string[] }): IFormTabs | null {
        const { tabsInfo, parents = [] } = props;
        const pathKey = parents.length ? parents.join(JOIN_SYMBOL) : ROOT_KEY;

        return tabsInfo[pathKey];
    }

    static changeCurrentTabByPath(props: {
        tabsInfo: Dict<IFormTabs>;
        parents?: string[];
        currentTab: string;
    }): Dict<IFormTabs> {
        const { tabsInfo: tabsInfoInit, parents = [], currentTab } = props;
        const pathKey = parents.length ? parents.join(JOIN_SYMBOL) : ROOT_KEY;
        const tabsInfo = deepCopy(tabsInfoInit);

        if (tabsInfo[pathKey]) {
            const tabNames = tabsInfo[pathKey]?.tabs.map(tabInfo => tabInfo.name);

            if (tabNames.includes(currentTab)) {
                tabsInfo[pathKey].currentTab = currentTab;
            } else {
                console.warn('currentTab is not included in tabs array in FormConstructorTabsWorker.changeCurrentTabByPath()');
            }
        }

        return tabsInfo;
    }

    static changeArrayTabInfo(props: {
        schema?: Dict<ISchemaItem>;
        tabsInfo: Dict<IFormTabs>;
        parents?: string[];
        index?: number;
        action: ArrayAction;
    }): any {
        const { schema, tabsInfo: tabInfoInit, action, parents = [], index = 0 } = props;
        const tabsInfo = deepCopy(tabInfoInit);
        const pathKey = parents.length ? parents.join(JOIN_SYMBOL) : ROOT_KEY;
        const pathRegExp = new RegExp(`${pathKey}\\.\\d+`, 'ig');

        if (action === ArrayAction.ADD) {
            const tabsArrayPath = Object.keys(tabInfoInit).filter(tabsPath => {
                return tabsPath.match(pathRegExp);
            });

            const arrayIndexes: number[] = tabsArrayPath.map(tabsArrayPathItem => {
                const splitPath = tabsArrayPathItem.split(JOIN_SYMBOL);
                const lastItem = splitPath[splitPath.length - 1];

                return +lastItem;
            }).sort((a: number, b: number) => a - b);

            const newIndex = tabsArrayPath.length ? arrayIndexes[arrayIndexes.length - 1] + 1 : 0;

            const tabInfo = FormConstructorTabsWorker.constructTabsBySchema({ parents, schemaInit: schema ?? {} });

            Object.entries(tabInfo).forEach(tabInfoItem => {
                const [key, tabsValue] = tabInfoItem;
                const newPathKey = `${key}${JOIN_SYMBOL}${newIndex}`;

                tabsInfo[newPathKey] = tabsValue;
            });
        }

        if (action === ArrayAction.REMOVE) {
            const tabsArrayKeys: string[] = Object.keys(tabInfoInit).filter(tabsPath => {
                return tabsPath.match(pathRegExp);
            });

            const tabsArrayInfo = tabsArrayKeys.reduce((result: [string, IFormTabs][], tabsArrayKey) => {
                result.push([tabsArrayKey, tabInfoInit[tabsArrayKey]]);

                return result;
            }, []);

            tabsArrayKeys.forEach(tabsArrayKey => {
                delete tabsInfo[tabsArrayKey];
            });

            let lastIndex = 0;
            const newTabsArray = tabsArrayInfo
                .sort((tabsArrayInfoItem1, tabsArrayInfoItem2) => {
                    const [keyPath1] = tabsArrayInfoItem1;
                    const [keyPath2] = tabsArrayInfoItem2;
                    const splitPath1 = keyPath1.split(JOIN_SYMBOL);
                    const splitPath2 = keyPath2.split(JOIN_SYMBOL);
                    const lastItem1 = +splitPath1[splitPath1.length - 1];
                    const lastItem2 = +splitPath2[splitPath2.length - 1];

                    return lastItem1 - lastItem2;
                })
                .reduce((result: any[], tabsArrayInfoItem, itemIndex) => {
                    const [keyPath, tabsValue] = tabsArrayInfoItem;
                    const splitPath = keyPath.split(JOIN_SYMBOL);

                    if (index !== itemIndex) {
                        splitPath[splitPath.length - 1] = lastIndex.toString();
                        lastIndex++;

                        result.push([splitPath.join(JOIN_SYMBOL), tabsValue]);
                    }

                    return result;

                }, []);

            newTabsArray.forEach(newTab => {
                const [key, tabsValue] = newTab;
                tabsInfo[key] = tabsValue;
            });
        }

        if (action === ArrayAction.MOVE_UP || action === ArrayAction.MOVE_DOWN) {
            const tabsInfoForSwipePath1 = `${pathKey}${JOIN_SYMBOL}${index?.toString()}`;
            const tabsInfoForSwipePath2 = `${pathKey}${JOIN_SYMBOL}${(action === ArrayAction.MOVE_UP
                ? index + 1
                : index - 1)?.toString()}`;

            if (tabsInfo[tabsInfoForSwipePath1] && tabsInfo[tabsInfoForSwipePath2]) {
                const tmp = deepCopy(tabsInfo[tabsInfoForSwipePath1]);

                tabsInfo[tabsInfoForSwipePath1] = tabsInfo[tabsInfoForSwipePath2];
                tabsInfo[tabsInfoForSwipePath2] = tmp;
            } else {
                console.warn('No item to swipe in FormConstructorTabsWorker.changeArrayTabInfo()');
            }
        }

        return tabsInfo;
    }

    static changeControlVariableTabInfo(props: {
        schema?: Dict<ISchemaItem>;
        tabsInfo: Dict<IFormTabs>;
        parents?: string[];
        values;
    }) {
        const { tabsInfo, schema, parents, values } = props;
        const pathKey = parents?.length ? parents?.join(JOIN_SYMBOL) : ROOT_KEY;

        const newTabsInfo = deepCopy(tabsInfo);

        delete newTabsInfo?.[pathKey];

        const newTabs = FormConstructorTabsWorker.constructTabsBySchema({ schemaInit: schema ?? {}, parents, values });

        Object.entries(newTabs).forEach((newTabsEntry: [string, IFormTabs]) => {
            const [key, tabsInfo] = newTabsEntry;
            newTabsInfo[key] = tabsInfo;
        });

        return newTabsInfo;
    }
}
