import { Dict } from '../../../../types';
import { COINS_PER_RUBLE } from '../../../constants';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { STRING_VECTOR_SPLITTER } from '../constants';
import { FormConstructorSchemaWorker } from '../FormConstructorSchemaWorker';
import { controlType, ISchemaItem, SchemaItemVisual } from '../types';
import { isValueExist } from '../utils/utils';

export const PATH_JOINER = '.';

export class FormConstructorValuesWorker {
    static getValuesItem(values: Dict<any>, parents: string[]) {
        return parents && parents.length
            ? parents.reduce((resultValue: Dict<any>, parentItem: string) => {
                return resultValue?.[parentItem] ?? null;
            }, values)
            : values;
    }

    //depreacated saved for debugging, remove all code after 31.07.2021
    // static OLD_formatValues(values: Dict<any>, parents, schema, rawResult?: boolean, excludedStructures: Dict<boolean> = {}, controlsValues: string[] = []) {
    //     console.warn('formatValues is deprecated method');
    //     let copiedControlsValues = [...controlsValues];
    //     return Object.entries(values).reduce((result: Dict<any>, entry: [string, any]) => {
    //         let [key, value] = entry;
    //
    //         if (value !== null || rawResult) {
    //             let schemaItem = FormConstructorSchemaWorker.getSchemaItem({
    //                 schema,
    //                 key,
    //                 parents,
    //                 controlsValues: copiedControlsValues
    //             });
    //
    //             if (schemaItem && schemaItem.type === controlType.variable) {
    //                 let controlFieldKey = schemaItem.control_field && Object.keys(schemaItem.control_field)[0];
    //                 let controlFieldValue = value[controlFieldKey ?? ''];
    //                 copiedControlsValues.push(controlFieldValue);
    //
    //                 Object.entries(value).forEach(entry => {
    //                     let [entryKey, entryValue] = entry;
    //                     if (entryValue !== null) {
    //                         result[entryKey] = this.checkFormattedValue(entryValue, entryKey, [...parents, key], schema, rawResult, excludedStructures, copiedControlsValues);
    //                     }
    //                 });
    //             } else {
    //                 if (this.checkFormattedValue(value, key, parents, schema, rawResult, excludedStructures, copiedControlsValues) !== null) {
    //                     result[key] = this.checkFormattedValue(value, key, parents, schema, rawResult, excludedStructures, copiedControlsValues);
    //                 }
    //             }
    //         }
    //
    //         return result;
    //     }, {});
    // }
    //
    // static checkFormattedValue(value, key, parents, schema, rawResult, excludedStructures: Dict<boolean> = {}, controlsValues: string[] = []): any {
    //
    //     if (Array.isArray(value)) {
    //         let fullPathString = [...parents, key].join(PATH_JOINER);
    //
    //         return excludedStructures?.[fullPathString]
    //             ? null
    //             : value.map((item, index) => this.checkFormattedValue(item, index.toString(), [...parents, key], schema, rawResult, excludedStructures, controlsValues));
    //
    //     } else if (isObject(value)) {
    //         let fullPathString = [...parents, key].join(PATH_JOINER);
    //
    //         return excludedStructures?.[fullPathString]
    //             ? null
    //             : FormConstructorValuesWorker.formatValues(value, [...parents, key], schema, rawResult, excludedStructures, controlsValues);
    //
    //     } else {
    //         return this.formatValue(value, key, parents, schema, controlsValues);
    //     }
    // }
    //
    // static formatValue(value, key, parents, schema, controlsValues: string[] = []) {
    //     let schemaItemProps = {schema, key, parents, controlsValues};
    //     let schemaItem = FormConstructorSchemaWorker.getSchemaItem(schemaItemProps);
    //
    //     if (schemaItem && schemaItem.type === controlType.numeric && value !== null) {
    //         if (value === '' || value === undefined) {
    //             return null;
    //         } else {
    //             if (schemaItem.visual === SchemaItemVisual.RUBS || schemaItem.visual === SchemaItemVisual.MONEY) {
    //                 return Math.floor(Math.fround(+value * COINS_PER_RUBLE));
    //             }
    //             return +value;
    //         }
    //     }
    //
    //     if (schemaItem && schemaItem.type === controlType.string_vector) {
    //         return value ? value.split(STRING_VECTOR_SPLITTER).map(item => item.trim()) : [];
    //     }
    //
    //     if (schemaItem && schemaItem.type === controlType.json) {
    //         return isValidJSONString(value) ? JSON.parse(value) : value;
    //     }
    //     return value;
    // }

    static formatValues(props: {
        valuesFull: Dict<any>; schemaFull: Dict<ISchemaItem>; parents?: string[];
        excludedStructures?: Dict<boolean>; rawResult?: boolean;
    }) {
        const { valuesFull, schemaFull, parents = [], excludedStructures = {}, rawResult = false } = props;

        const values = FormConstructorValuesWorker.getValuesItem(valuesFull, parents);

        return Object.entries(values).reduce((result, entry: [string, any]) => {
            const [key, value] = entry;

            const schemaItem = FormConstructorSchemaWorker.getSchemaItem({
                schema: schemaFull,
                parents,
                key,
                values: valuesFull,
                controlsValues: [],
            });
            const { type } = schemaItem ?? {};

            switch (type) {
            case controlType.numeric:
                if (value !== '' && value !== undefined && value !== null && !isNaN(value)) {
                    if (schemaItem?.visual === SchemaItemVisual.RUBS || schemaItem?.visual === SchemaItemVisual.MONEY) {
                        result[key] = Math.floor(Math.fround(+value * COINS_PER_RUBLE));
                    } else {
                        result[key] = +value;
                    }
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            case controlType.string_vector:
                result[key] = value
                    ? value.split(STRING_VECTOR_SPLITTER).map(item => item.trim())
                    : [];
                break;
            case controlType.json:
                if (value !== '' && value !== undefined && value !== null) {
                    result[key] = isValidJSONString(value)
                        ? JSON.parse(value)
                        : value;
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            case controlType.array_types:
                const fullPathArrayString = [...parents, key].join(PATH_JOINER);

                if (value !== '' && value !== undefined && value !== null
                        && Array.isArray(value) && !excludedStructures?.[fullPathArrayString]) {
                    const arraySchemaItem = FormConstructorSchemaWorker.getSchemaItem({
                        schema: schemaFull,
                        parents,
                        key,
                        values: valuesFull,
                    });

                    const isOneElement = arraySchemaItem?.array_type?.type
                        ? typeof arraySchemaItem?.array_type?.type === 'string'
                        : false;

                    if (isOneElement && arraySchemaItem?.array_type?.type !== controlType.variable) {
                        const resultObject = FormConstructorValuesWorker.formatValues({
                            valuesFull,
                            schemaFull,
                            parents: [...parents, key],
                            excludedStructures,
                            rawResult,
                        });

                        result[key] = Object.values(resultObject).reduce((result: any[], resultValue: any) => {
                            result.push(resultValue);

                            return result;
                        }, []);
                    } else {
                        result[key] = value.map((_, index) => {
                            return FormConstructorValuesWorker.formatValues({
                                valuesFull,
                                schemaFull,
                                parents: [...parents, key, index.toString()],
                                excludedStructures,
                                rawResult,
                            });
                        });
                    }
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            case controlType.structure:
                const fullPathStructureString = [...parents, key].join(PATH_JOINER);

                if (value !== undefined && value !== null && !excludedStructures?.[fullPathStructureString]) {
                    result[key] = FormConstructorValuesWorker.formatValues({
                        valuesFull,
                        schemaFull,
                        parents: [...parents, key],
                        excludedStructures,
                    });
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            case controlType.variable:
                if (value !== undefined && value !== null) {
                    const variableResult = FormConstructorValuesWorker.formatValues({
                        valuesFull,
                        schemaFull,
                        parents: [...parents, key],
                        excludedStructures,
                        rawResult,
                    });

                    Object.entries(variableResult).forEach((variableEntry: [string, any]) => {
                        const [variableKey, variableValue] = variableEntry;
                        result[variableKey] = variableValue;
                    });
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            default:
                if (value !== '' && value !== undefined && value !== null) {
                    result[key] = value;
                } else if (rawResult) {
                    result[key] = value;
                }

                break;
            }

            return result;
        }, {});
    }

    //Get validation result
    static isValuesValid(values: Dict<any>) {
        return Object.values(values).every(value => this.isValueValid(value));
    }

    static isValueValid(value: any) {
        const isPrimitiveValueValid = (value: boolean | null) => value !== false;

        if (typeof value === 'object' && value !== null) {
            if (Array.isArray(value)) {
                return value.every(arrItem => this.isValueValid(arrItem));
            }

            return this.isValuesValid(value);

        }

        return isPrimitiveValueValid(value);

    }

    //Validation
    static validateValues(props: {
        valuesFull: Dict<any>;
        schemaFull: Dict<ISchemaItem>;
        parents?: string[];
    }): Dict<boolean | Dict<boolean>> {
        const { valuesFull, schemaFull, parents = [] } = props;

        const values = FormConstructorValuesWorker.getValuesItem(valuesFull, parents);

        return Object.entries(values).reduce((result, entry: [string, any]) => {
            const [key, value] = entry;

            const schemaItem = FormConstructorSchemaWorker.getSchemaItem({
                schema: schemaFull,
                parents,
                key,
                values: valuesFull,
                controlsValues: [],
            });
            const { type } = schemaItem ?? {};

            switch (type) {
            case controlType.numeric:
                const min = schemaItem?.min ?? -Infinity;
                const max = schemaItem?.max ?? Infinity;

                //if numeric value exist, it always must be valid. If value don't exist, validation depends on 'required' field
                result[key] = isValueExist(value)
                    ? !isNaN(+value) && +value >= min && +value <= max
                    : !schemaItem?.required;
                break;
            case controlType.json:
                //if json value exist, it always must be valid. If value don't exist, validation depends on 'required' field
                result[key] = value
                    ? isValidJSONString(value)
                    : !schemaItem?.required;
                break;
            case controlType.bool:
                //boolean must be only true or false

                //Custom string values is valid
                const STRING_BOOL_VARIANTS = ['0', '1', 'true', 'false', 'да', 'нет', 'da', 'net'];

                result[key] = typeof value === 'boolean'
                        || STRING_BOOL_VARIANTS.includes(value?.toString()?.toLowerCase());
                break;
            case controlType.structure:
                //structure only checks nested values, empty structure is still valid
                result[key] = value !== null && value !== undefined && value !== ''
                    ? FormConstructorValuesWorker.validateValues({
                        valuesFull,
                        schemaFull,
                        parents: [...parents, key],
                    })
                    : !schemaItem?.required;

                break;
            case controlType.array_types:
                if (value !== '' && value !== undefined && value !== null && Array.isArray(value)) {
                    const arraySchemaItem = FormConstructorSchemaWorker.getSchemaItem({
                        schema: schemaFull,
                        parents,
                        key,
                        values: valuesFull,
                    });

                    const isOneElement = arraySchemaItem?.array_type?.type
                        ? typeof arraySchemaItem?.array_type?.type === 'string'
                        : false;

                    if (isOneElement && arraySchemaItem?.array_type?.type !== controlType.variable) {
                        const resultObject = FormConstructorValuesWorker.validateValues({
                            valuesFull,
                            schemaFull,
                            parents: [...parents, key],
                        });

                        result[key] = Object.values(resultObject).reduce((result: any[], resultValue: any) => {
                            result.push(resultValue);

                            return result;
                        }, []);
                    } else {
                        result[key] = value.map((_, index) => {
                            return FormConstructorValuesWorker.validateValues({
                                valuesFull,
                                schemaFull,
                                parents: [...parents, key, index.toString()],
                            });
                        });
                    }
                } else {
                    result[key] = !schemaItem?.required;
                }

                break;
            case controlType.variable:
                if (value !== undefined && value !== null) {
                    const variableResult = FormConstructorValuesWorker.validateValues({
                        valuesFull,
                        schemaFull,
                        parents: [...parents, key],
                    });

                    Object.entries(variableResult).forEach((variableEntry: [string, any]) => {
                        const [variableKey, variableValue] = variableEntry;
                        result[variableKey] = variableValue;
                    });
                } else {
                    result[key] = !schemaItem?.required;
                }

                break;
            default:
                result[key] = schemaItem?.required ? isValueExist(value) : true;
            }

            return result;
        }, {});
    }
}
