import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { Link } from '../../../ui/Link';
import { ISchemaItem } from '../types';
import * as style from './index.css';

interface IFormStructureProps {
    element: string;
    schema: ISchemaItem;
    placeholder: string;
    deprecated?: boolean;
    collapsed?: boolean;
    onIncludeStatusChange: () => {};
    isExcluded: boolean;
    description?: string;
}

interface IFormStructureState {
    collapsed: boolean;
}

export default class FormStructure extends React.Component<IFormStructureProps, IFormStructureState> {
    state: IFormStructureState = {
        collapsed: true,
    };
    id = Math.random();

    constructor(props: IFormStructureProps) {
        super(props);

        this.state = { collapsed: !!this.props.collapsed };
    }

    changeCollapsedStatus() {
        this.setState({ collapsed: !this.state.collapsed });
    }

    render() {
        const { schema, element, placeholder, onIncludeStatusChange, description } = this.props;
        let { collapsed } = this.state;
        const required = !!schema.required;
        const isExcluded = required ? false : this.props.isExcluded ?? false;
        collapsed = collapsed || isExcluded;
        const id = `description_structure_${this.id}`;

        return <fieldset key={element}
                         className={`${style.form_constructor__fieldset} ${required ? style.required : ''} ${isExcluded ? style.excluded : ''}`}>
            <legend onClick={isExcluded ? null : this.changeCollapsedStatus.bind(this)}>
                <span className={style.collapse_icon}>
                    {collapsed ? '+' : '-'}
                </span>
                <span>
                    {placeholder}
                </span>
                <span>
                    {this.props.deprecated ? ' (Deprecated)' : ''}
                </span>
                {!required
                    ? <span>
                    (<Link onClick={onIncludeStatusChange?.bind(this)}>
                            {
                                isExcluded
                                    ? 'Включить'
                                    : 'Исключить'
                            }
                        </Link>)
                    </span>
                    : null}
                {description
                    ? <>
                        <span data-tip data-for={id} className={style.description_icon}>?</span>
                        <ReactTooltip className={style.description_tooltip}
                                      place={'left'}
                                      id={id}
                                      effect="solid">
                            {description}
                        </ReactTooltip>
                    </>
                    : null}
            </legend>
            {collapsed
                ? <div className={style.collapsed_info}
                       onClick={isExcluded ? null : this.changeCollapsedStatus.bind(this)}>
                    {isExcluded ? 'Структура исключена' : null}...
                </div>
                : this.props.children}
        </fieldset>;
    }
}
