import * as React from 'react';

import { Dict } from '../../../../types';
import { CancelButton, SaveButton } from '../../../ui/Button';
import { Cross } from '../../../ui/Cross';
import { Confirm, Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { SimpleError } from '../../SimpleError';
import { ISchemaItem } from '../types';
import * as style from './index.css';

const DATA_SEPARATOR = ',';
const EMPTY_VALUE = '-';

enum TABLE_FIELDS {
    COLS = 'cols',
    ROWS = 'rows',
    VALUES = 'values',
}

interface ITableConfigModalProps {
    onClose: () => void;
    onSave: (data: { cols: string; rows: string; values: string }) => void;
    tableData: { schema: ISchemaItem; parents: string[] } | null;
    values: Dict<any>;
}

interface ITableConfigModalState {
    absentFieldsError: Error | null;
    valuesAreaError: Error | null;
    cols: string[];
    rows: string[];
    values: string[];
    confirmIsOpen: boolean;
    question: string;
    accept: () => void;
}

export default class TableConfigModal extends React.Component<ITableConfigModalProps, ITableConfigModalState> {
    state: ITableConfigModalState = {
        absentFieldsError: null,
        valuesAreaError: null,
        cols: [],
        rows: [],
        values: [],
        confirmIsOpen: false,
        question: '',
        accept: () => {},
    };

    componentDidMount(): void {
        const { tableData, values: formValues = {} } = this.props;
        const { parents = [] } = tableData ?? {};

        //get current structure object from form data
        const valuesObject = parents?.length
            ? parents.reduce((resultValue: Dict<any>, parentItem: string) => {
                return resultValue[parentItem];
            }, formValues)
            : formValues;

        const { cols: colsString, rows: rowsString, values: valuesString } = valuesObject;
        let [cols, rows, values] = [colsString ?? '', rowsString ?? '', valuesString ?? '']
            .map(stringData => this.getArrayFromString(stringData));

        const matrixArea = cols.length * rows.length;
        let valuesAreaError: Error | null = null;

        if (matrixArea !== values.length) {
            valuesAreaError = new Error(`Значений слишком ${matrixArea < values.length ? 'много' : 'мало'}`);

            //if not enough values, fill it with empty strings
            const emptyFullValues = Array(matrixArea).fill('');
            emptyFullValues.splice(0, values.length, ...values);
            values = emptyFullValues;
        }

        this.setState({
            cols,
            rows,
            values,
            absentFieldsError: this.getAbsentFieldsError(valuesObject),
            valuesAreaError,
        });
    }

    getAbsentFieldsError(valuesObject: Dict<any>): Error | null {
        const absentFields = [TABLE_FIELDS.ROWS, TABLE_FIELDS.COLS, TABLE_FIELDS.VALUES]
            .reduce((result: string[], key) => {
                if (!valuesObject.hasOwnProperty(key)) {
                    result.push(key);
                }

                return result;
            }, []);

        return absentFields.length
            ? new Error(`В данных таблицы ${absentFields.length > 1 ? 'отсутствуют поля' : 'отсутствует поле'} `
                + `${absentFields.join(', ')}`)
            : null;
    }

    getArrayFromString(data: string) {
        const splitData = data.split(DATA_SEPARATOR);

        return splitData.join('') === ''
            ? []
            : splitData.map(splitDataItem => splitDataItem === EMPTY_VALUE ? '' : splitDataItem);
    }

    getStringFromArray(data: string[]) {
        return data.map(dataItem => dataItem === '' ? EMPTY_VALUE : dataItem).join(DATA_SEPARATOR);
    }

    onValueChange(rowIndex: number, colIndex: number, value: string) {
        const { values, cols } = this.state;
        values[rowIndex * cols.length + colIndex] = value;
        this.setState({ values });
    }

    onDimensionChange(type: TABLE_FIELDS.COLS | TABLE_FIELDS.ROWS, index: number, value: string) {
        const isCols = type === TABLE_FIELDS.COLS;
        const dimensionArray = isCols ? this.state.cols : this.state.rows;
        dimensionArray[index] = value;

        if (isCols) {
            this.setState({ cols: dimensionArray });
        } else {
            this.setState({ rows: dimensionArray });
        }
    }

    getValuesRows() {
        const { cols, values, rows } = this.state;
        const colsLength = cols.length;

        return rows.reduce((result: string[][], row, rowIndex) => {
            const startRowIndex = rowIndex * colsLength;
            const valueRow = values.slice(startRowIndex, startRowIndex + colsLength);
            result.push(valueRow);

            return result;
        }, []);
    }

    addCol() {
        const { cols } = this.state;

        const newValues = this.getValuesRows().reduce((result: string[], valueRow) => {
            valueRow.push('');
            result.push(...valueRow);

            return result;
        }, []);
        cols.push('');

        this.setState({ cols, values: newValues });
    }

    addRow() {
        const { cols, values, rows } = this.state;
        const colsLength = cols.length;

        const newValues = [...values, ...Array(colsLength).fill('')];
        rows.push('');

        this.setState({ rows, values: newValues });
    }

    deleteDimension(type: TABLE_FIELDS.COLS | TABLE_FIELDS.ROWS, index: number) {
        const { cols, rows } = this.state;
        const isColumn = type === TABLE_FIELDS.COLS;

        this.setState({
            confirmIsOpen: true,
            question: `Удалить ${isColumn ? 'стоблец' : 'строку'} ${isColumn ? cols[index] : rows[index]}?`,
            accept: () => {
                const valuesRows = this.getValuesRows();

                if (isColumn) {
                    valuesRows.map(valuesRow => valuesRow.splice(index, 1));
                    cols.splice(index, 1);
                } else {
                    valuesRows.splice(index, 1);
                    rows.splice(index, 1);
                }

                const newValues = valuesRows.reduce((result: string[], valuesRow: string[]) => {
                    result.push(...valuesRow);

                    return result;
                }, []);

                this.setState({ cols, rows, values: newValues, confirmIsOpen: false });
            },
        });
    }

    closeConfirm() {
        this.setState({ confirmIsOpen: false });
    }

    onSave() {
        const { onSave } = this.props;
        const { cols, rows, values } = this.state;
        const [colsString, rowsString, valuesString] = [cols, rows, values]
            .map(data => this.getStringFromArray(data));

        onSave?.({ cols: colsString, rows: rowsString, values: valuesString });
    }

    render() {
        const { onClose } = this.props;
        const { absentFieldsError, valuesAreaError, cols, rows, values, confirmIsOpen, question, accept } = this.state;

        return <Window onClose={onClose?.bind(this)} title={'Настройка таблицы'}>
            <div className={style.table_config_modal}>
                {absentFieldsError
                    ? <SimpleError error={absentFieldsError} data={{ label: 'Ошибка при получении данных таблицы' }}/>
                    : null}
                {valuesAreaError
                    ? <SimpleError error={valuesAreaError} data={{ label: 'Ошибка при заполнении таблицы' }}/>
                    : null}
                <div className={style.table_container}>
                    <table>
                        <thead>
                            <tr>
                                <th/>
                                {cols.map((col, colIndex) => {
                                    return <th key={colIndex}>
                                        <DimensionCell type={TABLE_FIELDS.COLS}
                                                       value={col}
                                                       index={colIndex}
                                                       deleteDimension={this.deleteDimension.bind(this)}
                                                       onDimensionChange={this.onDimensionChange.bind(this)}/>
                                    </th>;
                                })}
                                <th>
                                    <div className={style.add_col_container}>
                                        <Link className={style.add_dimension}
                                              onClick={this.addCol.bind(this)}>
                                            Добавить
                                        </Link>
                                    </div>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            {rows.map((row, rowIndex) => {
                                const colsLength = cols.length;
                                const startRowIndex = rowIndex * colsLength;
                                const rowValues = values.slice(startRowIndex, startRowIndex + colsLength);

                                const inputs: React.ReactElement[] = [];
                                for (let colIndex = 0; colIndex < colsLength; colIndex++) {
                                    const rowValue = rowValues[colIndex];

                                    inputs.push(<td key={`${rowIndex}_${colIndex}`}>
                                        <Input placeholder={`${cols[colIndex]}:${rows[rowIndex]}`}
                                               value={rowValue}
                                               className={style.table_cell_input}
                                               onChange={this.onValueChange.bind(this, rowIndex, colIndex)}/>
                                    </td>);
                                }

                                return <tr key={rowIndex}>
                                    <td>
                                        <DimensionCell type={TABLE_FIELDS.ROWS}
                                                       value={row}
                                                       index={rowIndex}
                                                       deleteDimension={this.deleteDimension.bind(this)}
                                                       onDimensionChange={this.onDimensionChange.bind(this)}/>
                                    </td>
                                    {inputs}
                                </tr>;
                            })}
                            <tr>
                                <td>
                                    <div className={style.add_row_container}>
                                        <Link className={style.add_dimension}
                                              onClick={this.addRow.bind(this)}>
                                            Добавить
                                        </Link>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                <div className={style.controls}>
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onClose.bind(this)}/>
                        <SaveButton onClick={this.onSave.bind(this)}/>
                    </div>
                </div>
            </div>
            {confirmIsOpen
                ? <Confirm error={null}
                           question={question}
                           onClose={this.closeConfirm.bind(this)}
                           accept={accept}/>
                : null}
        </Window>;
    }
}

interface IDimensionCellProps {
    type: TABLE_FIELDS.COLS | TABLE_FIELDS.ROWS;
    index: number;
    value: string;
    onDimensionChange: () => {};
    deleteDimension: () => {};
}

const DimensionCell = React.memo((props: IDimensionCellProps) => {
    const { index, value, type, onDimensionChange, deleteDimension } = props;
    const isColumn = type === TABLE_FIELDS.COLS;

    return <div className={style.row_col_cell}>
        <Input placeholder={`${isColumn ? 'Столбец' : 'Строка'} ${index + 1}`}
               value={value}
               className={style.table_cell_input}
               onChange={onDimensionChange.bind(null, isColumn ? TABLE_FIELDS.COLS : TABLE_FIELDS.ROWS, index)}/>
        <div className={style.delete_cross_container}>
            <Cross className={style.delete_cross}
                   onClick={deleteDimension.bind(null, isColumn ? TABLE_FIELDS.COLS : TABLE_FIELDS.ROWS, index)}/>
        </div>
    </div>;
});
