import * as React from 'react';
import Scrollbars from 'react-custom-scrollbars';

import { Dict, ICar } from '../../../../types';
import { CITIES, EMPTY_DATA } from '../../../constants';
import CarInfo, { CarInfoHandler } from '../../../models/car';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import CarNumber from '../../CarNumber';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { FUELING_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface IFuelTrucksState {
    trucks: IFuelTruck[];
    error: Error | null;
    selectCityOptions: IOptionInfo[];
    selectStatusOptions: IOptionInfo[];
    selectModelOptions: IOptionInfo[];
    isLoading: boolean;
    filters: Dict<string[]>;
}

interface IFuelTrucksProps {

}

interface IFuelTruck {
    isTankerMajor: boolean;
    id: string;
    number: string;
    model: string;
    status: string;
    city: string;
    max_aux_fuel_volume: number;
    max_aux_fuel_volume_1: number;
    aux_fuel_level_1?: string;
    aux_fuel_level_2?: string;
    mileage: number;
    maintenance_mileage?: number;
}

interface IModelItem {
    manufacturer: string;
    short_name: string;
}

const CITIES_KEYS = Object.keys(CITIES);
const TELEMATICS_CONFIGURATION = 'telematics_configuration';
const AUX_FUEL_LEVEL_1 = 'aux_fuel_level_1';
const AUX_FUEL_LEVEL_2 = 'aux_fuel_level_2';
const CLASS_REFUELLER = 'class_refueller';

const DefaultMileage = {
    WARNING: 14500,
    NEGATIVE: 14800,
    NEGATIVE_DARK: 15000,
};

const AnotherMileage = {
    WARNING: 9500,
    NEGATIVE: 9800,
    NEGATIVE_DARK: 10000,
};

enum FilterKeys {
    CITY = 'city',
    STATUS = 'status',
    MODEL = 'model',
}

const FRACTION_DIGITS = 2;

export class FuelTrucks extends React.Component<IFuelTrucksProps, IFuelTrucksState> {
    state: IFuelTrucksState = {
        trucks: [],
        error: null,
        selectCityOptions: [],
        selectStatusOptions: [],
        selectModelOptions: [],
        isLoading: true,
        filters: {},
    };

    request = new Request2({ requestConfigs: FUELING_REQUESTS });
    models: IModelItem[] = [];
    statuses = new Set();
    trucks: IFuelTruck[] = [];

    filterByKey(currentKey, currentArgs) {
        let trucks = this.trucks;

        Object.values(FilterKeys).forEach((key) => {
            if (currentKey === key && currentArgs && currentArgs[0]) {
                trucks = trucks.filter((truck) => currentArgs.includes(truck[key]));
            } else if (currentKey !== key && this.state.filters[key] && this.state.filters[key][0]) {
                trucks = trucks.filter((truck) => this.state.filters[key].includes(truck[key]));
            }
        });

        return trucks;
    }

    filterData(key, args) {
        this.setState((prev) => {
            return {
                filters: {
                    ...prev.filters,
                    [key]: args,
                },
                trucks: this.filterByKey(key, args),
            };
        });
    }

    writeDataFromCarInfo(car) {
        const id = car.id;
        let city = EMPTY_DATA;
        const number = CarInfoHandler.getNumber.call(car);
        const model = CarInfoHandler.getModelNameManufacturer.call(car);
        const status = CarInfoHandler.getStatus.call(car);

        const mileage = CarInfoHandler.getMileage.call(car);

        const telematics_configuration = car.tags.find((tag) => tag.tag === TELEMATICS_CONFIGURATION);
        const max_aux_fuel_volume = telematics_configuration && telematics_configuration.max_aux_fuel_volume;
        const max_aux_fuel_volume_1 = telematics_configuration && telematics_configuration.max_aux_fuel_volume_1;
        const isTankerMajor = car.tags.some(tag => tag.tag === 'tanker_major_leasing');

        if (car.location && car.location.tags) {
            car.location.tags.forEach((location) => {
                if (CITIES_KEYS.includes(location)) {
                    city = location;
                }
            });
        }

        return {
            mileage,
            id,
            number,
            model,
            status,
            city,
            max_aux_fuel_volume,
            max_aux_fuel_volume_1,
            isTankerMajor,
        };
    }

    static writeDataFromTelematics(data) {
        if (data.sensors) {
            const { sensors } = data;
            let tank_levelObj = sensors && sensors.find((record) => record.name === AUX_FUEL_LEVEL_1);
            const aux_fuel_level_1 = tank_levelObj && `${tank_levelObj.value}/${tank_levelObj.calibrated?.toFixed(FRACTION_DIGITS)}`;

            tank_levelObj = sensors && sensors.find((record) => record.name === AUX_FUEL_LEVEL_2);
            const aux_fuel_level_2 = tank_levelObj && `${tank_levelObj.value}/${tank_levelObj.calibrated?.toFixed(FRACTION_DIGITS)}`;

            return {
                aux_fuel_level_1,
                aux_fuel_level_2,
            };
        }

        return;
    }

    async getData() {
        const response = await this.request.exec(REQUESTS.GET_CARS, {
            queryParams: {
                tags_filter: CLASS_REFUELLER,
            },
        });

        const trucks = response.cars;
        this.models = response.models;

        const rawTrucks = await Promise.all(trucks.map((truck) => {
            this.statuses.add(truck.status);

            const promises = [
                this.request.exec(REQUESTS.GET_CAR_INFO, {
                    queryParams: {
                        car_id: truck.id,
                    },
                }),
                this.request.exec(REQUESTS.GET_TELEMATICS, {
                    queryParams: {
                        car_id: truck.id,
                    },
                }),
            ].map((promise) => promise.catch((error) => error));

            return Promise.all(promises);
        }));

        this.trucks = await Promise.all(rawTrucks.map(async (responses: [ICar, any]) => {
            const mileage = await this.request.exec(REQUESTS.GET_MAINTENANCE, {
                queryParams: {
                    vin: responses[0].vin,
                },
            }).catch((error) => error);

            return {
                ...this.writeDataFromCarInfo(responses[0]),
                ...FuelTrucks.writeDataFromTelematics(responses[1]),
                maintenance_mileage: mileage && mileage.maintenance
                    && mileage.maintenance[0] && mileage.maintenance[0].mileage,
            };
        }));

        this.setFiltersOptions();
        this.setState({
            trucks: this.trucks,
            isLoading: false,
        });
    }

    setFiltersOptions() {
        const selectCityOptions: IOptionInfo[] = [];
        CITIES_KEYS.forEach((city) => {
            selectCityOptions.push({
                text: CITIES[city].name,
                value: city,
            });
        });
        selectCityOptions.push({
            text: 'Не указан',
            value: EMPTY_DATA,
        });

        const selectModelOptions: IOptionInfo[] = [];

        for (const model in this.models) {
            selectModelOptions.push({
                text: `${this.models[model].manufacturer} ${this.models[model].short_name}`,
                value: `${this.models[model].manufacturer} ${this.models[model].short_name}`,
            });
        }

        const selectStatusOptions: IOptionInfo[] = [];
        this.statuses.forEach((status: string) => {
            selectStatusOptions.push({
                text: status,
                value: status,
            });
        });

        this.setState({
            selectCityOptions,
            selectModelOptions,
            selectStatusOptions,
        });
    }

    componentDidMount() {
        this.getData();
        this.setFiltersOptions();
    }

    render() {
        const { error, trucks, selectCityOptions, selectModelOptions, selectStatusOptions, isLoading } = this.state;

        return (
            <>
                {error ? <SimpleError error={error}/> : isLoading ? <Spin/> : (
                    <>
                        <div className={styles.filters}>
                            <Select options={selectCityOptions}
                                    placeholder={'Фильтрация по городу'}
                                    onSelect={this.filterData.bind(this, FilterKeys.CITY)}
                                    multiSelect/>
                            <Select options={selectStatusOptions}
                                    placeholder={'Фильтрация по статусу машины'}
                                    onSelect={this.filterData.bind(this, FilterKeys.STATUS)}
                                    multiSelect/>
                            <Select options={selectModelOptions}
                                    placeholder={'Фильтрация по модели машины'}
                                    onSelect={this.filterData.bind(this, FilterKeys.MODEL)}
                                    multiSelect/>
                        </div>
                        {trucks.length
                            ? <Scrollbars style={{ maxHeight: 'calc(100vh - 220px)' }}>
                                <table className={`${tblStyle.table} ${styles.table}`}>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Госномер</th>
                                            <th>Модель, марка</th>
                                            <th>Статус</th>
                                            <th>Город</th>
                                            <th>Текущий пробег</th>
                                            <th>Пробег на ТО</th>
                                            <th>Пробег после ТО</th>
                                            <th>ДУТ1 / лит.</th>
                                            <th>Объём 1 бака</th>
                                            <th>ДУТ2 / лит.</th>
                                            <th>Объём 2 бака</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {
                                            trucks.map((truck, index) => {
                                                return (
                                                    <FuelTruckRow truck={truck}
                                                                  key={index}
                                                                  index={index + 1}/>
                                                );
                                            })
                                        }
                                    </tbody>
                                </table>
                            </Scrollbars>
                            : <NoInformation/>}
                    </>
                )}
            </>
        );
    }
}

interface IFuelTruckRowProps {
    truck: IFuelTruck;
    index: number;
}

const FuelTruckRow = React.memo((props: IFuelTruckRowProps) => {
    const { truck, index } = props;

    const lastMileage = truck.maintenance_mileage
        ? truck.mileage && truck.mileage - truck.maintenance_mileage
        : truck.mileage;
    const car: typeof CarInfo = deepCopy(truck || {});
    const lastMileageLabel = getFuelLabel(car, lastMileage);

    return (
        <tr>
            <td>{index}</td>
            <td>
                {truck.number !== EMPTY_DATA ? (
                    <CarNumber carInfo={car}/>
                ) : EMPTY_DATA}
            </td>
            <td>
                <Link href={`#/cars/${truck.id}/info`} target="_blank">{truck.model}</Link>
            </td>
            <td>{truck.status || EMPTY_DATA}</td>
            <td>{CITIES[truck.city] && CITIES[truck.city].name || EMPTY_DATA}</td>
            <td>{truck.mileage ? truck.mileage : EMPTY_DATA}</td>
            <td>{truck.maintenance_mileage ? truck.maintenance_mileage : EMPTY_DATA}</td>
            <td>{lastMileageLabel ? lastMileageLabel : truck.mileage ? truck.mileage : EMPTY_DATA}</td>
            <td>{truck.aux_fuel_level_1 || EMPTY_DATA}</td>
            <td>{truck.max_aux_fuel_volume || EMPTY_DATA}</td>
            <td>{truck.aux_fuel_level_2 || EMPTY_DATA}</td>
            <td>{truck.max_aux_fuel_volume_1 || EMPTY_DATA}</td>
        </tr>
    );
});

function getFuelLabel(car, mileage) {
    const MILEAGE_LABELS = car.isTankerMajor ? DefaultMileage : AnotherMileage;

    if (mileage) {
        return mileage > MILEAGE_LABELS.WARNING && mileage < MILEAGE_LABELS.NEGATIVE
            ? <TLabel text={mileage.toString()} status={LabelStatus.WARNING}/>

            : mileage > MILEAGE_LABELS.WARNING && mileage < MILEAGE_LABELS.NEGATIVE_DARK
                ? <TLabel text={mileage.toString()} status={LabelStatus.NEGATIVE}/>

                : mileage > MILEAGE_LABELS.NEGATIVE_DARK
                    ? <TLabel text={mileage.toString()} status={LabelStatus.NEGATIVE_DARK}/>

                    : mileage;
    }

    return EMPTY_DATA;
}
