import produce from 'immer';
import moment from 'moment';
import React from 'react';

import { ISearchControls, WithSearchControls } from '../../../../decorators/SearchControls';
import { ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import FormatDate from '../../../ui/FormatDate';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import CarNumber from '../../CarNumber';
import { Copy } from '../../Copy';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { FUELING_REQUESTS, REQUESTS } from '../request';
import * as style from './index.css';

type IFuelinfB2CProps = ISearchControls

interface IFuelingDataItem {
    objects: {};
    sessions: {}[];
    tails: [];
    users: {};
}

interface IFuelingB2CState {
    isLoading: boolean;
    data: IFuelingDataItem;
    error: any;
    infoIsShowing: boolean;
    selectedItem: number;
    sessions: any[];
}

const DAYS_AGO = 90;

@WithSearchControls({ since: moment(new Date()).subtract(DAYS_AGO, 'd') })
export class FuelingB2C extends React.Component<IFuelinfB2CProps, IFuelingB2CState> {
    state: IFuelingB2CState = {
        isLoading: false,
        data: {} as IFuelingDataItem,
        error: null,
        infoIsShowing: false,
        selectedItem: 0,
        sessions: [],
    };
    request = new Request2({ requestConfigs: FUELING_REQUESTS });

    getData(props) {
        this.setState(produce(this.state, draft => {
            draft.isLoading = true;
        }), () => {
            this.request.exec(REQUESTS.GET_DATA, {
                queryParams: {
                    since: props.since,
                    until: props.until,
                },
            })
                .then(response => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.data = response;
                        draft.error = null;
                        draft.sessions = response
                            && response.sessions
                            && Array.isArray(response.sessions)
                            && response.sessions.reverse() || [];
                    }));
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.error = error;
                    }));
                });
        });
    }

    componentDidMount(): void {
        this.getData(this.props);
    }

    UNSAFE_componentWillReceiveProps(nextProps: Readonly<IFuelinfB2CProps>, nextContext: any): void {
        if (nextProps.since !== this.props.since || nextProps.until !== this.props.until) {
            this.getData(nextProps);
        }

    }

    showInfo(state: boolean, index: number) {
        this.setState(produce(this.state, draft => {
            draft.infoIsShowing = state;
            draft.selectedItem = index;
        }));
    }

    render() {
        return <div className={style.component}>
            {
                this.state.infoIsShowing
                && <JsonModal onClose={this.showInfo.bind(this, false)}
                              title={'Сырые данные по заправочной сессии'}
                              obj={this.state.sessions[this.state.selectedItem]}/>
            }
            {
                this.state.isLoading
                    ? <Spin/>
                    : this.state.error
                        ? <SimpleError error={this.state.error}/>
                        : <div className={style.content}>
                            <table className={tblStyle.table}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>дата</th>
                                        <th>клиент</th>
                                        <th/>
                                        <th>статус</th>
                                        <th>₽/сервис</th>
                                        <th>₽/литр</th>
                                        <th>депозит</th>
                                        <th>доступ</th>
                                        <th>исполнитель</th>
                                        <th/>
                                    </tr>
                                </thead>
                                <tbody>
                                    {
                                        this.state.sessions
                                            .map((session, index) => {
                                                return <FuelSessionItem key={index}
                                                                        session={session}
                                                                        index={index}
                                                                        showInfo={this.showInfo.bind(this, true, index)}
                                                                        data={this.state.data}/>;
                                            })
                                    }
                                </tbody>
                            </table>
                        </div>
            }
        </div>;
    }
}

const FuelSessionItem = React.memo((props: { session; index; data; showInfo }) => {
    let { index, session, data, showInfo } = props;
    const events = session.events || [];
    const object = events[0] && events[0].object_id && data && data.objects && data.objects[events[0].object_id] || {};
    const car = object.Object && object.Object.ExternalCar || {};
    const model = car.Model;
    const owner = data.users && data.users[car.Owner];
    const lastTagDetails = events[events.length - 1] && events[events.length - 1].tag_details || {};
    const performer = data.users && data.users[lastTagDetails.performer];
    const status = lastTagDetails.FuelingResult;
    const liters = lastTagDetails.LitersResult;
    const fuelingOffer = lastTagDetails.FuelingOffer && lastTagDetails.FuelingOffer.FuelingOffer || {};
    let tlabelStatus = LabelStatus.DEFAULT;
    switch (status) {
    case 'Success':
        tlabelStatus = LabelStatus.POSITIVE;
        break;
    case 'CannotFindCar':
        tlabelStatus = LabelStatus.WARNING;
        break;
    case 'OtherProblem':
        tlabelStatus = LabelStatus.NEGATIVE;
        break;
    default:
        tlabelStatus = LabelStatus.DEFAULT;
        break;
    }

    return <tr>
        <td>{++index}</td>
        <td className={style.date}>
            <div><strong>{events[0].action}:</strong> <FormatDate value={events[0].timestamp * ONE_SECOND}
                                                                  withSecond/></div>
            <div>
                <strong>
                    {events[events.length - 1].action}:
                </strong> <FormatDate value={events[events.length - 1].timestamp * ONE_SECOND}
                                      withSecond/>
            </div>
        </td>
        <td>
            <Link href={`${location.origin}/#/clients/${car.Owner}/info`} target={'_blank'}>
                {UserInfoHandler.getPrintNameWithoutPn.call(owner)}
            </Link>
        </td>
        <td>
            <div><strong className={style.model}>{model}:</strong> {car.FuelTypes}</div>
            <div><CarNumber value={car.Number}/></div>

        </td>
        <td>{status &&
        <TLabel className={style.tlabel} status={tlabelStatus} text={`${status} ${liters ? `: ${liters}l.` : ''}`}/>
        }
        <div className={style.phone}><Copy internalTooltipId={`fuel_offer_phone_${index}`}>{fuelingOffer.PhoneNumber}</Copy></div>
        </td>
        <td>{fuelingOffer.ServicePrice}</td>
        <td>{fuelingOffer.LiterPrice}</td>
        <td>{fuelingOffer.LitersDeposit}</td>
        <td><TLabel status={fuelingOffer.AccessMethod == 'by_phone' ? LabelStatus.DEFAULT : LabelStatus.INFO}
                    text={fuelingOffer.AccessMethod}/>
        </td>
        <td>
            <Link href={`${location.origin}/#/clients/${lastTagDetails.performer}/info`} target={'_blank'}>
                {UserInfoHandler.getPrintNameWithoutPn.call(performer)}
            </Link>
        </td>
        <td>
            <Link onClick={showInfo}>events</Link>
        </td>
    </tr>;
});
