import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';
import XLSX from 'xlsx';

import { CITIES, EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { useWindowResize } from '../../../hooks/useWindowResize';
import { RoleItem } from '../../../models/role';
import { UserInfoHandler } from '../../../models/user';
import { Button } from '../../../ui/Button';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { AccountDataType } from '../../Settings/Wallets/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { FUELING_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface IRefueler {
    id: string;
    printName: string;
    tag: string;
    status: string;
    city: IOptionInfo;
    shift: IOptionInfo;
    contract: string;
    client_id: string;
    billing_status: string;
    comment: string;
}

interface IRefuelersProps {

}

interface IRefuelersState {
    [x: number]: any;
    data: IRefueler[];
    isLoading: boolean;
    error: Error | null;
    cities: IOptionInfo[];
    shifts: IOptionInfo[];
    citiesOptions: IOptionInfo[];
    shiftsOptions: IOptionInfo[];
}

const REFUELER_TAGS = 'user_refueling_refueller,user_refueling_old,' +
    'user_refueling_candidate,user_refueling_active_refueler';

const ACTIVE_REFUELER = 'user_refueling_active_refueler';
const DAY_REFUELER = 'GR_refueler_day';
const NIGHT_REFUELER = 'GR_refueler_night';
const CITIES_KEYS = Object.keys(CITIES);
const CITIES_TAGS = {
    'user_location_kazan': 'kazan_area',
    'user_location_moscow': 'msc_area',
    'user_location_spb': 'spb_area',
};
const CITIES_TAGS_KEYS = Object.keys(CITIES_TAGS);

export class Refuelers extends React.Component<IRefuelersProps, IRefuelersState> {
    state = {
        data: [],
        isLoading: false,
        error: null,
        cities: [],
        shifts: [],
        citiesOptions: [],
        shiftsOptions: [],
    };

    request = new Request2({ requestConfigs: FUELING_REQUESTS });

    getData() {
        this.setState({
            isLoading: true,
        }, async () => {
            let response;

            try {
                response = await this.request.exec(REQUESTS.SEARCH_TAGS, {
                    queryParams: {
                        has_one_of: REFUELER_TAGS,
                        limit: 1000,
                    },
                });
            } catch(error) {
                this.setState({
                    error,
                    isLoading: false,
                });

                return;
            }

            const initialUsers = response
                && response.results
                && response.results[0]
                && response.results[0].report
                && response.results[0].report.slice() || [];
            const initialRolesRequests: Promise<any>[] = [];
            const initialTagsRequests: Promise<any>[] = [];
            const initialBillingRequests: Promise<any>[] = [];

            initialUsers.forEach((refueler) => {
                initialRolesRequests.push(
                    this.request.exec(REQUESTS.GET_USER_ROLES, { queryParams: {
                        user_id: refueler.id,
                    },
                    }),
                );
                initialTagsRequests.push(
                    this.request.exec(REQUESTS.GET_USER_TAGS, { queryParams: {
                        object_id: refueler.id,
                    },
                    }),
                );
                initialBillingRequests.push(
                    this.request.exec(REQUESTS.GET_USER_ACCOUNTS, { queryParams: {
                        user_id: refueler.id,
                    },
                    }),
                );
            });

            let initialTagsResponse;
            let initialRolesResponse;
            let initialBillingResponse;

            try {
                initialTagsResponse = await Promise.all(initialTagsRequests);
                initialRolesResponse = await Promise.all(initialRolesRequests);
                initialBillingResponse = await Promise.all(initialBillingRequests);
            } catch(error) {
                this.setState({
                    error,
                    isLoading: false,
                });

                return;
            }

            const data: IRefueler[] = initialUsers.map((refueler, index) => {
                const role = initialRolesResponse
                    && initialRolesResponse[index]
                    && initialRolesResponse[index].report
                    && initialRolesResponse[index].report.find((role: RoleItem) => {
                        return role.role_id === DAY_REFUELER || role.role_id === NIGHT_REFUELER;
                    });
                const status = role && role.active ? 'Активный' : role ? 'Заблокирован' : EMPTY_DATA;
                const shift = refueler.tags && refueler.tags[0] && refueler.tags[0].tag === ACTIVE_REFUELER ?
                    { text: 'Активный техник', value: refueler.tags[0].tag } :
                    role && role.role_info && role.role_info.role_description ?
                        { text: role.role_info.role_description, value: role.role_id } : '';

                const account = initialBillingResponse
                    && initialBillingResponse[index]
                    && initialBillingResponse[index].accounts.find((account) => {
                        return account.data_type === AccountDataType.SALARY;
                    });

                const contract = account && account.details && account.details.contract_id;
                const client_id = account && account.details && account.details.client_id;
                const billing_status = account && account.is_active ? 'Активный' : 'Заблокирован';

                let tag;
                let city = { text: EMPTY_DATA, value: EMPTY_DATA };

                initialTagsResponse
                && initialTagsResponse[index]
                && initialTagsResponse[index].records
                && initialTagsResponse[index].records.forEach((userTag) => {
                    const tagValue = userTag.tag;
                    if (tagValue === refueler.tags[0].tag) {
                        tag = userTag;
                    }

                    if (CITIES_TAGS_KEYS.includes(tagValue)) {
                        const cityValue = CITIES_TAGS[tagValue];
                        city = {
                            text: CITIES[cityValue].name,
                            value: cityValue,
                        };
                    }
                });

                return {
                    id: refueler.id,
                    printName: UserInfoHandler.getPrintNameWithoutUserName.call(refueler),
                    status,
                    city,
                    shift,
                    contract,
                    client_id,
                    billing_status,
                    comment: tag && tag.comment,
                    tag: tag && tag.display_name,
                };
            });
            this.setFiltersOptions();
            this.setState({
                data: data.sort((a, b) => a.printName.localeCompare(b.printName)),
                isLoading: false,
            });
        });
    }

    setFiltersOptions() {
        const citiesOptions: IOptionInfo[] = [];
        CITIES_KEYS.forEach((city) => {
            citiesOptions.push({
                text: CITIES[city].name,
                value: city,
            });
        });
        citiesOptions.push({
            text: 'Не указан',
            value: EMPTY_DATA,
        });

        const shiftsOptions: IOptionInfo[] = [
            { text: 'Заправщик дневной', value: DAY_REFUELER },
            { text: 'Заправщик ночной', value: NIGHT_REFUELER },
            { text: 'Активный техник', value: ACTIVE_REFUELER },
            { text: 'Не указан', value: EMPTY_DATA },
        ];
        this.setState({
            citiesOptions,
            shiftsOptions,
        });
    }

    generateTable() {
        const { data } = this.state;
        const wb = XLSX.utils.book_new();
        const ws_data: any = [];
        ws_data.push(
            [
                '#',
                'ФИО',
                'Тег',
                'Статус',
                'Город',
                'Смена',
                'Номер договора',
                'Client ID',
                'Статус кошелька',
                'Комментарий',
            ],
            ...data.map((refueler: IRefueler, index) => {
                return [
                    index + 1,
                    refueler.printName,
                    refueler.tag || EMPTY_DATA,
                    refueler.status || EMPTY_DATA,
                    refueler.city.text || EMPTY_DATA,
                    refueler.shift.text || EMPTY_DATA,
                    refueler.contract || EMPTY_DATA,
                    refueler.client_id || EMPTY_DATA,
                    refueler.billing_status || EMPTY_DATA,
                    refueler.comment || EMPTY_DATA,
                ];
            }));

        const ws = XLSX.utils.aoa_to_sheet(ws_data);

        XLSX.utils.book_append_sheet(wb, ws, 'Заправщики');
        XLSX.writeFile(wb, `Заправщики-${Math.round(Date.now() / ONE_SECOND)}.xlsx`);
    }

    onSelect(key, value) {
        this.setState({
            [key]: value,
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { isLoading, data, cities, shifts, citiesOptions, shiftsOptions, error } = this.state;

        return (
            <div>
                {error ? <SimpleError error={error}/> : isLoading ? <Spin /> : (
                    <>
                        <div className={styles.filters}>
                            <Select options={citiesOptions}
                                    initialValues={cities}
                                    placeholder={'Фильтрация по городу'}
                                    onSelect={this.onSelect.bind(this, 'cities')}
                                    multiSelect/>
                            <Select options={shiftsOptions}
                                    initialValues={shifts}
                                    placeholder={'Фильтрация по смене'}
                                    onSelect={this.onSelect.bind(this, 'shifts')}
                                    multiSelect/>
                            <Button onClick={this.generateTable.bind(this)}>Сохранить весь список в .xlsx</Button>
                        </div>
                        {data.length
                            ? <RefuelersTable data={data}
                                              filter={{ cities, shifts }}/>
                            : <NoInformation/>}
                    </>
                )}
            </div>
        );
    }
}

const LIST_PADDING = 257;

const RefuelersTable = React.memo((props: {data: any; filter: {'cities': any; 'shifts': any}}) => {
    let { data, filter } = props;
    const windowSize = useWindowResize();

    data = data.filter((refueler) => {
        if (filter.shifts.length) {
            return filter.shifts.some((filter) => refueler.shift.value === filter);
        }

        if (filter.cities.length) {
            return filter.cities.some((filter) => refueler.city.value === filter);
        }

        return true;
    });

    return (
        <div className={styles.content}>
            <div className={`${styles.header} ${styles.row}`}>
                <span>#</span>
                <span>ФИО</span>
                <span>Тег</span>
                <span>Статус</span>
                <span>Город</span>
                <span>Смена</span>
                <span>Номер договора</span>
                <span>Client ID</span>
                <span>Статус кошелька</span>
                <span>Комментарий</span>
            </div>
            <VirtualList width={'100%'}
                         height={windowSize.height - LIST_PADDING}
                         itemCount={data.length}
                         itemSize={60}
                         renderItem={({ index, style }) => {
                             const item = data[index];

                             return (
                                 <div style={style}
                                      className={styles.row}
                                      key={index}>
                                     <span>{index + 1}</span>
                                     <span><Link href={`#/clients/${item.id}/info`}>{item.printName}</Link></span>
                                     <span>{item.tag || EMPTY_DATA}</span>
                                     <span>{item.status || EMPTY_DATA}</span>
                                     <span>{item.city.text}</span>
                                     <span>{item.shift.text || EMPTY_DATA}</span>
                                     <span>{item.contract || EMPTY_DATA}</span>
                                     <span>{item.client_id || EMPTY_DATA}</span>
                                     <span>{item.billing_status || EMPTY_DATA}</span>
                                     <span title={item.comment || ''}
                                           className={styles.comment}>
                                         {item.comment || EMPTY_DATA}
                                     </span>
                                 </div>
                             );
                         }}/>
        </div>
    );
});
