import moment from 'moment';
import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';
import XLSX from 'xlsx';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import CarInfo from '../../../models/car';
import { Button } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select, { IOptionInfo } from '../../../ui/Select';
import * as styleTable from '../../../ui/Table/index.css';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { makeKirCarNumber, makeLatCarNumber } from '../../../utils/utils';
import CarNumber from '../../CarNumber';
import { ExcelDownloader, ExcelUploader, IJSONData, ISheetData } from '../../ExcelUploader';
import * as style from './index.css';
import { REQUEST, REQUESTS } from './request';

const CAR_NUMBER_LENGTH = 9;

const AIRPORTS = {
    AER: { display_name: 'Адлер - Сочи' },
    DME: { display_name: 'Домодедово' },
    VKO: { display_name: 'Внуково' },
    SVO: { display_name: 'Шереметьево' },
    KZN: { display_name: 'Казань' },
    LED: { display_name: 'Пулково' },
};

const AIRPORTS_OPTIONS: IOptionInfo[] = Object.entries(AIRPORTS)
    .map((airportEntry: [string, { display_name: string }]) => {
        const [code, description] = airportEntry;

        return { value: code, text: description?.display_name ?? '' };
    });
const MONTH_SHORT = ['ЯНВ', 'ФЕВ', 'МАР', 'АПР', 'МАЙ', 'ИЮН', 'ИЮЛ', 'АВГ', 'СЕН', 'ОКТ', 'НОЯ', 'ДЕК'];
const MONTH_SHORT_OPTIONS: IOptionInfo[] = MONTH_SHORT.map(month => {
    return { value: month };
});

interface IAirportCarInfo {
    car_number: string;
    car_number_old?: string;
    car_number_kir?: string;
    date: string;
    isCarNumberValid?: { isValid: boolean; errors?: string[] };
    supposedNumbers?: Dict<any>;
}

interface ITableData {
    list: IAirportCarInfo[];
    errors: Dict<any>[];
}

interface IHahnCarInfo {
    car_number_lat: string;
    start_airport: string;
    date: string;
}

interface ITableDiff {
    common: IAirportCarInfo[];
    diff: { table1: IAirportCarInfo[]; table2: IAirportCarInfo[] };
}

interface IAirportsInfoComparatorProps {
}

interface IAirportsInfoComparatorState {
    carsList: Dict<Dict<any>[]>;
    tableType: string;
    month: string;
    tableData: ITableData;
    foundedCars: IAirportCarInfo[];
    notFoundedCars: IAirportCarInfo[];
    isCarListLoading: boolean;
    carListLoadingError: Error | null;
    isHahnTableLoading: boolean;
    hahnTableLoadingError: Error | null;
    notFoundedModalOpen: boolean;
    errorsModalOpen: boolean;
    hahnData: IHahnCarInfo[];
    compareResult: ITableDiff | null;
    airportData: IAirportCarInfo[] | null;
    isComparingTables: boolean;
}

export default class AirportsInfoComparator extends
    React.Component<IAirportsInfoComparatorProps, IAirportsInfoComparatorState> {
    state: IAirportsInfoComparatorState = {
        carsList: {},
        tableType: '',
        month: '',
        tableData: { list: [], errors: [] },
        foundedCars: [],
        notFoundedCars: [],
        isCarListLoading: false,
        carListLoadingError: null,
        isHahnTableLoading: false,
        hahnTableLoadingError: null,
        notFoundedModalOpen: false,
        errorsModalOpen: false,
        hahnData: [],
        compareResult: null,
        airportData: null,
        isComparingTables: false,
    };
    request = new Request2({ requestConfigs: REQUEST });

    componentDidMount() {
        this.setState({ isHahnTableLoading: true, hahnTableLoadingError: null }, () => {
            this.request.exec(REQUESTS.GET_HAHN_DATA, {
                queryParams: {
                    path: '//home/carsharing/production/projects/DRIVEANALYTICS_129/airport_leavings{"date","start_airport","car_number_lat"}',
                    ['output_format[$value]']: 'json',
                    ['output_format[$attributes][encode_utf8]']: false,
                    dump_error_into_response: true,
                },
            })
                .then(res => {
                    const hahnData = res.split('\n')
                        .filter(item => item)
                        .map(item => {
                            if (isValidJSONString(item)) {
                                return JSON.parse(item);
                            }

                            return item;

                        });

                    this.setState({ hahnData, isHahnTableLoading: false });
                })
                .catch(hahnTableLoadingError => {
                    this.setState({ isHahnTableLoading: false, hahnTableLoadingError });
                });
        });

        this.setState({ isCarListLoading: true }, () => {
            this.request.exec(REQUESTS.GET_CARS)
                .then(response => {
                    const carsArray = response?.cars ?? [];

                    const carsList = carsArray.reduce((result: Dict<any>, car) => {
                        const { number } = car;

                        const number_lat = makeLatCarNumber(number);

                        if (result[number_lat]) {
                            result[number_lat].push(car);
                        } else {
                            result[number_lat] = [car];
                        }

                        return result;
                    }, {});

                    this.setState({ carsList, isCarListLoading: false });
                })
                .catch(carListLoadingError => {
                    this.setState({ carListLoadingError, isCarListLoading: false });
                });
        });
    }

    async JSONHandler(JSONData: IJSONData[], workbook: XLSX.WorkBook) {
        const tableInfo = this.getInfoFromTable(workbook);
        const { carsList } = this.state;
        const { tableType, tableData } = tableInfo;

        const { foundedCars, notFoundedCars } = tableData.list
            .reduce((result: { foundedCars: IAirportCarInfo[]; notFoundedCars: IAirportCarInfo[] }, listItem) => {
                if (this.state.carsList?.[listItem.car_number]) {
                    result.foundedCars.push(listItem);
                } else {
                    result.notFoundedCars.push(listItem);
                }

                return result;
            }, { foundedCars: [], notFoundedCars: [] });

        await notFoundedCars.map(async notFoundedCar => {
            const { car_number } = notFoundedCar;
            if (car_number) {
                notFoundedCar.isCarNumberValid = AirportsInfoComparator.isCarNumberValid(car_number);
                Object.keys(carsList).forEach(carListNumber => {
                    if (car_number.length < CAR_NUMBER_LENGTH) {
                        if (carListNumber.includes(car_number)) {
                            if (notFoundedCar.isCarNumberValid && notFoundedCar.supposedNumbers) {
                                notFoundedCar.supposedNumbers[carListNumber] = carsList[carListNumber];
                            } else {
                                notFoundedCar.supposedNumbers = { [carListNumber]: carsList[carListNumber] };
                            }
                        }
                    }
                });

                if (car_number.length < CAR_NUMBER_LENGTH) {
                    Object.keys(carsList).forEach(carListNumber => {
                        if (carListNumber.includes(car_number)) {
                            if (notFoundedCar.isCarNumberValid && notFoundedCar.supposedNumbers) {
                                notFoundedCar.supposedNumbers[carListNumber] = carsList[carListNumber];
                            } else {
                                notFoundedCar.supposedNumbers = { [carListNumber]: carsList[carListNumber] };
                            }
                        }
                    });
                } else {
                    const carInfo = await this.request.exec(REQUESTS.SEARCH, {
                        queryParams: {
                            has_all_of: makeKirCarNumber(car_number),
                            limit: 100,
                            what: 'cars',
                        },
                    });

                    const founded = carInfo?.objects?.cars?.[0];

                    if (founded) {
                        const isFormerNumber = founded?.former_numbers
                            ?.find(formerNumber => formerNumber?.number === makeKirCarNumber(car_number));
                        if (isFormerNumber) {
                            if (notFoundedCar.supposedNumbers) {
                                notFoundedCar.supposedNumbers[founded.number] = [founded];
                            } else {
                                notFoundedCar.supposedNumbers = { [founded.number]: [founded] };
                            }
                        }
                    }
                }
            }

            return notFoundedCar;
        });

        const firstItemDate = tableData?.list?.[0]?.date ?? null;
        const momentDate = moment(firstItemDate, 'DD.MM.YYYY');
        const monthNumber = +momentDate.format('M');
        const month = MONTH_SHORT?.[monthNumber - 1] ?? '';

        this.setState({
            tableType: tableType === 'default' ? '' : tableType,
            month,
            tableData,
            foundedCars,
            notFoundedCars,
        });
    }

    getInfoFromTable(workbook: XLSX.WorkBook): { tableType: string; tableData: ITableData } {
        const sheets = workbook?.Sheets && Object.keys(workbook.Sheets) || [];
        let tableType = 'default';

        if (sheets[0] === 'выезды') {
            tableType = 'VKO';

            return { tableType, tableData: this.getVnukovoInfoFromTable(workbook) };
        }

        if (sheets[0] === 'ЯНДЕКС.ДРАЙВ') {
            tableType = 'DME';

            return { tableType, tableData: this.getDomodedovoInfoFromTable(workbook) };
        }

        if ((/_Yandex_/ig).test(sheets[0])) {
            tableType = 'LED';

            return { tableType, tableData: this.getPulkovoInfoFromTable(workbook) };
        }

        return { tableType, tableData: this.getInfoFromDefaultTable(workbook) };

    }

    getVnukovoInfoFromTable(workbook: XLSX.WorkBook): ITableData {
        const list: Dict<any>[] = XLSX.utils.sheet_to_json(workbook.Sheets['выезды'], { raw: false });

        return list.reduce((result: ITableData, listItem) => {
            try {
                const datetime = listItem?.['Время выезда'];
                const date = datetime.split(' ')[0];
                const car_number = listItem?.['Гос. номер']?.toLowerCase();

                result.list.push({
                    date,
                    car_number,
                });
            } catch (e) {
                result.errors.push(listItem);
            }

            return result;
        }, { list: [], errors: [] }) as ITableData;
    }

    getDomodedovoInfoFromTable(workbook: XLSX.WorkBook): ITableData {
        const list: Dict<any>[] = XLSX.utils.sheet_to_json(workbook.Sheets['ЯНДЕКС.ДРАЙВ'], { raw: false });

        const result = list.reduce((result, listItem) => {
            const { __EMPTY, __EMPTY_5 } = listItem;

            try {
                const car_number = __EMPTY_5?.replace(/\s|\|/ig, '').toLowerCase();

                result.list.push({
                    date: __EMPTY,
                    car_number,
                });
            } catch (e) {
                result.list.push(listItem);
            }

            return result;
        }, { list: [], errors: [] });

        result.list = result.list.slice(1);

        return result as ITableData;
    }

    getPulkovoInfoFromTable(workbook: XLSX.WorkBook): ITableData {

        const list: Dict<any>[] = XLSX.utils.sheet_to_json(Object.values(workbook.Sheets)[0], { raw: false });

        return list.reduce((result, listItem) => {
            try {
                const date = listItem?.['* Время действия'];
                const car_number = listItem?.['* Гос. номер']?.toLowerCase();

                result.list.push({
                    date,
                    car_number,
                });
            } catch (e) {
                result.errors.push(listItem);
            }

            return result;
        }, { list: [], errors: [] }) as ITableData;
    }

    getInfoFromDefaultTable(workbook: XLSX.WorkBook): ITableData {

        const list: Dict<any>[] = XLSX.utils.sheet_to_json(Object.values(workbook.Sheets)[0], { raw: false });

        return list.reduce((result, listItem) => {
            try {
                const date = listItem?.date;
                const car_number = listItem?.car_number?.toLowerCase();

                result.list.push({
                    date,
                    car_number,
                });
            } catch (e) {
                result.errors.push(listItem);
            }

            return result;
        }, { list: [], errors: [] }) as ITableData;
    }

    compareFileAndHahnData() {
        this.setState({ isComparingTables: true }, () => {
            const { foundedCars, hahnData, month, tableType } = this.state;
            const monthNumber = MONTH_SHORT.indexOf(month) + 1;

            const filteredHahnData = hahnData.filter(hahnDataItem => {
                const { date, start_airport } = hahnDataItem;

                const momentDate = moment(date);
                const hahnMonthNumber = +momentDate.format('M');

                if (hahnMonthNumber !== monthNumber) {
                    return false;
                }

                return hahnMonthNumber === monthNumber && start_airport.toLowerCase().includes(tableType.toLowerCase());

            }).map(hahnDataItem => {
                return {
                    date: hahnDataItem.date,
                    car_number: hahnDataItem.car_number_lat,
                };
            });

            const compareResult = this.compareTables(foundedCars, filteredHahnData);

            this.setState({ compareResult, isComparingTables: false });
        });
    }

    compareTables(table1: IAirportCarInfo[], table2: IAirportCarInfo[]): ITableDiff {
        const result: ITableDiff = {
            common: [],
            diff: { table1: [], table2: [] },
        };
        const carNumbersHash: Dict<boolean> = {};

        const table1Formatted = table1.map(table1Item => {
            const { car_number, date } = table1Item;

            return `${car_number.toLowerCase()}_${moment(date, 'DD.MM.YYYY').format('DD.MM.YYYY')}`;
        });

        const table2Formatted = table2.map(table2Item => {
            const { car_number, date } = table2Item;

            return `${car_number.toLowerCase()}_${moment(date).format('DD.MM.YYYY')}`;
        });

        const compare = (table1: string[], table2: string[], diffKey: string) => {
            table1.forEach(table1Item => {

                if (!carNumbersHash[table1Item]) {
                    const resultTable1Item = {
                        car_number: table1Item.split('_')[0],
                        date: table1Item.split('_')[1],
                    };

                    if (table2.filter(table2Item => {
                        return table2Item === table1Item;
                    })?.length) {
                        result.common.push(resultTable1Item);
                    } else {
                        result.diff[diffKey].push(resultTable1Item);
                    }

                    carNumbersHash[table1Item] = true;
                }

            });
        };

        compare(table1Formatted, table2Formatted, 'table1');
        compare(table2Formatted, table1Formatted, 'table2');

        return result;
    }

    static isCarNumberValid(carNumber: any): { isValid: boolean; errors?: string[] } {
        const validCarNumberRegExp = new RegExp(/^[A-Za-zа-яА-ЯёЁ]\d{3}[A-Za-zа-яА-ЯёЁ]{2}\d{3}$/ig);
        const carNumberRegExpWithoutRegion = new RegExp(/^[A-Za-zа-яА-ЯёЁ]\d{3}[A-Za-zа-яА-ЯёЁ]{2}\d{0,2}$/ig);
        const carNumberRegExpWrongOrderLettersFirst = new RegExp(/^[A-Za-zа-яА-ЯёЁ]{3}\d{3}\d{0,3}$/ig);
        const carNumberRegExpWrongOrderNumberFirst = new RegExp(/^\d{3}[A-Za-zа-яА-ЯёЁ]{3}\d{0,3}$/ig);

        if (typeof carNumber !== 'string') {
            return { isValid: false, errors: ['Номер не является строкой'] };
        }

        if (validCarNumberRegExp.test(carNumber)) {
            return { isValid: true };
        }

        const errors: string[] = [];

        if (carNumber.length !== CAR_NUMBER_LENGTH) {
            errors.push('Неверное число символов');
        }

        if (carNumberRegExpWithoutRegion.test(carNumber)) {
            errors.push('Отсутствует регион');
        }

        if (carNumberRegExpWrongOrderLettersFirst.test(carNumber)
                    || carNumberRegExpWrongOrderNumberFirst.test(carNumber)) {

            errors.push('Неверный порядок символов');
        }

        return { isValid: false, errors };

    }

    openNotFoundedModal() {
        this.setState({ notFoundedModalOpen: true });
    }

    closeNotFoundedModal() {
        this.setState({ notFoundedModalOpen: false });
    }

    openErrorsModal() {
        this.setState({ errorsModalOpen: true });
    }

    closeErrorsModal() {
        this.setState({ errorsModalOpen: false });
    }

    onChangeTableType(tableType: string) {
        this.setState({ tableType });
    }

    onChangeMonth(month: string) {
        this.setState({ month });
    }

    getCurrentAction() {
        const { isCarListLoading, isHahnTableLoading, isComparingTables } = this.state;

        const currentActions: string[] = [];

        if (isCarListLoading) {
            currentActions.push('загрузка списка авто');
        }

        if (isHahnTableLoading) {
            currentActions.push('загрузка таблицы из БД');
        }

        if (isComparingTables) {
            currentActions.push('сравнение таблиц');
        }

        return currentActions.length ? currentActions.join(', ') : <span className={style.waiting}>ожидание...</span>;
    }

    openAirportsDataModal(type: string) {
        const { compareResult } = this.state;
        let airportData: { car_number: string; date: string }[] = [];

        if (type === 'compared_common') {
            airportData = compareResult?.common ?? [];
        }

        if (type === 'compared_diff_table_1') {
            airportData = compareResult?.diff.table1 ?? [];
        }

        if (type === 'compared_diff_table_2') {
            airportData = compareResult?.diff.table2 ?? [];
        }

        this.setState({ airportData });
    }

    closeAirportsDataModal() {
        this.setState({ airportData: null });
    }

    downLoadFileCheck() {
        let { tableData, foundedCars, notFoundedCars } = this.state;
        const fileName = `Аэропорты отчёт проверка файла ${moment().format('DD.MM.YYYY HH:MM')}.xlsx`;

        const errorsKeys: string[] = tableData.errors.reduce((result: string[], errorRow) => {
            const keys = Object.keys(errorRow);
            keys.forEach(key => {
                if (!result.includes(key)) {
                    result.push(key);
                }
            });

            return result;
        }, []) as string[];

        const sheets: ISheetData[] = [];
        if (tableData.errors?.length) {
            sheets.push({
                sheetName: 'errors',
                columnNames: errorsKeys,
                rows: tableData.errors,
            });
        }

        if (foundedCars?.length) {
            foundedCars = foundedCars.map(foundedCar => {
                foundedCar.car_number_kir = makeKirCarNumber(foundedCar.car_number);

                return foundedCar;
            });
            sheets.push({
                sheetName: 'founded',
                columnNames: ['date', 'car_number', 'car_number_kir'],
                rows: foundedCars,
            });
        }

        if (notFoundedCars?.length) {
            notFoundedCars = notFoundedCars.map(notFoundedCar => {
                notFoundedCar.car_number_kir = makeKirCarNumber(notFoundedCar.car_number);

                return notFoundedCar;
            });
            sheets.push({
                sheetName: 'Not founded',
                columnNames: ['date', 'car_number', 'car_number_kir'],
                rows: notFoundedCars,
            });
        }

        ExcelDownloader.downloadFile({
            fileName,
            sheets,
        });
    }

    downLoadFileCompare() {
        const compareResult = this.state.compareResult as ITableDiff;
        const { common, diff } = compareResult;
        const { table1, table2 } = diff;

        const fileName = `Аэропорты отчёт сравнение ${moment().format('DD.MM.YYYY HH:MM')}.xlsx`;

        const sheets: ISheetData[] = [
            {
                sheetName: 'Common',
                columnNames: ['date', 'car_number', 'car_number_kir'],
                rows: common.map(item => {
                    item.car_number_kir = makeKirCarNumber(item.car_number);

                    return item;
                }),
            },
            {
                sheetName: 'Odd in File',
                columnNames: ['date', 'car_number', 'car_number_kir'],
                rows: table1.map(item => {
                    item.car_number_kir = makeKirCarNumber(item.car_number);

                    return item;
                }),
            },
            {
                sheetName: 'Odd in DB',
                columnNames: ['date', 'car_number', 'car_number_kir'],
                rows: table2.map(item => {
                    item.car_number_kir = makeKirCarNumber(item.car_number);

                    return item;
                }),
            },
        ];

        ExcelDownloader.downloadFile({
            fileName,
            sheets,
        });
    }

    moveToFounded(movingCarNumber: string, newCarInfo: Dict<any>) {
        const { foundedCars, notFoundedCars } = this.state;

        const movingCarIndex: number = notFoundedCars
            .map(notFoundedCar => notFoundedCar.car_number)
            .indexOf(movingCarNumber);
        if (movingCarIndex > -1) {
            foundedCars.push({
                car_number: newCarInfo?.number,
                date: notFoundedCars[movingCarIndex]?.date,
                car_number_old: movingCarNumber,
            });
            notFoundedCars.splice(movingCarIndex, 1);

            this.setState({ foundedCars, notFoundedCars });
        }

    }

    render() {
        const {
            tableData, tableType, month, foundedCars, notFoundedCars, isCarListLoading, carListLoadingError,
            errorsModalOpen, notFoundedModalOpen, compareResult, airportData, carsList,
        } = this.state;

        return <div>
            <div className={style.current_actions}>
                <span className={style.title}>Текущие действия:</span>
                {this.getCurrentAction()}
            </div>
            <div className={style.upload_buttons}>
                <div>
                    <ExcelUploader disabled={isCarListLoading || !!carListLoadingError}
                                   JSONHandler={this.JSONHandler.bind(this)}
                                   title={'Загрузить файл аэропорта'}/>
                </div>
                {tableData.list.length || tableData.errors.length
                    ? <>
                        <div className={`${style.block} ${style.select_controls}`}>
                            <Select placeholder={'Аэропорт'}
                                    containerClassName={style.table_type_select}
                                    options={AIRPORTS_OPTIONS}
                                    initialValues={tableType ? [tableType] : []}
                                    addingOddVariants
                                    required
                                    onSelect={this.onChangeTableType.bind(this)}/>
                            <Select placeholder={'Месяц'}
                                    containerClassName={style.table_month_select}
                                    options={MONTH_SHORT_OPTIONS}
                                    initialValues={month ? [month] : []}
                                    required
                                    onSelect={this.onChangeMonth.bind(this)}/>
                        </div>
                        <div className={style.block}>
                            <div>Длина проверенных: <span className={style.value}>{tableData.list.length}</span></div>
                            <div>Длина ошибок:
                                <span className={style.value}>
                                    {tableData.errors.length
                                        ? tableData.errors.length
                                        : <Link onClick={this.openErrorsModal.bind(this)}>
                                            {tableData.errors.length}
                                        </Link>}
                                </span>
                            </div>
                        </div>
                    </>
                    : null}
                {notFoundedCars.length
                    ? <div className={style.block}>Длина не найденных машин:
                        <span className={style.value}>
                            <Link onClick={this.openNotFoundedModal.bind(this)}>{notFoundedCars.length}</Link>
                        </span>
                    </div>
                    : null}
                {foundedCars.length
                    ? <div className={style.block}>Длина найденных машин:
                        <span className={style.value}>
                            {foundedCars.length}
                        </span>
                    </div>
                    : null}
                {foundedCars.length || notFoundedCars.length
                    ? <div className={style.download_link}>
                        <Link onClick={this.downLoadFileCheck.bind(this)}>Скачать отчёт по проверке файла в Excel</Link>
                    </div>
                    : null}
                {foundedCars.length
                    ? <div>
                        <Button disabled={!tableType || !month}
                                onClick={this.compareFileAndHahnData.bind(this)}>Сравнить</Button>
                    </div>
                    : null}
            </div>
            {errorsModalOpen
                ? <Window title={'Ошибки в обработке данных'} onClose={this.closeErrorsModal.bind(this)}>
                    <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Данные</th>
                            </tr>
                        </thead>
                        <tbody>
                            {tableData.errors.map((item, index) => {
                                return <tr key={index}>
                                    <td>{index + 1}</td>
                                    <td>{JSON.stringify(item)}</td>
                                </tr>;
                            })}
                        </tbody>
                    </table>
                </Window>
                : null}
            {compareResult
                ? <div className={style.block}>
                    <div>Совпадений: <Link onClick={this.openAirportsDataModal.bind(this, 'compared_common')}>
                        {compareResult.common.length}
                    </Link>
                    </div>
                    <div>Лишнее из файла: <Link onClick={this.openAirportsDataModal
                        .bind(this, 'compared_diff_table_1')}>
                        {compareResult.diff.table1.length}
                    </Link>
                    </div>
                    <div>Лишнее из БД: <Link onClick={this.openAirportsDataModal.bind(this, 'compared_diff_table_2')}>
                        {compareResult.diff.table2.length}
                    </Link>
                    </div>
                    <div className={style.download_link}>
                        <Link onClick={this.downLoadFileCompare.bind(this)}>Скачать отчёт по сравнению в Excel</Link>
                    </div>
                </div>
                : null}
            {notFoundedModalOpen
                ? <NotFoundedCarsModal closeNotFoundedModal={this.closeNotFoundedModal.bind(this)}
                                       moveToFounded={this.moveToFounded.bind(this)}
                                       notFoundedCars={notFoundedCars}/>
                : null}
            {airportData
                ? <AirportDataModal onClose={this.closeAirportsDataModal.bind(this)}
                                    carsList={carsList}
                                    carsArray={airportData}/>
                : null}
        </div>;
    }
}

interface INotFoundedCarsModalProps {
    closeNotFoundedModal: () => void;
    moveToFounded: () => void;
    notFoundedCars: Dict<any>[];
}

interface INotFoundedCarsModalState {
    filterValue: string;
    antiFilterValue: string;
    filteredNotFoundedCars: Dict<any>[];
}

class NotFoundedCarsModal extends React.Component<INotFoundedCarsModalProps, INotFoundedCarsModalState> {
    state: INotFoundedCarsModalState = {
        filterValue: '',
        antiFilterValue: '',
        filteredNotFoundedCars: [],
    };

    componentDidMount() {
        const filteredNotFoundedCars = this.props.notFoundedCars ?? [];
        this.setState({ filteredNotFoundedCars });
    }

    componentDidUpdate(prevProps: Readonly<INotFoundedCarsModalProps>): void {
        if (this.props.notFoundedCars !== prevProps.notFoundedCars) {
            const filteredNotFoundedCars = this.props.notFoundedCars ?? [];
            this.setState({ filteredNotFoundedCars });
        }
    }

    onFilterValueChange(filterValue: string) {
        this.setState({ filterValue }, () => {
            this.filterCars();
        });
    }

    onAntiFilterValueChange(antiFilterValue: string) {
        this.setState({ antiFilterValue }, () => {
            this.filterCars();
        });
    }

    filterCars() {
        const notFoundedCars = this.props.notFoundedCars ?? [];
        const { filterValue, antiFilterValue } = this.state;

        const filteredNotFoundedCars = notFoundedCars.filter(notFoundedCar => {

            const isFiltered = filterValue ? notFoundedCar?.car_number?.includes(filterValue) : true;
            const isAntiFiltered = antiFilterValue ? !notFoundedCar?.car_number?.includes(antiFilterValue) : true;

            return isFiltered && isAntiFiltered;
        });

        this.setState({ filteredNotFoundedCars });
    }

    render() {
        const { closeNotFoundedModal, moveToFounded } = this.props;
        const { filterValue, antiFilterValue, filteredNotFoundedCars } = this.state;

        return <Window title={'Не найденные автомобили'} onClose={closeNotFoundedModal.bind(this)}>
            <div className={style.not_founded_controls}>
                <Input className={style.not_founded_control}
                       value={filterValue}
                       placeholder={'Фильтр'}
                       onChange={this.onFilterValueChange.bind(this)}/>
                <Input className={style.not_founded_control}
                       value={antiFilterValue}
                       placeholder={'Антифильтр'}
                       onChange={this.onAntiFilterValueChange.bind(this)}/>
            </div>
            <div className={style.header}>
                <span className={style.cell}>#</span>
                <span className={style.cell}>Номер латиница</span>
                <span className={style.cell}>Номер кириллица</span>
                <span className={style.cell}>Возможные ошибки</span>
                <span className={style.cell}>Возможные ГРЗ</span>
            </div>
            <VirtualList width={'100%'}
                         height={750}
                         itemCount={filteredNotFoundedCars.length}
                         itemSize={60}
                         renderItem={({ index, style }) => {
                             return <NotFoundedCarItem _style={style}
                                                       index={index}
                                                       item={filteredNotFoundedCars[index]}
                                                       moveToFounded={moveToFounded}
                                                       key={index}/>;
                         }}/>
        </Window>;
    }
}

interface INotFoundedCarItemProps {
    index: number;
    item: any;
    _style: any;
    moveToFounded: () => void;
}

class NotFoundedCarItem extends React.Component<INotFoundedCarItemProps> {

    render() {
        const { index, item, _style, moveToFounded } = this.props;
        const errors = item?.isCarNumberValid?.errors ?? [];
        const supposedNumbers = item?.supposedNumbers ?? {};

        return <div style={_style} className={style.row}>
            <span className={style.cell}>{index}</span>
            <span className={style.cell}>
                {item?.car_number
                    ? <CarNumber key={item?.car_number}
                                 carInfo={{ number: item?.car_number } as typeof CarInfo}/>
                    : EMPTY_DATA}
            </span>
            <span className={style.cell}>
                {item?.car_number
                    ? <CarNumber key={makeKirCarNumber(item?.car_number)}
                                 carInfo={{ number: makeKirCarNumber(item?.car_number) } as typeof CarInfo}/>
                    : EMPTY_DATA}
            </span>
            <span className={style.cell}>
                {errors?.length
                    ? errors.join(', ')
                    : EMPTY_DATA}
            </span>
            <span className={style.cell}>
                {Object.values(supposedNumbers).length
                    ? Object.values(supposedNumbers).map((carInfoArray: Dict<any>[]) => {
                        return carInfoArray.map(carInfo => {
                            const carNumberKir = carInfo.number;
                            const id = carInfo.id;

                            return <div key={id}>
                                <CarNumber key={carNumberKir} carInfo={{ number: carNumberKir } as typeof CarInfo}/>
                                <div>
                                    <Link target={'_blank'} href={`#/cars/${id}/info`}>info</Link>
                                    /
                                    <Link onClick={moveToFounded.bind(this, item?.car_number, carInfo)}>Верный
                                        номер</Link>
                                </div>
                            </div>;
                        });
                    })
                    : null}
            </span>
        </div>;
    }
}

interface IAirportDataModalProps {
    onClose: () => void;
    carsArray: { car_number: string; date: string }[];
    carsList: Dict<Dict<any>>;
}

interface IAirportDataModalState {
    filterValue: string;
    antiFilterValue: string;
    filteredCars: Dict<any>[];
}

class AirportDataModal extends React.Component<IAirportDataModalProps, IAirportDataModalState> {
    state: IAirportDataModalState = {
        filterValue: '',
        antiFilterValue: '',
        filteredCars: [],
    };

    componentDidMount() {
        this.filterCars();
    }

    componentDidUpdate(prevProps: Readonly<IAirportDataModalProps>): void {
        if (this.props.carsArray !== prevProps.carsArray) {
            this.filterCars();
        }
    }

    onFilterValueChange(filterValue: string) {
        this.setState({ filterValue }, () => {
            this.filterCars();
        });
    }

    onAntiFilterValueChange(antiFilterValue: string) {
        this.setState({ antiFilterValue }, () => {
            this.filterCars();
        });
    }

    filterCars() {
        const { carsList } = this.props;
        const carsArray = this.props.carsArray ?? [];
        const { filterValue, antiFilterValue } = this.state;

        const filteredCars: { car_number: string; date: string; carInfo?: Dict<any> }[] = filterValue || antiFilterValue
            ? carsArray.filter(notFoundedCar => {
                const carNumberKir = makeKirCarNumber(notFoundedCar?.car_number);

                const isFiltered = filterValue ? notFoundedCar?.car_number?.includes(filterValue)
                    || carNumberKir?.includes(filterValue) : true;
                const isAntiFiltered = antiFilterValue ? !notFoundedCar?.car_number?.includes(antiFilterValue)
                    || !carNumberKir?.includes(antiFilterValue) : true;

                return isFiltered && isAntiFiltered;
            })
            : carsArray;

        filteredCars.map(filteredCar => {
            const { car_number } = filteredCar;
            filteredCar.carInfo = carsList?.[car_number]?.[0] ?? {};
        });

        this.setState({ filteredCars });
    }

    render() {
        const { onClose } = this.props;
        const { filterValue, antiFilterValue, filteredCars } = this.state;

        return <Window title={'Информация по авто'} onClose={onClose.bind(this)}>
            <div className={style.not_founded_controls}>
                <Input className={style.not_founded_control}
                       value={filterValue}
                       placeholder={'Фильтр по ГРЗ'}
                       onChange={this.onFilterValueChange.bind(this)}/>
                <Input className={style.not_founded_control}
                       value={antiFilterValue}
                       placeholder={'Антифильтр по ГРЗ'}
                       onChange={this.onAntiFilterValueChange.bind(this)}/>
            </div>
            <div className={style.header}>
                <span className={style.cell}>#</span>
                <span className={style.cell}>Дата</span>
                <span className={style.cell}>Номер латиница</span>
                <span className={style.cell}>Номер кириллица</span>
                <span className={style.cell}>Авто</span>
            </div>
            <VirtualList width={'100%'}
                         height={750}
                         itemCount={filteredCars.length}
                         itemSize={60}
                         renderItem={({ index, style }) => {
                             return <AirportDataItem _style={style}
                                                     index={index}
                                                     item={filteredCars[index]}
                                                     key={index}/>;
                         }}/>
        </Window>;
    }
}

interface IAirportDataItemProps {
    index: number;
    item: any;
    _style: any;
}

class AirportDataItem extends React.Component<IAirportDataItemProps> {

    render() {
        const { index, item, _style } = this.props;
        const carInfo = item?.carInfo || {};
        const { id, model_id } = carInfo;

        const sessionDate: moment.Moment | null = item?.date
            ? moment(item?.date, 'DD.MM.YYYY').startOf('day')
            : null;
        const sessionDateBefore: moment.Moment | null = sessionDate
            ? moment(sessionDate).subtract(1, 'days').startOf('day')
            : null;
        const sessionDateAfter: moment.Moment | null = sessionDate
            ? moment(sessionDate).add(1, 'days').startOf('day')
            : null;

        const tracksLink = `#/tracks?car_id=${id}`
            + `${sessionDateBefore ? `&since=${+sessionDateBefore}` : ''}`
            + `${sessionDateAfter ? `&until=${+sessionDateAfter}` : ''}`;

        return <div style={_style} className={style.row}>
            <span className={style.cell}>{index}</span>
            <span className={style.cell}>
                <FormatDate value={moment(item.date, 'DD.MM.YYYY')} onlyDate/>
            </span>
            <span className={style.cell}>
                {item?.car_number
                    ? <CarNumber key={item?.car_number}
                                 carInfo={{ number: item?.car_number } as typeof CarInfo}/>
                    : EMPTY_DATA}
            </span>
            <span className={style.cell}>
                {item?.car_number
                    ? <CarNumber key={makeKirCarNumber(item?.car_number)}
                                 carInfo={{ number: makeKirCarNumber(item?.car_number) } as typeof CarInfo}/>
                    : EMPTY_DATA}
            </span>
            <span className={style.cell}>
                {item?.carInfo?.id
                    ? <div>
                        <Link target={'_blank'} href={`#/cars/${id}/info`}>
                            {model_id ?? 'Авто'}
                        </Link>
                        <div>
                            <Link target={'_blank'} href={tracksLink}>
                                Треки
                            </Link>
                        </div>
                    </div>
                    : null}
            </span>
        </div>;
    }
}
